/*
 * Copyright (c) 2011 NetApp
 * All rights reserved
 */

package com.netapp.collectors.vmware;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


import com.netapp.collectors.vmware.exceptions.*;
import com.netapp.collectors.vmware.logger.LogsManager;
import com.netapp.collectors.vmware.util.ContextUtil;
import com.netapp.collectors.vmware.builders.PropertySpecBuilder;
import com.netapp.collectors.vmware.util.SpecBuilderUtil;
import com.netapp.collectors.vmware.util.TimeUtil;
import com.netapp.collectors.vmware.util.VMWareDataFetcher;
import com.vmware.vim25.*;


import java.util.concurrent.ConcurrentHashMap;
import javax.xml.datatype.XMLGregorianCalendar;
import org.slf4j.Logger;

/**
 * Provides access to VMWare services with the vim25 api.
 * 
 * @author slaplante, dhetti
 */
public class VMWare25Service implements IVMWareService {
        
    /* Logger. */
    private static final Logger logger = LogsManager.getLogger(VMWare25Service.class);

    /** Uuid does not exist in uuid to managed object reference map based on the key {0}. */
    private static final String ERR_NO_UUID_BY_MANAGED_OBJECT_REFERENCE = "Uuid does not exist in uuid to managed object reference map based on the key {0} from access handle id: {1}.";

    /** Host does not support datastore statistics counters. */
    protected static final String MSG_HOST_DOESNT_SUPPORT_DATASTORE_STATS = "Host {0} does not support datastore stats counters, skipping query for datastore stats";

    /** Host does not support virtual disk statistics counters. */
    protected static final String MSG_HOST_DOESNT_SUPPORT_VIRTUALDISK_STATS = "Host {0} does not support virtualDisk stats counters, skipping query for virtualDisk stats";

    private static final String ERR_NULL_UUID = "Uuid is null for AH = {0} MOR = {1}";

    public static final String MSG_NETWORK_ERROR =
            "A network error occurred. This could be caused by a network problem or service no longer responding.";

    /** Message that is checked to determine if an exception occurred as a result of querying an object that has been deleted or not fully created. */
    public static final String OBJECT_NOT_CREATED_ERROR = "object has already been deleted or has not been completely created";

    public static final String MSG_OBJECT_DELETED_OR_NOT_FULLY_CREATED = "Error querying object because it has already been deleted or has not been completely created.";

    public static final String VIM_25_PACKAGE = "com.vmware.vim25";

        
    /** Instance of the VMWare 25 Service. */
    private static volatile VMWare25Service vmware25Service;

    public String getVirtualCenterUrl() {
        return virtualCenterUrl;
    }

    public void setVirtualCenterUrl(String virtualCenterUrl) {
        this.virtualCenterUrl = virtualCenterUrl;
    }

    private String virtualCenterUrl;

        
    /** ManagedObjectReference to UUID map. */
    private Map<Long, Map<Object, String>> virtualCenterUuidToMoRefMap = Collections.synchronizedMap(new HashMap<Long, Map<Object, String>>());



    /**
     * Default Constructor. Protected to avoid direct instantiation by non-subclasses.
     */
    protected VMWare25Service(){
    }
        
    /**
     * This creates an instance of the VMWare 25 Service.
     * 
     * @return vmware25Service vmware service instance.
     */
    public static VMWare25Service getInstance(){
        if (vmware25Service == null){
            synchronized (VMWare25Service.class) {
                if(vmware25Service == null) {
                    vmware25Service = new VMWare25Service();
                }
            }
        }
        return vmware25Service;
    }

        
    /**
     * {@inheritDoc}
     */
    public com.netapp.collectors.vmware.IVMWareContext connectAndGetContext(String userName, String password, String serviceURL, String hostNameOrIp,
                                                                            boolean ignoreCertificate, String deviceType, long accessHandleId, int sslConnectionTimeout) throws VMWareConnectionException, VMWareCertificateException, VMWareServiceException {
        

        boolean isEmbeddedEsx = false;
        boolean isEsx = false;
        com.netapp.collectors.vmware.ServiceConnection25 serviceConnection = null;
        ServiceContent serviceContent = null;
        Integer vcInstanceId = null;

        try {

            serviceConnection = new ServiceConnection25();
            serviceConnection.connect(serviceURL, userName, password, sslConnectionTimeout, ignoreCertificate);
            serviceContent = serviceConnection.getServiceContent();

            List<OptionValue> optionValues = VMWareDataFetcher.retrieveServiceContentSettings(new VMWare25Context(serviceConnection, serviceContent, accessHandleId));
            //TODO parameterize this for UT


            if(optionValues == null){
                // settings do not exist. If this code is executed then the user has the correct privileges to log in, 
                // but the privileges may be set at the wrong level.
                
                // Check where permissions have been set. This only handles the case when the correct permissions are set 
                // at the wrong level. It is currently required for the user to be assigned to a role with read permissions 
                // at the root level. Other situations such as a user existing, but not being assigned to a role or a user
                // assigned to the No Access role will be handled by the AxisFaultTranslator.
                
                // A limitation here is that the user cannot be checked to see what group it belongs too. If the permissions
                // retrieved contain groups and not users then there is no way of knowing the user has the correct permission
                // levels set. If the user given is assigned to a role with permissions defined in the root level, optionValues array 
                // would not be null and this code will not be executed. 
                ManagedObjectReference authorizationManager = serviceContent.getAuthorizationManager();
                List<Permission> permissions = serviceConnection.getService().retrieveAllPermissions(authorizationManager);
                  
                // get list of permissions containing the user name which will be used to connect and collect
                List<Permission> permissionsWithUserName = getPermissionsContainingUserName(userName, permissions);
                    
                // check if the privileges of this user have been applied at the root level
                List<Permission> permissionsWithRootLevelEntity = getPermissionsContainedInFolder(serviceContent.getRootFolder(), 
                                                                                                  permissionsWithUserName);
                
                if(permissionsWithRootLevelEntity.size() == 0){
                    throw new VMWareConnectionException(CommunicationStateReason.PERMISSION_DENIED, "User with the name " + userName
                                                  + " may not have been assigned privileges at the root level.");
                }
            }
            
            if(deviceType.equals(VMWareConsts.VIRTUAL_CENTER)){
                
                for(OptionValue optionValue : optionValues){
                    if (optionValue.getKey().equals("instance.id")) {
                        vcInstanceId = (Integer)optionValue.getValue();
                    }
                }
                
                // Verify that we have a valid Virtual Center Unique Id for this device
                if(vcInstanceId == null){
                    throw new VMWareServiceException("Failed to retrieve Virtual Center instance Id for device");
                } else {
                    logger.debug("Retrieved the VC instance id {}", vcInstanceId);
                }
            }

        } catch (Exception ex) { //If VCenter connection fails for any other reason unknown, log and rethrow the exception,
            logger.error(String.format("Error while connecting to VCenter %s",hostNameOrIp));
            throw new VMWareConnectionException(ex);
        }/* TODO : Revisit these exception handling catch (InvalidLogin e) {
            String msg = MessageFormat.format(MSG_INVALID_LOGIN, serviceURL, userName);
            throw new DeviceCommException(CommunicationStateReason.INVALID_PRIMARY_CREDENTIAL, e, msg);
        } catch (AxisFault e) {
            DeviceCommException translatedFault = AxisFaultTranslator.translateException(e);
            throw translatedFault;
        }catch (DeviceCommException e){
            // throw any DeviceCommException so the collection server can handle accordingly
            throw e;
        } catch (Exception e) {
            String msg = MessageFormat.format(MSG_ERROR_BAD_REQUEST, serviceURL, userName);
            throw new RuntimeException(msg, e);
        } finally {

            if (deviceType.equals("VIRTUAL_CENTER")) {
                // verify we are talking to a virtual center server
                if (serviceContent != null && serviceContent.getAbout().getApiType().compareTo("VirtualCenter") != 0) {
                    String msg = MessageFormat.format(MSG_UNSUPPORTED_VMWARE_API_TYPE, serviceContent.getAbout()
                                                      .getApiType());
                    throw new DeviceCommException(CommunicationStateReason.INVALID_DEVICE_TYPE, msg);
                }
                if (serviceContent != null && serviceContent.getAbout().getProductLineId().compareTo("vpx") != 0) {
                    String msg = MessageFormat.format(MSG_UNSUPPORTED_VMWARE_PRODUCT_LINE, serviceContent.getAbout()
                                                      .getProductLineId());
                    throw new DeviceCommException(CommunicationStateReason.INVALID_DEVICE_TYPE, msg);
                }
            } else {
                // Verify that we are talking to an ESX
                if (serviceContent != null && serviceContent.getAbout().getApiType().compareTo("HostAgent") != 0) {
                    // this is not an ESX host (perhaps a VirtualCenter?). Quit.
                    String msg = MessageFormat.format("API type is {0}, not HostAgent", serviceContent.getAbout()
                                                      .getApiType());
                    throw new DeviceCommException(CommunicationStateReason.INVALID_DEVICE_TYPE, msg);
                }

                if (serviceContent != null && serviceContent.getAbout().getProductLineId().compareTo("esx") != 0
                    && serviceContent.getAbout().getProductLineId().compareTo("embeddedEsx") != 0) {
                    // this is not an ESX host (perhaps a GSX, or other VMWare Server?).
                    // Quit.
                    String msg = MessageFormat.format("Product line is {0}, not esx or embeddedEsx", serviceContent
                                                      .getAbout().getProductLineId());
                    throw new DeviceCommException(CommunicationStateReason.INVALID_DEVICE_TYPE, msg);
                }

                // Verify that we are talking to a supported VMWare platform
                if (serviceContent != null && serviceContent.getAbout() != null) {
                    if (serviceContent.getAbout().getApiType().compareTo("HostAgent") != 0) {
                        // this is not an ESX host (perhaps a VirtualCenter?). Quit.
                        String msg = MessageFormat.format("API type is {0}, not HostAgent", serviceContent.getAbout()
                                                          .getApiType());
                        throw new DeviceCommException(CommunicationStateReason.INVALID_DEVICE_TYPE, msg);
                    }

                    if (serviceContent.getAbout().getProductLineId().compareTo("esx") == 0) {
                        isEmbeddedEsx = false;
                        isEsx = true;
                    } else if (serviceContent.getAbout().getProductLineId().compareTo("embeddedEsx") == 0) {
                        isEmbeddedEsx = true;
                        isEsx = false;
                    } else {
                        // this is not an ESX or 3i host (perhaps a GSX, or other VMWare Server?).
                        // Quit.
                        String msg = MessageFormat.format("Product line is {0}, not esx or embeddedEsx", serviceContent
                                                          .getAbout().getProductLineId());
                        throw new DeviceCommException(CommunicationStateReason.INVALID_DEVICE_TYPE, msg);
                    }
                }
            }
        } // end finally*/
        XMLGregorianCalendar now = getCurrentTime(serviceConnection);
        logger.debug("Current timestamp for context: " + TimeUtil.fromXMLGregorianCalendar(now).getTimeInMillis());
        IVMWareContext vmwareContext;
        
        if(deviceType.equals(VMWareConsts.VIRTUAL_CENTER)){
            vmwareContext = new VMWare25Context(serviceConnection, serviceContent, VIM_25_PACKAGE,
                                          isEmbeddedEsx,isEsx, vcInstanceId, accessHandleId, now, serviceURL, hostNameOrIp);
        }else{
            vmwareContext = new VMWare25Context(serviceConnection, serviceContent, VIM_25_PACKAGE, isEmbeddedEsx,
                                          isEsx, accessHandleId, now);
        }
        return vmwareContext;
    }

    //TODO : Uncomment this disconnect logic once we have caching mechanism for vcenter connection context in place.


    /*@Override
    public void disconnect(String virtualCenterUrl) {
        IVMWareContext context = vcToContextMap.remove(virtualCenterUrl);
        if(context != null) {
            context.destroy();
        }
    } */

    /**
     * This method makes use of the newly available method retrievePropertiesEx to fetch the data from VMware SDK
     * @param vmWareContext
     * @return RetrieveResult
     */
    public static RetrieveResult retrieveAllVmDetails(IVMWareContext vmWareContext) {

        ManagedObjectReference propertyCollector = (ManagedObjectReference) vmWareContext.getPropertyCollector();
        ManagedObjectReference containerViewReference = createContainerView(vmWareContext, VMWareConsts.VIRTUAL_MACHINE);
        PropertyFilterSpec filterSpec = new PropertyFilterSpec();
        filterSpec.getObjectSet().addAll(SpecBuilderUtil.buildObjectSetForVm(containerViewReference));
        filterSpec.getPropSet().addAll(SpecBuilderUtil.buildPropertySetForVm());
        List<PropertyFilterSpec> propertyFilterSpecList = new ArrayList<>();
        propertyFilterSpecList.add(filterSpec);
        RetrieveOptions ro = new RetrieveOptions();
        try {
            return ContextUtil.getVimPortService(vmWareContext).retrievePropertiesEx(propertyCollector,propertyFilterSpecList,ro);
        } catch (InvalidPropertyFaultMsg invalidPropertyFaultMsg) {
            throw new RuntimeException("Error while fetching data - invalid property " + invalidPropertyFaultMsg.getMessage());
        } catch (RuntimeFaultFaultMsg runtimeFaultFaultMsg) {
            throw new RuntimeException("Error while fetching data - server error" + runtimeFaultFaultMsg.getMessage());
        }
    }

    /**
     * Method to retrieve properties from VMWare Vcenter, it just takes teh root folder and propertyspec and traversal spec
     * @param mor
     * @param pSpec
     * @param tSpec
     * @param skipRoot
     * @param context
     * @param maxRecords
     * @param nextToken
     * @return
     */
    public static RetrieveResult retrieveManagedObjectsAndProperties(ManagedObjectReference mor, List<PropertySpec> pSpec, List<SelectionSpec> tSpec, boolean skipRoot, IVMWareContext context, Integer maxRecords, String nextToken) {
        List<ObjectSpec> obSpecList = new ArrayList<>();
        ObjectSpec objectSpec = new ObjectSpec();
        objectSpec.setObj(mor);
        objectSpec.setSkip(skipRoot);
        objectSpec.getSelectSet().addAll(tSpec);
        obSpecList.add(objectSpec);
        return retrieveManagedObjectsAndProperties(obSpecList, pSpec, skipRoot, context, maxRecords, nextToken);

    }
    private static RetrieveResult retrieveManagedObjectsAndProperties(List<ObjectSpec> obSpec,
                                                                      List<PropertySpec> pSpec, boolean skipRoot, IVMWareContext context, Integer maxRecords, String nextToken) {
        PropertyFilterSpec fSpec = new PropertyFilterSpec();
        fSpec.getObjectSet().addAll(obSpec);
        fSpec.getPropSet().addAll(pSpec);
        List<PropertyFilterSpec> fSpecList = new ArrayList<PropertyFilterSpec>();
        fSpecList.add(fSpec);
        RetrieveOptions retrieveOptions = new RetrieveOptions();
        if (maxRecords != null) {
            retrieveOptions.setMaxObjects(maxRecords);
        }
        try {
            if (nextToken == null) {
                return ContextUtil.getVimPortService(context).retrievePropertiesEx(
                        (ManagedObjectReference) context.getPropertyCollector(), fSpecList, retrieveOptions);
            } else {
                return ContextUtil.getVimPortService(context).continueRetrievePropertiesEx((ManagedObjectReference) context.getPropertyCollector(), nextToken);
            }

        /* TODO : Revisit these exception handling} catch (RemoteException e) {
            if ((e.getMessage() != null) && (e.getMessage().contains("(0)null"))) {
                // This is a known issue with axis, 'caused by: (0)null' occurs when a network error occurs in
                // the axis code. The error message description is 'null' because the exception didn't assign a message
                // to it.
                throw new DeviceCommException(CommunicationStateReason.CONNECTION_REFUSED, e, MSG_NETWORK_ERROR);
            } else if (e.getCause() instanceof SocketException) {
                throw new DeviceCommException(CommunicationStateReason.CONNECTION_REFUSED, e, e.getCause().getMessage());
            } else if (e.getCause() instanceof SocketTimeoutException) {
                throw new DeviceCommException(CommunicationStateReason.TIMEOUT, e, e.getCause().getMessage());
            } else {
                String exceptionMessage = e.getCause().getMessage();
                if(exceptionMessage != null && exceptionMessage.contains(OBJECT_NOT_CREATED_ERROR)){
                    // there was a problem querying the vmware device for a given object because it either no longer exists or is not done being created.
                    throw new VMWareServiceException(MSG_OBJECT_DELETED_OR_NOT_FULLY_CREATED, e);
                }else{
                    String msg = MessageFormat.format(MSG_ERROR_RETRIEVING_MANAGED_OBJECTS_PROPS,e.getMessage());
                    throw new RuntimeException(msg,e);
                }
            }*/
        } catch (InvalidPropertyFaultMsg invalidPropertyFaultMsg) {
            throw new InvalidPropertyException(invalidPropertyFaultMsg.getMessage());
        } catch (RuntimeFaultFaultMsg runtimeFaultFaultMsg) {
            throw new RuntimeException(runtimeFaultFaultMsg.getMessage());
        }
    }

    public static List<OptionValue> retreiveServiceContentSettings(VMWare25Context vmWare25Context) {

        ManagedObjectReference optionManager = ((ServiceContent) vmWare25Context.getServiceContent()).getSetting();
        if (optionManager != null) {
            List<PropertySpec> pSpec = new ArrayList<>();
            PropertySpec serviceContentPropSet = new PropertySpecBuilder()
                    .all(Boolean.FALSE)
                    .addToPathSet(Arrays.asList("setting"))
                    .type(optionManager.getType());
            pSpec.add(serviceContentPropSet);
            //This is a sample line
            List<ObjectSpec> oSpec = new ArrayList<>();
            ObjectSpec serviceContentObjectSpec = new ObjectSpec();
            serviceContentObjectSpec.setObj(optionManager);
            oSpec.add(serviceContentObjectSpec);
            RetrieveResult retrieveResult = retrieveManagedObjectsAndProperties(oSpec, pSpec, false, vmWare25Context, null, null);

            for (ObjectContent objectContent : retrieveResult.getObjects()) {
                List<DynamicProperty> propSet = objectContent.getPropSet();
                if (propSet != null) {
                    for (DynamicProperty dynamicProperty : propSet) {
                        if(dynamicProperty.getName().equals("setting")){
                            ArrayOfOptionValue vals = (ArrayOfOptionValue)dynamicProperty.getVal();
                            return vals.getOptionValue();
                        }
                    }
                }
            }
        }

        return null;
    }
    private static ManagedObjectReference createContainerView(final IVMWareContext vmWareContext, String objectType) {
        ManagedObjectReference viewManagerReference = ContextUtil.getViewManagerReference(vmWareContext);
        List<String> typeList = new ArrayList<String>();
        typeList.add(objectType);
        try {
            return ContextUtil.getVimPortService(vmWareContext).createContainerView(viewManagerReference,
                    ContextUtil.getRootFolder(vmWareContext), typeList, true);
        } catch (RuntimeFaultFaultMsg runtimeFaultFaultMsg) {
            throw new RuntimeException("Error while creating the container view" + runtimeFaultFaultMsg.getMessage());
        }
    }

    private List<Permission> getPermissionsContainedInFolder(ManagedObjectReference folder, List<Permission> permissions){
        // check if any of the permissions have been applied at the virtual center root level
        ArrayList<Permission> permissionsWithGivenLevelEntity = new ArrayList<Permission>();
        for(Permission permission : permissions){
            // check if this permission applies to the given root folder
            if(permission.getEntity().equals(folder)){
                // permission has been applied at the given level, add it to the permissions list
                permissionsWithGivenLevelEntity.add(permission);
            }
        }
        
        return permissionsWithGivenLevelEntity;
    }
        

    private XMLGregorianCalendar getCurrentTime(ServiceConnection25 svc) {
        XMLGregorianCalendar now = null;
        ManagedObjectReference svcRef = svc.getServiceInstanceRef();
        try {
            now = svc.getService().currentTime(svcRef);
        } catch (RuntimeFaultFaultMsg runtimeFaultFaultMsg) {
            runtimeFaultFaultMsg.printStackTrace();
        }
        return now;
    }

    /**
     * Retrieve a list of permissions which contain the given user name.
     *
     * @param userName name of user which is connecting to this vmware device.
     * @param permissions array of permissions which to search from
     * @return permissionsWithUserName permissions which contain the given user name
     */
    private List<Permission> getPermissionsContainingUserName(String userName, List<Permission> permissions) {
        ArrayList<Permission> permissionsWithUserName = new ArrayList<Permission>();
        for (Permission permission : permissions) {
            String principal = permission.getPrincipal();

            if (userName.equals(principal)) {
                // this permission contains the given user name
                permissionsWithUserName.add(permission);
            }
        }

        return permissionsWithUserName;
    }
}
