/*
 * Copyright (c) 2008 Akorri Networks, Inc. All Rights Reserved. This software is provided
 * under license and is not sold by Akorri Networks. This software may be used only in accordance
 * with the terms of an express written license agreement with Akorri Networks, and cannot be used,
 * copied, modified, or in any way distributed without the express permission of Akorri Networks.
 *
 */

package com.netapp.collectors.vmware;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Collection domain object for ESX Hosts.
 * 
 * @author slaplante, dhetti
 */
public class VMWareHost{

    /**
     * Overwritten toString
     */
    public String toString() {
        String s = "\nVMWareHost:\n";
        s += "    Name                : " + name + "\n";
        s += "    MO                  : " + managedObject + "\n";
        s += "    Num of console IPs  : " + consoleVNicIpAddress.size() + "\n";
        s += "    Num of IPs          : " + vnicIPAddress.size() + "\n";
        s += "    Product Name        : " + productName + "\n";
        s += "    UUID                : " + uuid + "\n";
        s += "    Power State         : " + powerState + "\n";
        s += "    Connection State    : " + connectionState + "\n";
        s += "    Maintenance Mode    : " + maintenanceMode + "\n";
        s += "    Num of VMs          : " + virtualMachines.size() + "\n";
        s += "    Num of DataStores   : " + dataStores.size() + "\n";
        s += "    Num of Physical NICs: " + physicalNics.size() + "\n";
        s += "    Root Resource Pool  : " + rootResourcePool + "\n"; 
        if ( availablePerfCounters != null ) {
            s += "    Size of Avail Cntrs : " + availablePerfCounters.size() + "\n";
        } else {
            s += "    Size of Avail Cntrs : null\n";
        }
        if ( lunStats != null ) {
            s += "    Size of LUN Stats   : " + lunStats.size() + "\n";
        } else {
            s += "    Size of LUN Stats   : null";
        }
        s += "    Num of PNIC Stats    : " + physicalNicStats.size() + "\n";
        s += "    Sample Period        : " + samplePeriod + "\n";
        s += "    Host Name            : " + hostName + "\n";
        s += "    Num SCSCI LUN Arrays : " + scsiLUNArray.length + "\n";
        s += "    Num Host File Mnts   : " + hostFileSystemMountInfo.length + "\n";
        s += "    Num Host Bus Adapters: " + hostBusAdapter.length + "\n";
        s += "    Numer of Samples     : " + numSamples + "\n";
        s += "    Memory Size          : " + memorySize + "\n";
        s += "    System Up Time       : " + systemUpTime + "\n";
        s += "    Memory Usage Avg     : " + memoryUsageAvg + "\n";
        s += "    Memory Swap Out Avg  : " + memorySwapOutAvg + "\n";
        s += "    Memory Swap In Avg   : " + memorySwapInAvg + "\n";
        s += "    CPU Usage Avg        : " + cpuUsageAvg + "\n";
        s += "    CPU Used Mhz Avg     : " + cpuUsageMhzAvg + "\n";
        s += "    Memory Consumed Avg  : " + memoryConsumedAvg + "\n";
        s += "    Memory Overhead Avg  : " + memoryOverheadAvg + "\n";
        s += "    Memory Balloon       : " + memoryBalloon + "\n";
        s += "    Memory Ballon Target : " + memoryBalloonTarget + "\n";
        s += "    Avg Network Usage    : " + avgNetworkUsage + "\n";
        s += "    Num Valid CPU Samples: " + numValidCpuUsageSamples + "\n";
        s += "    CPU Hz               : " + cpuHz + "\n";
        s += "    Number of CPU Cores  : " + numberOfCpuCores + "\n";
        s += "    Number of CPU Threads: " + numberOfCpuThreads + "\n";
        s += "    Product Full Name    : " + productFullName + "\n";
        s += "    ESX Version          : " + esxVersion + "\n";
        s += "    ESX Build            : " + esxBuild + "\n";
        s += "    Effective CPU        : " + effectiveCpu + "\n";
        s += "    Effective Memory     : " + effectiveMemory + "\n";
        s += "    Storage System Mor   : " + storageSystemMor + "\n";

        return s;
    }


    /** Name of this esx host. */
    private String name;
    
    /** Managed object of this host. */
    private Object managedObject;
    
    /** Console vnic ip addresses of this esx host. */
    private ArrayList<String> consoleVNicIpAddress = new ArrayList<String>();
    
    /** Vnic ip addresses of this esx host. */
    private ArrayList<String> vnicIPAddress = new ArrayList<String>();
    
    /** Short form of the product name of this host. */
    private String productName;
    
    /** Virtual uuid of this host. */
    private String uuid;
    
    /** Host Power State. */
    private String powerState;

    /** Host connection state. */
    private String connectionState;

    /** Maintenance Mode. */
    private Boolean maintenanceMode;
    
    /** List of virtual machines related associated to this host. */
    private LinkedHashMap<Object, VMWareVirtualMachine> virtualMachines = new LinkedHashMap<Object, VMWareVirtualMachine>();
    
    /** List of data stores related associated to this host. */
    private LinkedHashMap<Object, VMWareDatastore> dataStores = new LinkedHashMap<Object, VMWareDatastore>();

    /** List of physical network adapters associated to this host. */
    private LinkedHashMap<String, VMWarePhysicalNic> physicalNics = new LinkedHashMap<String, VMWarePhysicalNic>();

    /** Root resource pool related associated to this host. */
    private VMWareResourcePool rootResourcePool; 
    
    /** All Available performance counters for stats.*/
    private Map<String, Integer> availablePerfCounters;

    /**All LUN Statistics.*/
    private HashMap<String, VMWareLUN> lunStats;

    /** All Physical NIC statistics. */
    private HashMap<String, VMWarePhysicalNic> physicalNicStats = new HashMap<String, VMWarePhysicalNic>();

    /** Host Sampling period.*/
    private int samplePeriod;
    
    /** Host name of this host. */
    private String hostName;
    
    /** This is the SCSI LUns for the host. This is a VMWare API object need to cast it to ScsiLun object in VMWare.*/
    private Object[] scsiLUNArray = new Object[0];
    
    private Object[] hostFileSystemMountInfo = new Object[0];
    private Object[] hostBusAdapter = new Object[0];

    /** Number of statistic samples. */
    private int numSamples;
    /** Memory Size.*/
    private long memorySize;
    /** System Up Time.*/
    private long systemUpTime;
    /** Memory Usage Avg.*/
    private float memoryUsageAvg;
    /** Memory Swap out avg.*/
    private long memorySwapOutAvg;
    /** Memory swap in avg.*/
    private long memorySwapInAvg;
    /** CPU usage avg.*/
    private float cpuUsageAvg;
    /** CPU usage MHz avg.*/
    private float cpuUsageMhzAvg;
    /** Memory Consumed Avg.*/
    private float memoryConsumedAvg;
    /** Memory Overhead Avg.*/
    private float memoryOverheadAvg;
    /** Memory.*/
    private float memoryBalloon;
    /** Memory.*/
    private float memoryBalloonTarget;
    /** Network usage in KBps. */
    private float avgNetworkUsage;

    /** Number of valid CPU Usage Samples.*/
    private int numValidCpuUsageSamples;
    
    /** CPU Hz.*/
    private long cpuHz;
    
    /** Number of CPU Cores.*/
    private int numberOfCpuCores;
    /** Number of CPU Threads.*/
    private int numberOfCpuThreads;
    /** The complete product name, including the version information of the host. */
    private String productFullName;

    /** The version of the host. i.e. '3.0.0', '3.5.0' or '4.0.0' */
    private String esxVersion;

    /** The build string of the host. */
    private String esxBuild;

    /* systemResources.config.cpuAllocation.limit */
    private Long effectiveCpu;

    /* systemResources.config.memoryAllocation.limit */
    private Long effectiveMemory;

    /** Managed Object Reference of the Storage System. */
    private Object storageSystemMor;

	/**
     * Constructor.
     * 
     * @param managedObject managed object of this host
     */
    public VMWareHost(Object managedObject){
        this.managedObject = managedObject;
    }

    /**
     * Gets the name of this host.
     * 
     * @return name name of this host
     */
    public String getName() {
        return name;
    }

    /**
     * Sets the name of this host.
     * 
     * @param name name of this host
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Gets the managed object of this host.
     * 
     * @return managedObject managed object id of this host
     */
    public Object getManagedObject(){
        return this.managedObject;
    }
    
    /**
     * Sets the managed object of this host.
     * 
     * @param managedObject managed object of this host
     */
    public void setManagedObjectId(Object managedObject){
        this.managedObject = managedObject;
    }
    
    /**
     * Gets the console vnic ip address of this esx host.
     * 
     * @return consoleVNicIpAddress console vnic ip address of this esx host
     */
    public ArrayList<String> getConsoleVNicIpAddress() {
        return consoleVNicIpAddress;
    }
    
    /**
     * Sets the console vnic ip address of this esx host.
     * 
     * @param consoleVNicIpAddress console vnic ip address of this esx host
     */
    public void setConsoleVNicIpAddress(ArrayList<String> consoleVNicIpAddress) {
        this.consoleVNicIpAddress = consoleVNicIpAddress;
    }
    
    /**
     * Gets the vnic ip address of this esx host.
     * 
     * @return vNicIPAddress vnic ip address of this esx host
     */
    public ArrayList<String> getVnicIPAddress() {
        return vnicIPAddress;
    }

    /**
     * Sets the vnic ip address of this esx host.
     * 
     * @param nicIPAddress vnic ip address of this esx host
     */
    public void setVnicIPAddress(ArrayList<String> nicIPAddress) {
        vnicIPAddress = nicIPAddress;
    }
    
    /**
     * Gets the product name of this esx host.
     * 
     * @return productName product name of this esx host
     */
    public String getProductName() {
        return this.productName;
    }

    /**
     * Sets the product name of this esx host.
     * 
     * @param productName product name of this esx host
     */
    public void setProductName(String productName) {
        this.productName = productName;
    }
    
    /**
     * Gets the virtual uuid of this esx host.
     * 
     * @return uuid virtual uuid of this esx host
     */
    public String getUuid() {
        return uuid;
    }

    /**
     * Sets the virtual uuid of this esx host.
     * 
     * @param uuid virtual uuid of this esx host
     */
    public void setUuid(String uuid) {
        this.uuid = uuid;
    }
    
    /**
     * Gets a list of all virtual machines associated to this host.
     * 
     * @return virtualMachines virtual machines associated to this host
     */
    public LinkedHashMap<Object, VMWareVirtualMachine> getAllVirtualMachines() {
        return this.virtualMachines;
    }
    
    /**
     * This will sets all the virtual machines.
     * 
     * @param allVirtualMachines all virtual machines.
     */
    public void setAllVirtualMachines(LinkedHashMap<Object, VMWareVirtualMachine> allVirtualMachines){
    	this.virtualMachines = allVirtualMachines;
    }

    /**
     * Gets a virtual machine associated to this host.
     * 
     * @param key managed object of this virtual machine
     * @return virtualMachine virtual machine object associated to this host
     */
    public VMWareVirtualMachine getVirtualMachine(Object key) {
        return this.virtualMachines.get(key);
    }
    
    /**
     * Sets a virtual machine contained associated to this host.
     * 
     * @param key managed object of the virtual machine
     * @param virtualMachine virtual machine object associated to this host
     */
    public void putVirtualMachine(Object key, VMWareVirtualMachine virtualMachine) {
        this.virtualMachines.put(key, virtualMachine);
    }

    /**
     * Gets a data store associated to this host.
     * 
     * @param key managed object of this data store
     * @return dataStore data store object associated to this host
     */
    public VMWareDatastore getDatastore(Object key) {
        return this.dataStores.get(key);
    }
    
    /**
     * Sets a data store contained associated to this host.
     * 
     * @param key managed object of the data store
     * @param dataStore data store object associated to this host
     */
    public void putDataStore(Object key, VMWareDatastore dataStore) {
        this.dataStores.put(key, dataStore);
    }    
    
    /**
     * Gets a list of all data stores associated to this host.
     * 
     * @return dataStores data stores associated to this host
     */
    public LinkedHashMap<Object, VMWareDatastore> getAllDataStores() {
        return this.dataStores;
    }
    
    /**
     * This sets all the data stores.
     * 
     * @param allDataStores - All the data stores.
     */
    public void setAllDataStores(LinkedHashMap<Object, VMWareDatastore> allDataStores){
    	this.dataStores = allDataStores;
    }

    /**
     * Returns all available performance counters for the host.
     * 
     * @return a map of performance counters.
     */
	public Map<String, Integer> getAvailablePerfCounters() {
		return availablePerfCounters;
	}

	/**
	 * sets the available performance counters as a map.
	 * 
	 * @param availablePerfCounters - performance counters.
	 */
	public void setAvailablePerfCounters(Map<String, Integer> availablePerfCounters) {
		this.availablePerfCounters = availablePerfCounters;
	}

	/**
	 * Returns the sampling period for the host.
	 *  
	 * @return sampling period.
	 */
	public int getSamplePeriod() {
		return samplePeriod;
	}

	/**
	 * Sets the sampling period for the host.
	 * 
	 * @param samplePeriod sampling period.
	 */
	public void setSamplePeriod(int samplePeriod) {
		this.samplePeriod = samplePeriod;
	}

	/**
	 * Returns the SCSI luns for the host.
	 * 
	 * @return Array of SCSI Luns.
	 */
    public Object[] getAllScsiLUNs() {
		return scsiLUNArray;
	}
    
    /**
     * This method returns a ScsiLun object.
     * 
     * @param i - index of the lun object.
     * @return - Lun object.
     */
    public Object getScsiLUN(int i) {
		return scsiLUNArray[i];
	}

    /**
     * Sets the SCSI LUN Array for the host.
     * 
     * @param scsiLUNArray - SCSI Luns.
     */
	public void setAllScsiLUNs(Object[] scsiLUNArray) {
		this.scsiLUNArray = scsiLUNArray;
	}

	/**
	 * Gets the Host File System Mount Info.
	 * 
	 * @return hostFileSystemMountInfo mount info for hosts containing HostVmfsVolumes
	 */
	public Object[] getHostFileSystemMountInfo(){
	    return this.hostFileSystemMountInfo;
	}
	
	/**
	 * Sets the Host File System Mount Info. 
	 * 
	 * @param hostFileSystemMountInfo mount info for hosts containing HostVmfsVolumes
	 */
	public void setHostFileSystemMountInfo(Object[] hostFileSystemMountInfo){
	    this.hostFileSystemMountInfo = hostFileSystemMountInfo;
	}
	
    /**
     * Gets the general number of statistic samples.
     * 
     * @return the general number of statistic samples
     */
    public int getNumSamples() {
        return numSamples;
    }

    /**
     * Sets the general number of statistic samples.
     * 
     * @param numSamples the general number of statistic samples to set
     */
    public void setNumSamples(int numSamples) {
        this.numSamples = numSamples;
    }

	/**
	 * Returns the host memory size.
	 * 
	 * @return memory size.
	 */
	public long getMemorySize() {
		return memorySize;
	}

	/**
	 * sets the memory size of the host.
	 * 
	 * @param memorySize - memory size.
	 */
	public void setMemorySize(long memorySize) {
		this.memorySize = memorySize;
	}
	
	/**
	 * Returns a stat of a LUN object.
	 * 
	 * @param conanicalName - name of the LUN object.
	 * @return - lun stat.
	 */
	public VMWareLUN getLunStats(String conanicalName) {
		if(lunStats != null){
		    return lunStats.get(conanicalName);
		}

		return null;
	}

	/**
	 * Sets all the lun stats retrieved.
	 * 
	 * @param lunStats - all lun stats.
	 */
	public void setLunStats(HashMap<String, VMWareLUN> lunStats) {
		this.lunStats = lunStats;
	}

    /**
	 * Host System up time.
	 * 
	 * @return long.
	 */
	public long getSystemUpTime() {
		return systemUpTime;
	}

	/**
	 * Host System up time.
	 * 
	 * @param systemUpTime - system up time. 
	 */
	public void setSystemUpTime(long systemUpTime) {
		this.systemUpTime = systemUpTime;
	}

	/**
	 * Memory Usage Avg.
	 * 
	 * @return Memory Usage Avg.
	 */
	public float getMemoryUsageAvg() {
		return memoryUsageAvg;
	}

	/**
	 * Memory Usage Avg.
	 * 
	 * @param memoryUsageAvg -Memory Usage Avg.
	 */
	public void setMemoryUsageAvg(float memoryUsageAvg) {
		this.memoryUsageAvg = memoryUsageAvg;
	}

	/**
	 * Memory Swap Out AVG.
	 * 
	 * @return Memory Swap Out AVG.
	 */
	public long getMemorySwapOutAvg() {
		return memorySwapOutAvg;
	}

	/**
	 * Memory Swap Out AVG.
	 * 
	 * @param memorySwapOutAvg - Memory Swap Out AVG.
	 */
	public void setMemorySwapOutAvg(long memorySwapOutAvg) {
		this.memorySwapOutAvg = memorySwapOutAvg;
	}

	/**
	 * Memory SWAP in AVG.
	 * 
	 * @return - Memory SWAP in AVG.
	 */
	public long getMemorySwapInAvg() {
		return memorySwapInAvg;
	}

	/**
	 * Memory SWAP in AVG.
	 * 
	 * @param memorySwapInAvg - Memory SWAP in AVG.
	 */
	public void setMemorySwapInAvg(long memorySwapInAvg) {
		this.memorySwapInAvg = memorySwapInAvg;
	}

	/**
	 * CPU Usage Avg.
	 * 
	 * @return CPU Usage Avg. 
	 */
	public float getCpuUsageAvg() {
		return cpuUsageAvg;
	}

	/**
	 * CPU Usage Avg.
	 * 
	 * @param cpuUsageAvg - CPU Usage Avg.
	 */
	public void setCpuUsageAvg(float cpuUsageAvg) {
		this.cpuUsageAvg = cpuUsageAvg;
	}

	/**
	 * CPU Usage Avg.
	 * 
	 * @return CPU Usage Avg.
	 */
	public float getCpuUsageMhzAvg() {
		return cpuUsageMhzAvg;
	}

	/**
	 * CPU Usage Avg.
	 * 
	 * @param cpuUsageMhzAvg -CPU Usage Avg.
	 */
	public void setCpuUsageMhzAvg(float cpuUsageMhzAvg) {
		this.cpuUsageMhzAvg = cpuUsageMhzAvg;
	}

	/**
	 * Memory Consumed Avg.
	 * 
	 * @return - Memory Consumed Avg.
	 */
	public float getMemoryConsumedAvg() {
		return memoryConsumedAvg;
	}

	/**
	 * Memory Consumed Avg.
	 * 
	 * @param memoryConsumedAvg -Memory Consumed Avg.
	 */
	public void setMemoryConsumedAvg(float memoryConsumedAvg) {
		this.memoryConsumedAvg = memoryConsumedAvg;
	}

	/**
	 * Memory Overhead.
	 * 
	 * @return Memory Overhead.
	 */
	public float getMemoryOverheadAvg() {
		return memoryOverheadAvg;
	}

	/**
	 * Memory Overhead.
	 * 
	 * @param memoryOverheadAvg - Memory Overhead.
	 */
	public void setMemoryOverheadAvg(float memoryOverheadAvg) {
		this.memoryOverheadAvg = memoryOverheadAvg;
	}

	/**
	 * Memory Overhead.
	 * 
	 * @return Memory Overhead.
	 */
	public float getMemoryBalloonAvg() {
		return memoryBalloon;
	}

	/**
	 * Memory Overhead.
	 * 
	 * @param memoryVmMemCtlAvg - Memory Overhead.
	 */
	public void setMemoryBalloonAvg(float memoryVmMemCtlAvg) {
		this.memoryBalloon = memoryVmMemCtlAvg;
	}

	/**
	 * Memory Overhead.
	 * 
	 * @return Memory Overhead.
	 */
	public float getMemoryBalloonTargetAvg() {
		return memoryBalloonTarget;
	}

	/**
	 * Memory Overhead.
	 * 
	 * @param memoryVmMemCtlTargetAvg - Memory Overhead.
	 */
	public void setMemoryBalloonTargetAvg(float memoryVmMemCtlTargetAvg) {
		this.memoryBalloonTarget = memoryVmMemCtlTargetAvg;
	}

    /**
     * Gets the network usage in KBps.
     * 
     * @return the network usage in KBps
     */
    public Float getAvgNetworkUsage() {
        return this.avgNetworkUsage;
    }

    /**
     * Sets the network usage in KBps.
     * 
     * @param avgNetworkUsage - the network usage in KBps to set
     */
    public void setAvgNetworkUsage(Float avgNetworkUsage) {
        this.avgNetworkUsage = avgNetworkUsage;
    }

	/**
	 * Number of valid CPU usage samples.
	 * @return samples.
	 */
	public int getNumValidCpuUsageSamples() {
		return numValidCpuUsageSamples;
	}

	/**
	 * Number of valid CPU usage samples.
	 * 
	 * @param numValidCpuUsageSamples - Number of valid CPU usage samples.
	 */
	public void setNumValidCpuUsageSamples(int numValidCpuUsageSamples) {
		this.numValidCpuUsageSamples = numValidCpuUsageSamples;
	}	
	
	/**
	 * CPU Hz.
	 * 
	 * @return CPU Hz.
	 */
	public long getCpuHz() {
		return cpuHz;
	}

	/**
	 * CPU Hz.
	 * 
	 * @param cpuHz - CPU Hz.
	 */
	public void setCpuHz(long cpuHz) {
		this.cpuHz = cpuHz;
	}

	/**
	 * Number of CPU Cores.
	 * 
	 * @return - CPU Cores.
	 */
	public int getNumberOfCpuCores() {
		return numberOfCpuCores;
	}

	/**
	 * Number of CPU Cores.
	 * 
	 * @param numberOfCpuCores - CPU Cores.
	 */
	public void setNumberOfCpuCores(int numberOfCpuCores) {
		this.numberOfCpuCores = numberOfCpuCores;
	}

    /**
     * Gets the host name of this host.
     * 
     * @return hostName host name of this host
     */
    public String getHostName() {
       return hostName;
    }

    /**
     * Sets the host name of this host.
     * 
     * @param hostName host name of this host
     */
    public void setHostName(String hostName) {
       this.hostName = hostName;
    }
    
    /**
     * Returns the Host Bud Adapters.
     * 
     * @return - HostBusAdapter array.
     */
	public Object[] getHostBusAdapter() {
		return hostBusAdapter;
	}

	/**
	 * Sets the host bus adapters.
	 * 
	 * @param hostBusAdapter - Array of HostBusAdapters.
	 */
	public void setHostBusAdapter(Object[] hostBusAdapter) {
		this.hostBusAdapter = hostBusAdapter;
	}

	/**
	 * Gets the number of CPU threads.
	 * 
	 * @return number of cpu threads.
	 */
	public int getNumberOfCpuThreads() {
		return numberOfCpuThreads;
	}

	/**
	 * Sets the number of CPU threads.
	 * 
	 * @param numberOfCpuThreads - number of CPU Threads.
	 */
	public void setNumberOfCpuThreads(int numberOfCpuThreads) {
		this.numberOfCpuThreads = numberOfCpuThreads;
	}

	/**
	 * Returns the root resource pool associated with the host.
	 * 
	 * @return - Root ResourcePool.
	 */
	public VMWareResourcePool getRootResourcePool() {
		return rootResourcePool;
	}

	/**
	 * Sets the root resource pool.
	 * 
	 * @param resourcePool - root Resource pool.
	 */
	public void setRootResourcePool(VMWareResourcePool resourcePool) {
		this.rootResourcePool = resourcePool;
	}

	/**
     * Gets the physical network adapter associated to this host.
     * 
     * @param key - identifier of the physical network adapter associated to this host
     * @return the physical network adapter associated to this host
     */
    public VMWarePhysicalNic getPhysicalNic(String key) {
        return this.physicalNics.get(key);
    }

    /**
     * Sets the physical network adapter associated to this host.
     * 
     * @param key - identifier of the physical network adapter associated to this host
     * @param physicalNic - the physical network adapter associated to this host
     */
    public void putPhysicalNic(String key, VMWarePhysicalNic physicalNic) {
        this.physicalNics.put(key, physicalNic);
    }

    /**
     * Gets all the physical network adapters associated to this host.
     * 
     * @return all the physical network adapters associated to this host
     */
    public LinkedHashMap<String, VMWarePhysicalNic> getAllPhysicalNics() {
        return this.physicalNics;
    }

    /**
     * Sets all the physical network adapters associated to this host.
     * 
     * @param physicalNics - all the physical network adapters associated to this host
     */
    public void setAllPhysicalNics(LinkedHashMap<String, VMWarePhysicalNic> physicalNics) {
        this.physicalNics = physicalNics;
    }

    /**
     * Gets the physical network adapter statistics associated to this host.
     * 
     * @param key - identifier of the physical network adapter associated to this host
     * @return the physical network adapter statistics associated to this host
     */
    public VMWarePhysicalNic getPhysicalNicStats(String key) {
        return this.physicalNicStats.get(key);
    }

    /**
     * Sets the physical network adapter statistics associated to this host.
     * 
     * @param key - identifier of the physical network adapter associated to this host
     * @param physicalNicStats - the physical network adapter statistics associated to this host
     */
    public void putPhysicalNicStats(String key, VMWarePhysicalNic physicalNicStats) {
        this.physicalNicStats.put(key, physicalNicStats);
    }

    /**
     * Gets all the physical network adapter statistics associated to this host.
     * 
     * @return all the physical network adapter statistics associated to this host
     */
    public HashMap<String, VMWarePhysicalNic> getAllPhysicalNicStats() {
        return this.physicalNicStats;
    }

    /**
     * Sets all the physical network adapter statistics associated to this host.
     * 
     * @param physicalNicStats - all the physical network adapter statistics associated to this host
     */
    public void setAllPhysicalNicStats(HashMap<String, VMWarePhysicalNic> physicalNicStats) {
        this.physicalNicStats = physicalNicStats;
    }

    /**
     * Gets the product full name of the host.
     * 
     * @return the product full name of the host.
     */
    public String getProductFullName() {
        return this.productFullName;
    }

    /**
     * Sets the product full name of the host.
     * 
     * @param productFullName - the product full name of the host to set.
     */
    public void setProductFullName(String productFullName) {
        this.productFullName = productFullName;
    }

    /**
     * Gets the version of the host.
     * 
     * @return the version of the host
     */
    public String getEsxVersion(){
        return this.esxVersion;
    }

    /**
     * Sets the version of the host.
     * 
     * @param esxVersion - the version of the host to set
     */
    public void setEsxVersion(String esxVersion){
        this.esxVersion = esxVersion;
    }

    /**
     * Gets the build string of the host.
     * 
     * @return the build string of the host
     */
    public String getEsxBuild() {
        return this.esxBuild;
    }

    /**
     * Sets the build string of the host.
     * 
     * @param esxBuild - the build string of the host to set
     */
    public void setEsxBuild(String esxBuild) {
        this.esxBuild = esxBuild;
    }

    /**
     * Gets the power state of this virtual machine.
     * 
     * @return powerState power state of this virtual machine
     */
    public String getPowerState() {
        return this.powerState;
    }

    /**
     * Sets the power state of this virtual machine.
     * 
     * @param powerState power state of this virtual machine
     */
    public void setPowerState(String powerState) {
        this.powerState = powerState;
    }

    /**
     * Gets the connection state of this virtual machine.
     * 
     * @return connectionState connection state of this virtual machine
     */
    public String getConnectionState() {
        return this.connectionState;
    }

    /**
     * Sets the connection state of this virtual machine.
     * 
     * @param connectionState connection state of this virtual machine
     */
    public void setConnectionState(String connectionState) {
        this.connectionState = connectionState;
    }
    
    /**
     * Gets the Managed Object Reference of the Storage System of this Host.
     * 
     * @return storageSystemMor storage system managed object reference
     */
    public Object getStorageSystemMor(){
        return this.storageSystemMor;
    }
    
    /**
     * Sets the Managed Object Reference of the Storage System of this Host.
     * 
     * @param storageSystemMor storageSystemMor storage system managed object reference
     */
    public void setStorageSystemMor(Object storageSystemMor){
        this.storageSystemMor = storageSystemMor;
    }

    /**
     * Gets the effective cpu limit.
     * @return effective cpu limit
     */
    public Long getEffectiveCpu() {
        return effectiveCpu;
    }

    /**
     * Sets the effective cpu limit.
     * @param effectiveCpu cpu limit
     */
    public void setEffectiveCpu(Long effectiveCpu) {
        this.effectiveCpu = effectiveCpu;
    }

    /**
     * Gets the effective memory limit.
     * @return effective memory limit
     */
    public Long getEffectiveMemory() {
        return effectiveMemory;
    }

    /**
     * Sets the effective memory limit.
     * @param effectiveMemory memory limit
     */
    public void setEffectiveMemory(Long effectiveMemory) {
        this.effectiveMemory = effectiveMemory;
    }

    /**
     * Gets the value for maintenancemode.
     * @return maintenancemode
     */
    public Boolean getMaintenanceMode() {
        return maintenanceMode;
    }

    /**
     * Sets the inMaintenanceMode flag.
     * @param maintenanceMode flag
     */
    public void setMaintenanceMode(Boolean maintenanceMode) {
        this.maintenanceMode = maintenanceMode;
    }
}

