/*
 * Copyright (c) 2008 Akorri Networks, Inc. All Rights Reserved. This software is provided
 * under license and is not sold by Akorri Networks. This software may be used only in accordance
 * with the terms of an express written license agreement with Akorri Networks, and cannot be used,
 * copied, modified, or in any way distributed without the express permission of Akorri Networks.
 *
 */

package com.netapp.collectors.vmware;

import java.util.LinkedHashMap;

/**
 * Collection domain object for Resource Pools.
 * 
 * @author slaplante
 */
public class VMWareResourcePool {

    /** Name of this resource pool. */
    private String name;
    
    /** Managed object of this resource pool. */
    private Object managedObject;
    
    /** Parent managed object id of this resource pool. */
    private Object parentManagedObject;
    
    /** Overall status of this resource pool. */
    private String overallStatus;
    
    /** CPU Shares for this resource pool. */
    private int cpuShares;
    
    /** CPU limit for this resource pool. */
    private Long cpuLimit;
    
    /** CPU reservation for this resource pool. */
    private Long cpuReservation;
    
    /** CPU overall usage for this resource pool. */
    private Long cpuOverallUsage;
    
    /** Memory shares for this resource pool. */
    private int memoryShares;
    
    /** Memory limit for this resource pool. */
    private Long memoryLimit;
    
    /** Memory reservation for this resource pool. */
    private Long memoryReservation;
    
    /** Memory overall usage for this resource pool. */
    private Long memoryOverallUsage;
    
    /** List of virtual machines related associated to this host. */
    private LinkedHashMap<Object, VMWareVirtualMachine> virtualMachines = new LinkedHashMap<Object, VMWareVirtualMachine>();

    /** List of child resource pools associated to this resource. */
    private LinkedHashMap<Object, VMWareResourcePool> childResourcePools = new LinkedHashMap<Object, VMWareResourcePool>();
    
    /**
     * Constructor.
     * 
     * @param managedObject managed object id of this resource pool
     */
    public VMWareResourcePool(Object managedObject){
        this.managedObject = managedObject;
    }
    
    /**
     * Gets the name of this resource pool.
     * 
     * @return name name of this resource pool
     */
    public String getName() {
        return name;
    }

    /**
     * Sets the name of this resource pool.
     * 
     * @param name name of this resource pool
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Gets the managed object of this resource pool.
     * 
     * @return managedObject managed object of this resource pool
     */
    public Object getManagedObject(){
        return this.managedObject;
    }
    
    /**
     * Sets the managed object of this resource pool.
     * 
     * @param managedObject managed object of this resource pool
     */
    public void setManagedObject(String managedObject){
        this.managedObject = managedObject;
    }
    
    /**
     * Gets the parent managed object of this resource pool.
     * 
     * @return parentManagedObject parent managed object of this resource pool
     */
    public Object getParentManagedObject() {
        return parentManagedObject;
    }

    /**
     * Sets the parent managed object of this resource pool.
     * 
     * @param parentManagedObject parent managed object of this resource pool
     */
    public void setParentManagedObject(Object parentManagedObject) {
        this.parentManagedObject = parentManagedObject;
    }
    
    /**
     * Gets the overall status of this resource pool.
     * 
     * @return overallStatus overall status of this resource pool
     */
    public String getOverallStatus() {
        return overallStatus;
    }

    /**
     * Sets the overall status of this resource pool.
     * 
     * @param overallStatus overall status of this resource pool
     */
    public void setOverallStatus(String overallStatus) {
        this.overallStatus = overallStatus;
    }

    /**
     * Gets the number of CPU shares of this resource pool.
     * 
     * @return cpuShares number of CPU shares of this resource pool
     */
    public int getCpuShares() {
        return cpuShares;
    }

    /**
     * Sets the number of CPU shares of this resource pool.
     * 
     * @param cpuShares number of CPU shares of this resource pool
     */
    public void setCpuShares(int cpuShares) {
        this.cpuShares = cpuShares;
    }

    /**
     * Gets the CPU limit of this resource pool.
     * 
     * @return cpuLimit CPU limit of this resource pool
     */
    public Long getCpuLimit() {
        return cpuLimit;
    }

    /**
     * Sets the CPU limit of this resource pool.
     * 
     * @param cpuLimit CPU limit of this resource pool
     */
    public void setCpuLimit(Long cpuLimit) {
        this.cpuLimit = cpuLimit;
    }

    /**
     * Gets the CPU reservation of this resource pool.
     * 
     * @return cpuReservation CPU reservation of this resource pool
     */
    public Long getCpuReservation(){
    	return this.cpuReservation;
    }
    
    /**
     * Sets the CPU reservation of this resource pool.
     * 
     * @param cpuReservation CPU reservation of this resource pool
     */
    public void setCpuReservation(Long cpuReservation){
    	this.cpuReservation = cpuReservation;
    }
    
    /**
     * Gets the CPU overall usage of this resource pool.
     * 
     * @return cpuOverallUsage CPU overall usage of this resource pool
     */
    public Long getCpuOverallUsage(){
    	return this.cpuOverallUsage;
    }
    
    /**
     * Sets the CPU overall usage of this resource pool.
     * 
     * @param cpuOverallUsage CPU overall usage of this resource pool
     */
    public void setCpuOverallUsage(Long cpuOverallUsage){
    	this.cpuOverallUsage = cpuOverallUsage;
    }
    
    /**
     * Gets the number of memory shares of this resource pool.
     * 
     * @return memoryShares number of memory shares of this resource pool
     */
    public int getMemoryShares() {
        return memoryShares;
    }

    /**
     * Sets the number of memory shares of this resource pool.
     * 
     * @param memoryShares number of memory shares of this resource pool
     */
    public void setMemoryShares(int memoryShares) {
        this.memoryShares = memoryShares;
    }

    /**
     * Gets the memory limit of this resource pool.
     * 
     * @return memoryLimit memory limit of this resource pool
     */
    public Long getMemoryLimit() {
        return memoryLimit;
    }

    /**
     * Sets the memory limit of this resource pool.
     * 
     * @param memoryLimit memory limit of this resource pool
     */
    public void setMemoryLimit(Long memoryLimit) {
        this.memoryLimit = memoryLimit;
    }

    /**
     * Gets the memory reservation of this resource pool.
     * 
     * @return memoryReservation memory reservation of this resource pool
     */
    public Long getMemoryReservation(){
    	return this.memoryReservation;
    }
    
    /**
     * Sets the memory reservation of this resource pool.
     * 
     * @param memoryReservation memory reservation of this resource pool
     */
    public void setMemoryReservation(Long memoryReservation){
    	this.memoryReservation = memoryReservation;
    }
    
    /**
     * Gets the memory overall usage of this resource pool.
     * 
     * @return memoryOverallUsage memory overall usage of this resource pool
     */
    public Long getMemoryOverallUsage(){
    	return this.memoryOverallUsage;
    }
    
    /**
     * Sets the memory overall usage of this resource pool.
     * 
     * @param memoryOverallUsage memory overall usage of this resource pool
     */
    public void setMemoryOverallUsage(Long memoryOverallUsage){
    	this.memoryOverallUsage = memoryOverallUsage;
    }

    /**
     * Gets a list of all virtual machines associated to this resource pool.
     * 
     * @return virtualMachines virtual machines associated to this resource pool
     */
    public LinkedHashMap<Object, VMWareVirtualMachine> getAllVirtualMachines() {
        return this.virtualMachines;
    }

    /**
     * Gets a virtual machine contained associated to this resource pool.
     * 
     * @param key managed object of this virtual machine
     * @return virtualMachine virtual machine object associated to this resource pool
     */
    public VMWareVirtualMachine getVirtualMachine(Object key) {
        return this.virtualMachines.get(key);
    }

    /**
     * Sets a virtual machine associated to this resource pool.
     * 
     * @param key managed object of the virtual machine
     * @param virtualMachine virtual machine object associated to this resource pool
     */
    public void putVirtualMachine(Object key, VMWareVirtualMachine virtualMachine) {
        this.virtualMachines.put(key, virtualMachine);
    }

    /**
     * Sets all virtual machines associated to this resource pool.
     * 
     * @param virtualMachines - list of all the virtual machines.
     */
    public void setAllVirtualMachines(LinkedHashMap<Object, VMWareVirtualMachine> virtualMachines) {
        this.virtualMachines = virtualMachines;
    }

    /**
     * Gets a list of all child resource pools associated to this resource pool.
     * 
     * @return childResourcePools all child resource pools associated to this resource pool
     */
    public LinkedHashMap<Object, VMWareResourcePool> getAllChildResourcePools() {
        return this.childResourcePools;
    }

    /**
     * Gets a child resource pool associated to this resource pool.
     * 
     * @param key managed object of this resource pool
     * @return childResourcePool child resource pool object associated to this resource pool
     */
    public VMWareResourcePool getChildResourcePool(Object key) {
        return this.childResourcePools.get(key);
    }
    
    /**
     * Sets a child resource pool of this resource pool.
     * 
     * @param key managed object of the child resource pool
     * @param childResourcePool child resource pool associated to this resource pool
     */
    public void putChildResourcePool(Object key, VMWareResourcePool childResourcePool) {
        this.childResourcePools.put(key, childResourcePool);
    }
    
}
