/*
 * Copyright (c) 2008 Akorri Networks, Inc. All Rights Reserved. This software is provided
 * under license and is not sold by Akorri Networks. This software may be used only in accordance
 * with the terms of an express written license agreement with Akorri Networks, and cannot be used,
 * copied, modified, or in any way distributed without the express permission of Akorri Networks.
 *
 */

package com.netapp.collectors.vmware;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.lang.StringBuilder;


/**
 * Collection domain object for VMWare Virtual Machines.
 * 
 * @author slaplante
 */
public class VMWareVirtualMachine {

    /** Name of this virtual machine. */
    private String name;
    
    /** Managed object of this virtual machine. */
    private Object managedObject;
    
    /** This virtual machine's power state. */
    private String powerState;
    
    /** This virtual machine's connection state. */
    private String connectionState;
    
    /** This virtual machine's guest state. This typically is "running" or "notrunning".*/
    private String guestState;
    
    /** Holds the guest family information, typically the os type is determined from this. */
    private String guestFamily;

    /** The guest full name obtained from the "guest" property. */
    private String guestGuestFullName;

    /**
     * The guest full name obtained from the "config" property.
     * 
     * This field is used when the "guestFullName" of the "guest" property is not available.
     */
    private String configGuestFullName;

    /** Holds the status of vmware tools installed on this virtual machine. Typically "toolsOk" or "toolsOld". */
    private String toolsStatus;
    
    /** Holds the version of vmware tools installed on this virtual machine. Typically "8195" or "8290". */
    private String toolsVersion;
    
    /** The ip address list of this virtual machine. */
    private ArrayList<String> ipAddressList = new ArrayList<String>();
    
    /** Product name of this virtual machine. */
    private String productName;
    
    /** Virtual uuid of this virtual machine. */
    private String uuid;
    
    /** List of data stores related associated to this host. */
    private LinkedHashMap<Object, VMWareDatastore> dataStores = new LinkedHashMap<Object, VMWareDatastore>();
    
    /** List of virtual disks related associated to this host. */
    private LinkedHashMap<String, VMWareVirtualDisk> virtualDisks = new LinkedHashMap<String, VMWareVirtualDisk>();
    
    /** Host this virtual machine is associated with. */
    private VMWareHost host;
    
    /** Resource Pool this virtual machine is associated with. */
    private VMWareResourcePool resourcePool;
    
    /** Host Name of this virtual machine. */
    private String hostName;

    /** Number of CPUs.*/
    private int numberOfCpu;

    /** Number of statistic samples. */
    private int numSamples;
    /** CPU Usage MHz. */
    private float cpuUsedMHz;
    /** CPU Usage Avg. */
    private float cpuUsedAvg;
    /** CPU Ready. */
    private float cpuReady;
    /** CPU Ready Ms.*/
    private float cpuReadyMs;
    /** Memory Swap out avg.*/
    private long memorySwapOutAvg;
    /** Memory swap in avg.*/
    private long memorySwapInAvg; 
    /** Memory.*/
    private float memoryBalloon;
    /** Memory.*/
    private float memoryBalloonTarget;
    /** Memory Consumed Avg.*/
    private float memoryConsumedAvg;
    /**CPU Shares, Limit, Reservations.*/
    private long cpuShares, cpuLimit, cpuReservation;
    /**Memory Shares, Limit, Reservations.*/    
    private long memoryShares, memoryLimit, memoryReservation;
    /** Network usage in KBps. */
    private float avgNetworkUsage;
    /** Capacity Total.*/
    private Map<VMWareDatastore, Float> capacityTotal = new HashMap<VMWareDatastore, Float>();
    /** Capacity Total MBs.*/
    private Map<VMWareDatastore, Float> capacityTotalMBs = new HashMap<VMWareDatastore, Float>();
    /** vmdk Total.*/
    private Map<VMWareDatastore, Float> vmdkTotal = new HashMap<VMWareDatastore, Float>();
    /** vmdk Total MBs.*/
    private Map<VMWareDatastore, Float> vmdkTotalMBs = new HashMap<VMWareDatastore, Float>();
    /** This is the SCSI LUns for the VM. This is a VMWare API object need to cast it to ScsiLun object in VMWare.*/
    private ArrayList<Object> scsiLUNs = new ArrayList<Object>();
    /**All LUN Statistics.*/
    private HashMap<String, Object> lunStats = new HashMap<String, Object>();
    /** CPU and Memory share level.*/
    private String cpuShareLevel, memoryShareLevel;
    /** Memory MB.*/
    private long memoryMB;
    /** Disk layout information.*/
    private ArrayList<String> vmdkFolderPaths = new ArrayList<String>();
    /** Virtual Machine File Information and Locations.*/
    private String logDirectory, snapshotDirectory, suspendDirectory, vmPathName;
    /** Folder Information for the VM.*/
    private Set<String> folders = new HashSet<String>();
    /**Guest ID.*/
    private String guestId;
    /**Virtual Devices.*/
    private Object virtualDevice[];    
    /***/
    private Object[] fileLayoutDiskLayout; 
    
    /** VM hardware version. */
    private String vmHardwareVersion;
    /** Sorted map extra configuration parameters associated with this guest. */
    private HashMap<String, String> extraConfig = new HashMap<String, String>();
    
    /**Template.*/
    private boolean isTemplate;
    
    /** Boolean value describing if this virtual machine is the primary if it is fault tolerant enabled. */
    private boolean isPrimary = false;
    
    /** Boolean value describing if this virtual machine is the secondary if it is fault tolerant enabled. */
    private boolean isSecondary = false;
    
    /**
     * Constructor.
     * 
     * @param managedObject managed object of this virtual machine
     */
    public VMWareVirtualMachine(Object managedObject){
        this.managedObject = managedObject;
    }
    
    /**
     * Gets the name of this collected virtual machine.
     * 
     * @return name name of this collected virtual machine
     */
    public String getName(){
        return this.name;
    }
    
    /**
     * Sets the name of this collected virtual machine.
     * 
     * @param name name of this collected virtual machine
     */
    public void setName(String name){
        this.name = name;
    }
    
    /**
     * Gets the managed object of this virtual machine.
     * 
     * @return managedObjectId managed object of this virtual machine
     */
    public Object getManagedObject() {
        return managedObject;
    }

    /**
     * Sets the managed object of this virtual machine.
     * 
     * @param managedObject managed object of this virtual machine
     */
    public void setManagedObject(Object managedObject) {
        this.managedObject = managedObject;
    }
    
    /**
     * Gets the power state of this virtual machine.
     * 
     * @return powerState power state of this virtual machine
     */
    public String getPowerState() {
        return this.powerState;
    }

    /**
     * Sets the power state of this virtual machine.
     * 
     * @param powerState power state of this virtual machine
     */
    public void setPowerState(String powerState) {
        this.powerState = powerState;
    }

    /**
     * Gets the connection state of this virtual machine.
     * 
     * @return connectionState connection state of this virtual machine
     */
    public String getConnectionState() {
        return this.connectionState;
    }

    /**
     * Sets the connection state of this virtual machine.
     * 
     * @param connectionState connection state of this virtual machine
     */
    public void setConnectionState(String connectionState) {
        this.connectionState = connectionState;
    }
    
    /**
     * Gets the guest state of this virtual machine. 
     * 
     * @return guestState guest state of this virtual machine
     */
    public String getGuestState() {
        return this.guestState;
    }

    /**
     * Sets the guest state of this virtual machine.
     * 
     * @param guestState guest state of this virtual machine
     */
    public void setGuestState(String guestState) {
        this.guestState = guestState;
    }
    
    /**
     * Sets the guest family of this virtual machine.
     * 
     * @return guestFamily guest family of this virtual machine
     */
    public String getGuestFamily() {
        return this.guestFamily;
    }

    /**
     * Gets the guest family of this virtual machine.
     * 
     * @param guestFamily guest family of this virtual machine
     */
    public void setGuestFamily(String guestFamily) {
        this.guestFamily = guestFamily;
    }

    /**
     * Gets the guest full name obtained from the "guest" property.
     * 
     * @return the guest full name obtained from the "guest" property
     */
    public String getGuestGuestFullName(){
    	return this.guestGuestFullName;
    }

    /**
     * Sets the guest full name obtained from the "guest" property.
     * 
     * @param guestGuestFullName - the guest full name obtained from the "guest" property to set
     */
    public void setGuestGuestFullName(String guestGuestFullName){
    	this.guestGuestFullName = guestGuestFullName;
    }

    /**
     * Gets the guest full name obtained from the "config" property.
     * 
     * @return the guest full name obtained from the "config" property
     */
    public String getConfigGuestFullName() {
        return configGuestFullName;
    }

    /**
     * Sets the guest full name obtained from the "config" property.
     * 
     * @param configGuestFullName - the guest full name obtained from the "config" property to set
     */
    public void setConfigGuestFullName(String configGuestFullName) {
        this.configGuestFullName = configGuestFullName;
    }

    /**
     * Gets the tools status of this virtual machine.
     * 
     * @return toolsStatus tools status of this virtual machine
     */
    public String getToolsStatus() {
        return toolsStatus;
    }

    /**
     * Sets the tools status of this virtual machine.
     * 
     * @param toolsStatus tools status of this virtual machine
     */
    public void setToolsStatus(String toolsStatus) {
        this.toolsStatus = toolsStatus;
    }
    
    /**
     * Gets the tools version of this virtual machine.
     * 
     * @return toolsVersion tools version of this virtual machine
     */
    public String getToolsVersion() {
        return toolsVersion;
    }

    /**
     * Sets the tools version of this virtual machine.
     * 
     * @param toolsVersion tools version of this virtual machine
     */
    public void setToolsVersion(String toolsVersion) {
        this.toolsVersion = toolsVersion;
    }
    
    /**
     * Gets the ip address list of this virtual machine.
     * 
     * @return ipAddressList ip address list of this virtual machine
     */
    public ArrayList<String> getIpAddressList(){
        return this.ipAddressList;
    }
    
    /**
     * Sets the ip address list of this virtual machine.
     * 
     * @param ipAddressList ip address list of this virtual machine
     */
    public void setIpAddressList(ArrayList<String> ipAddressList){
        this.ipAddressList = ipAddressList;
    }

    /**
     * Gets the product name of this virtual machine.
     * 
     * @return productName product name of this virtual machine
     */
    public String getProductName() {
        return this.productName;
    }

    /**
     * Sets the product name of this virtual machine.
     * 
     * @param productName product name of this virtual machine
     */
    public void setProductName(String productName) {
        this.productName = productName;
    }

    /**
     * Gets the virtual uuid of this virtual machine.
     * 
     * @return uuid virtual uuid of this virtual machine
     */
    public String getUuid() {
        return uuid;
    }

    /**
     * Sets the virtual uuid of this virtual machine.
     * 
     * @param uuid virtual uuid of this virtual machine
     */
    public void setUuid(String uuid) {
        this.uuid = uuid;
    }
    
    /**
     * Sets the host name of this virtual machine.
     * 
     * @return hostName host name of this virtual machine
     */
    public String getHostName() {
        return hostName;
    }

    /**
     * Gets the host name of this virtual machine.
     * 
     * @param hostName host name of this virtual machine
     */
    public void setHostName(String hostName) {
        this.hostName = hostName;
    }

    /**
     * Gets a data store associated to this.
     * 
     * @param key managed object of this data store
     * @return dataStore data store object associated to this
     */
    public VMWareDatastore getDatastores(Object key) {
        return this.dataStores.get(key);
    }
    
    /**
     * Sets a data store contained associated to this.
     * 
     * @param key managed object of the data store
     * @param dataStore data store object associated to this
     */
    public void putDataStores(Object key, VMWareDatastore dataStore) {
        this.dataStores.put(key, dataStore);
    }
    
    /**
     * Gets a list of all data stores associated to this.
     * 
     * @return dataStores data stores associated to this.
     */
    public LinkedHashMap<Object, VMWareDatastore> getAllDataStores() {
        return this.dataStores;
    }
    
    /**
     * Set all the datastores.
     * 
     * @param datastores - list of datastores.
     */
    public void setAllDataStores(LinkedHashMap<Object, VMWareDatastore> datastores) {
        this.dataStores = datastores;
    }    

    /**
     * Gets a virtual disk associated to this scsiId.
     * 
     * @param scsiId scsiId from instance data for this virtual disk
     * @return virtualDisk object associated to this scsiId
     */
    public VMWareVirtualDisk getVirtualDisk(String scsiId) {
        return this.virtualDisks.get(scsiId);
    }
    
    /**
     * Sets a virtual disk contained associated to this scsiId.
     * 
     * @param scsiId scsiId from instance data for this virtual disk
     * @param virtualDisk virtualDisk object associated to this scsiId
     */
    public void putVirtualDisk(String scsiId, VMWareVirtualDisk virtualDisk) {
        this.virtualDisks.put(scsiId, virtualDisk);
    }
    
    /**
     * Gets a list of all virtual disks associated to this.
     * 
     * @return virtualDisks associated to this.
     */
    public LinkedHashMap<String, VMWareVirtualDisk> getAllVirtualDisks() {
        return this.virtualDisks;
    }
    
    /**
     * Set all the virtualDisks.
     * 
     * @param virtualDisks - list of virtualDisks.
     */
    public void setAllVirtualDisks(LinkedHashMap<String, VMWareVirtualDisk> virtualDisks) {
        this.virtualDisks = virtualDisks;
    }    

    /**
     * Gets the ESX host this virtual machine is associated with.
     *
     * @return host ESX host this virtual machine is associated with
     */
    public VMWareHost getHost(){
        return this.host;
    }

    /**
     * Sets the ESX host this virtual machine is associated with.
     *
     * @param host ESX host this virtual machine is associated with
     */
    public void setHost(VMWareHost host){
        this.host = host;
    }
    
    /**
     * Sets the Resource Pool this virtual machine is associated with.
     * 
     * @return resourcePool resource pool this virtual machine is associated with
     */
    public VMWareResourcePool getResourcePool(){
        return this.resourcePool;
    }
    
    /**
     * Gets the Resource Pool this virtual machine is associated with.
     * 
     * @param resourcePool resource pool this virtual machine is associated with
     */
    public void setResourcePool(VMWareResourcePool resourcePool){
        this.resourcePool = resourcePool;
    }
    
    /**
     * Number of CPUs.
     * 
     * @return Number of CPUs.
     */
    public int getNumberOfCpu() {
        return numberOfCpu;
    }

    /**
     * Sets the number of CPUs.
     * 
     * @param numberOfCpu number of CPUs.
     */
    public void setNumberOfCpu(int numberOfCpu) {
        this.numberOfCpu = numberOfCpu;
    }

    /**
     * Gets the general number of statistic samples.
     * 
     * @return the general number of statistic samples
     */
    public int getNumSamples() {
        return numSamples;
    }

    /**
     * Sets the general number of statistic samples.
     * 
     * @param numSamples the general number of statistic samples to set
     */
    public void setNumSamples(int numSamples) {
        this.numSamples = numSamples;
    }

    /**
     * CPU Used MHz.
     * 
     * @return MHz.
     */
    public float getCpuUsedMHz() {
        return cpuUsedMHz;
    }

    /**
     * CPU Used MHz.
     * 
     * @param cpuUsedMHz - MHz.
     */
    public void setCpuUsedMHz(float cpuUsedMHz) {
        this.cpuUsedMHz = cpuUsedMHz;
    }

    /**
     * CPU Used Avg.
     * 
     * @return CPU Used Avg.,
     */
    public float getCpuUsedAvg() {
        return cpuUsedAvg;
    }

    /**
     * CPU Used Avg.
     * 
     * @param cpuUsedAvg - CPU Used Avg.
     */
    public void setCpuUsedAvg(float cpuUsedAvg) {
        this.cpuUsedAvg = cpuUsedAvg;
    }

    /**
     * CPU Ready.
     * 
     * @return CPU Ready.
     */
    public float getCpuReady() {
        return cpuReady;
    }

    /**
     * CPU Ready.
     * 
     * @param cpuReady - CPU Ready.
     */
    public void setCpuReady(float cpuReady) {
        this.cpuReady = cpuReady;
    }

    /**
     * CPU Ready Ms.
     * 
     * @return CPU Ready Ms.
     */
    public float getCpuReadyMs() {
        return cpuReadyMs;
    }

    /**
     * CPU Ready Ms.
     * 
     * @param cpuReadyMs - CPU Ready Ms.
     */
    public void setCpuReadyMs(float cpuReadyMs) {
        this.cpuReadyMs = cpuReadyMs;
    }
        
    /**
     * Memory Swap Out AVG.
     * 
     * @return Memory Swap Out AVG.
     */
    public long getMemorySwapOutAvg() {
        return memorySwapOutAvg;
    }

    /**
     * Memory Swap Out AVG.
     * 
     * @param memorySwapOutAvg - Memory Swap Out AVG.
     */
    public void setMemorySwapOutAvg(long memorySwapOutAvg) {
        this.memorySwapOutAvg = memorySwapOutAvg;
    }

    /**
     * Memory SWAP in AVG.
     * 
     * @return - Memory SWAP in AVG.
     */
    public long getMemorySwapInAvg() {
        return memorySwapInAvg;
    }

    /**
     * Memory SWAP in AVG.
     * 
     * @param memorySwapInAvg - Memory SWAP in AVG.
     */
    public void setMemorySwapInAvg(long memorySwapInAvg) {
        this.memorySwapInAvg = memorySwapInAvg;
    }
        
    /**
     * Memory Overhead.
     * 
     * @return Memory Overhead.
     */
    public float getMemoryBalloonAvg() {
        return memoryBalloon;
    }

    /**
     * Memory Overhead.
     * 
     * @param memoryVmMemCtlAvg - Memory Overhead.
     */
    public void setMemoryBalloonAvg(float memoryVmMemCtlAvg) {
        this.memoryBalloon = memoryVmMemCtlAvg;
    }

    /**
     * Memory Overhead.
     * 
     * @return Memory Overhead.
     */
    public float getMemoryBalloonTargetAvg() {
        return memoryBalloonTarget;
    }

    /**
     * Memory Overhead.
     * 
     * @param memoryVmMemCtlTargetAvg - Memory Overhead.
     */
    public void setMemoryBalloonTargetAvg(float memoryVmMemCtlTargetAvg) {
        this.memoryBalloonTarget = memoryVmMemCtlTargetAvg;
    }

    /**
     * Memory Consumed Avg.
     * 
     * @return - Memory Consumed Avg.
     */
    public float getMemoryConsumedAvg() {
        return memoryConsumedAvg;
    }

    /**
     * Memory Consumed Avg.
     * 
     * @param memoryConsumedAvg -Memory Consumed Avg.
     */
    public void setMemoryConsumedAvg(float memoryConsumedAvg) {
        this.memoryConsumedAvg = memoryConsumedAvg;
    }

    /**
     * CPU Shares.
     * 
     * @return CPU Shares.
     */
    public long getCpuShares() {
        return cpuShares;
    }

    /**
     * CPU Shares.
     * 
     * @param cpuShares - CPU Shares.
     */
    public void setCpuShares(long cpuShares) {
        this.cpuShares = cpuShares;
    }

    /**
     * CPU Limit.
     * 
     * @return CPU Limit.
     */
    public long getCpuLimit() {
        return cpuLimit;
    }

    /**
     * CPU Limit.
     * 
     * @param cpuLimit - CPU Limit.
     */
    public void setCpuLimit(long cpuLimit) {
        this.cpuLimit = cpuLimit;
    }

    /**
     * CPU Reservation.
     * 
     * @return -CPU Reservation. 
     */
    public long getCpuReservation() {
        return cpuReservation;
    }

    /**
     * CPU Reservation.
     * 
     * @param cpuReservation - CPU Reservation.
     */
    public void setCpuReservation(long cpuReservation) {
        this.cpuReservation = cpuReservation;
    }

    /**
     * Memory Shares.
     * 
     * @return Memory Shares.
     */
    public long getMemoryShares() {
        return memoryShares;
    }

    /**
     * Memory Shares.
     * 
     * @param memoryShares - Memory Shares.
     */
    public void setMemoryShares(long memoryShares) {
        this.memoryShares = memoryShares;
    }
        
    /**
     * Memory Limit.
     * 
     * @return Memory Limit.
     */
    public long getMemoryLimit() {
        return memoryLimit;
    }

    /**
     * Memory Limit.
     * 
     * @param memoryLimit - Memory Limit.
     */
    public void setMemoryLimit(long memoryLimit) {
        this.memoryLimit = memoryLimit;
    }

    /**
     * Memory Reservation.
     * 
     * @return Memory Reservation. 
     */
    public long getMemoryReservation() {
        return memoryReservation;
    }

    /**
     * Memory Reservation.
     * 
     * @param memoryReservation - Memory Reservation.
     */
    public void setMemoryReservation(long memoryReservation) {
        this.memoryReservation = memoryReservation;
    }

    /**
     * Gets the network usage in KBps.
     * 
     * @return the network usage in KBps
     */
    public Float getAvgNetworkUsage() {
        return this.avgNetworkUsage;
    }

    /**
     * Sets the network usage in KBps.
     * 
     * @param avgNetworkUsage - the network usage in KBps to set
     */
    public void setAvgNetworkUsage(Float avgNetworkUsage) {
        this.avgNetworkUsage = avgNetworkUsage;
    }

    /**
     * Capacity Total.
     * 
     * @param dataStore - data store object.
     * @return - capacity total.
     */
    public Float getCapacityTotal(VMWareDatastore dataStore) {
        return capacityTotal.get(dataStore);
    }

    /**
     * Capacity Total.
     * 
     * @param dataStore - data store object.
     * @param capacityTotal - Capacity Total.
     */
    public void addCapacityTotal(VMWareDatastore dataStore, Float capacityTotal) {
        Float currentCapacity = this.getCapacityTotal(dataStore);
        currentCapacity = addTotals(capacityTotal, currentCapacity);
        this.capacityTotal.put(dataStore, currentCapacity);
    }

    /**
     * This is a private helper method to assist in the calculations.
     * 
     * @param newValue - New value to be added to the collection.
     * @param currentValue - Current existing value.
     * @return - a Float value.
     */
    private Float addTotals(Float newValue, Float currentValue) {
        if(currentValue !=null){
            currentValue += newValue;
        }else{
            currentValue = newValue;
        }
        return currentValue;
    }

    /**
     * Capacity Total MBs.
     * 
     * @param dataStore - data store object.
     * @return - Capacity Total MBs.
     */
    public Float getCapacityTotalMBs(VMWareDatastore dataStore) {
        return capacityTotalMBs.get(dataStore);
    }

    /**
     * Capacity Total MBs.
     * 
     * @param dataStore - data store object.
     * @param capacityTotalMBs - Capacity Total MBs.
     */
    public void addCapacityTotalMBs(VMWareDatastore dataStore, Float capacityTotalMBs) {
        Float currentCapacityMBs = this.getCapacityTotalMBs(dataStore);
        currentCapacityMBs = addTotals(capacityTotalMBs, currentCapacityMBs);
        this.capacityTotalMBs.put(dataStore, currentCapacityMBs);
    }

    /**
     * VMDK Total.
     * 
     * @param dataStore - data store object.
     * @return VMDK Total.
     */
    public Float getVmdkTotal(VMWareDatastore dataStore) {
        return vmdkTotal.get(dataStore);
    }

    /**
     * VMDK Total.
     * 
     * @param dataStore - data store object.
     * @param vmdkTotal - VMDK Total.
     */
    public void addVmdkTotal(VMWareDatastore dataStore, Float vmdkTotal) {
        Float currentVMDKTotal = this.getVmdkTotal(dataStore);
        currentVMDKTotal = addTotals(vmdkTotal, currentVMDKTotal);              
        this.vmdkTotal.put(dataStore, currentVMDKTotal);
    }

    /**
     * VMDK Total MBs.
     * 
     * @param dataStore - data store object.
     * @return - VMDK Total MBs.
     */
    public Float getVmdkTotalMBs(VMWareDatastore dataStore) {
        return vmdkTotalMBs.get(dataStore);
    }

    /**
     * VMDK Total MBs.
     * 
     * @param dataStore - data store object.
     * @param vmdkTotalMBs - VMDK Total MBs.
     */
    public void addVmdkTotalMBs(VMWareDatastore dataStore, Float vmdkTotalMBs) {
        Float currentVMDKTotalMBs = this.getVmdkTotalMBs(dataStore);
        currentVMDKTotalMBs = addTotals(vmdkTotalMBs, currentVMDKTotalMBs);
        this.vmdkTotalMBs.put(dataStore, currentVMDKTotalMBs);
    }
        
    /**
     * Returns the SCSI luns for the VM.
     * 
     * @return Array of SCSI Luns.
     */
    public ArrayList<Object> getAllScsiLUNs() {
        return scsiLUNs;
    }

    /**
     * Sets the SCSI LUN Array for the VM.
     * 
     * @param scsiLUN - SCSI Luns.
     */
    public void putScsiLUN(Object scsiLUN) {
        this.scsiLUNs.add(scsiLUN);
    }
        
    /**
     * Returns a stat of a LUN object.
     * 
     * @param conanicalName - name of the LUN object.
     * @return - lun stat.
     */
    public Object getLunStats(String conanicalName) {
        return lunStats.get(conanicalName);
    }

    /**
     * Puts the Lun stat in to the map.
     * 
     * @param conanicalName - name of the Lun.
     * @param stats - results from stats.
     */
    public void putLunStats(String conanicalName, Object stats) {
        this.lunStats.put(conanicalName, stats);
    }

    /**
     * CPU Share Level.
     * 
     * @return - CPU Share Level (High, normal,low).
     */
    public String getCpuShareLevel() {
        return cpuShareLevel;
    }

    /**
     * CPU Share level.
     * 
     * @param cpuShareLevel - CPU Share Level (High, normal,low).
     */
    public void setCpuShareLevel(String cpuShareLevel) {
        this.cpuShareLevel = cpuShareLevel;
    }

    /**
     * Memory Share Level.
     * 
     * @return Memory Share Level (High, normal,low).
     */
    public String getMemoryShareLevel() {
        return memoryShareLevel;
    }

    /**
     * Memory share level.
     * 
     * @param memoryShareLevel - Memory Share Level (High, normal,low).
     */
    public void setMemoryShareLevel(String memoryShareLevel) {
        this.memoryShareLevel = memoryShareLevel;
    }

    /**
     * Memory MB.
     * 
     * @return - Memory MB.
     */
    public long getMemoryMB() {
        return memoryMB;
    }

    /**
     * Memory MB.
     * 
     * @param memoryMB - Memory MB.
     */
    public void setMemoryMB(long memoryMB) {
        this.memoryMB = memoryMB;
    }
        
    /**
     * Gets the list of disk layout path list.
     * 
     * @return - ArrayList of String.
     */
    public ArrayList<String> getVMDKFolderPaths() {
        return vmdkFolderPaths;
    }
        
    /**
     * Returns the disk layout file path.
     * 
     * @param i - index of the array.
     * @return String.
     */
    public String getVMDKFolderPath(int i) {
        return vmdkFolderPaths.get(i);
    }

    /**
     * Sets the list of disk Layout Path list.
     * 
     * @param diskLayoutFilePaths - ArrayList of Strings.
     */
    public void setVMDKFolderPath(ArrayList<String> diskLayoutFilePaths) {
        this.vmdkFolderPaths = diskLayoutFilePaths;
        for(String path: diskLayoutFilePaths){
            if(path.lastIndexOf("/") != -1){
                this.folders.add(path.substring(0, path.lastIndexOf("/")));
            }else{
                this.folders.add(path.substring(0, path.lastIndexOf("]")+1));
            }
        }
    }
        
    /**
     * Adds the disk layout path information to the list.
     * 
     * @param diskLayoutFilePath - disk layout path information.
     */
    public void addVMDKFolderPath(String diskLayoutFilePath) {
        this.vmdkFolderPaths.add(diskLayoutFilePath);
        if(diskLayoutFilePath.lastIndexOf("/") != -1){
            this.folders.add(diskLayoutFilePath.substring(0, diskLayoutFilePath.lastIndexOf("/")));
        }else{
            this.folders.add(diskLayoutFilePath.substring(0, diskLayoutFilePath.lastIndexOf("]")+1));
        }
    }

    /**
     * Returns the log directory.
     * 
     * @return log directory.
     */
    public String getLogDirectory() {
        return logDirectory;
    }

    /**
     * Sets the log directory.
     * 
     * @param logDirectory - log directory.
     */
    public void setLogDirectory(String logDirectory) {
        this.logDirectory = logDirectory;
        this.folders.add(logDirectory);
    }

    /**
     * Returns the snapshot directory.
     * 
     * @return - snapshot directory.
     */
    public String getSnapshotDirectory() {
        return snapshotDirectory;
    }

    /**
     * sets the snapshot directory.
     * 
     * @param snapshotDirectory - snapshot directory.
     */
    public void setSnapshotDirectory(String snapshotDirectory) {
        this.snapshotDirectory = snapshotDirectory;
        this.folders.add(snapshotDirectory);
    }

    /**
     * returns the suspend directory.
     * 
     * @return - suspend directory.
     */
    public String getSuspendDirectory() {
        return suspendDirectory;
    }

    /**
     * sets the suspend directory.
     * 
     * @param suspendDirectory - suspend directory.
     */
    public void setSuspendDirectory(String suspendDirectory) {
        this.suspendDirectory = suspendDirectory;
        this.folders.add(suspendDirectory);
    }

    /**
     * returns the vm path name.
     * 
     * @return - vm path name.
     */
    public String getVmPathName() {
        return vmPathName;
    }

    /**
     * sets the vm path name.
     * 
     * @param vmPathName - vm path name.
     */
    public void setVmPathName(String vmPathName) {
        this.vmPathName = vmPathName;
        this.folders.add(vmPathName);
    }

    /**
     * This obtains all the folders associated with this VM by Datastore. 
     * 
     * @param datastoreName - Datastore name.
     * @return - List of unique folder names by Datastore.
     */
    public List<String> getFolderPathsForDatastore(String datastoreName){
        List<String> folderPaths = new ArrayList<String>();
                
        for(String path:folders){
            if(path.contains(datastoreName)){
                folderPaths.add(path);
            }
        }
                
        return folderPaths;             
    }

    /**
     * Returns the Guest Id. 
     * 
     * @return String.
     */
    public String getGuestId() {
        return guestId;
    }
        
    /**
     * Sets the Guest Id.
     * 
     * @param guestId - String value.
     */
    public void setGuestId(String guestId) {
        this.guestId = guestId;
    }
        
    /**
     * returns all the Virtual Devices.
     * 
     * @return - VirtualDevice array.
     */
    public Object[] getVirtualDevice() {
        return virtualDevice;
    }
        
    /**
     * sets all the virtual Devices.
     * 
     * @param virtualDevice - VirtualDevice array.
     */
    public void setVirtualDevice(Object[] virtualDevice) {
        this.virtualDevice = virtualDevice;
    }

    /**
     * This method returns the DiskLayout objects.
     * 
     * @return VirtualMachineFileLayoutDiskLayout objects.
     */
    public Object[] getFileLayoutDiskLayout() {
        return fileLayoutDiskLayout;
    }

    /**
     * This method sets the disk layout objects.
     * 
     * @param fileLayoutDiskLayout VirtualMachineFileLayoutDiskLayout.
     */
    public void setFileLayoutDiskLayout(Object[] fileLayoutDiskLayout) {
        this.fileLayoutDiskLayout = fileLayoutDiskLayout;
    }

    /**
     * Returns the VM hardware version. 
     * 
     * @return String.
     */
    public String getVmHardwareVersion() {
        return vmHardwareVersion;
    }
        
    /**
     * Sets the VM hardware version.
     * 
     * @param vmHardwareVersion - String value.
     */
    public void setVmHardwareVersion(String vmHardwareVersion) {
        this.vmHardwareVersion = vmHardwareVersion;
    }
        
    /**
     * Gets a map of extra configuration parameters associated with this guest.
     * 
     * @return extraConfig map of extra configuration parameters associated with this guest.
     */
    public HashMap<String, String> getExtraConfig() {
        return this.extraConfig;
    }
    
    /**
     * Set a map of extra configuration parameters associated with this guest.
     * 
     * @param datastores map of extra configuration parameters associated with this guest.
     */
    public void setExtraConfig(HashMap<String, String> extraConfig) {
        this.extraConfig = extraConfig;
    }    

    /**
     * Checks if this VM is a template.
     * 
     * @return true or false.
     */
    public boolean isTemplate() {
        return isTemplate;
    }

    /**
     * sets the whether this VM is a template.
     * 
     * @param isTemplate - True is it is a template.
     */
    public void setTemplate(boolean isTemplate) {
        this.isTemplate = isTemplate;
    }
    
    /**
     * Gets if this virtual machine is the primary if it is fault tolerant enabled.
     * 
     * @return isPrimary true if this virtual machine is the primary when it is fault tolerant enabled
     */
    public boolean isPrimary(){
        return this.isPrimary;
    }
    
    /**
     * Sets if this virtual machine is the primary if it is fault tolerant enabled.
     * 
     * @param isPrimary true if this virtual machine is the primary when it is fault tolerant enabled
     */
    public void setPrimary(boolean isPrimary){
        this.isPrimary = isPrimary;
    }
    
    /**
     * Gets if this virtual machine is the secondary if it is fault tolerant enabled.
     * 
     * @return isSecondary true if this virtual machine is the secondary when it is fault tolerant enabled
     */
    public boolean isSecondary(){
        return this.isSecondary;
    }
    
    /**
     * Sets if this virtual machine is the secondary if it is fault tolerant enabled.
     * 
     * @param isSecondary true if this virtual machine is the secondary when it is fault tolerant enabled
     */
    public void setSecondary(boolean isSecondary){
        this.isSecondary = isSecondary;
    }
    
    /**
     * Prints the contents of the object to a String.
     * @return a string containing the contents of the object.
     */
    public String toString() {
        String NEWLINE = "\n";
        StringBuilder sb = new StringBuilder();

        sb.append(NEWLINE);
        sb.append("name = " + name + NEWLINE);
        sb.append("hostname = " + hostName + NEWLINE);
        sb.append("powerstate = " + powerState + NEWLINE);
        sb.append("gueststate = " + guestState + NEWLINE);
        sb.append("guestfamily = " + guestFamily + NEWLINE);
        sb.append("toolsStatus = " + toolsStatus + NEWLINE);
        sb.append("toolsVersion = " + toolsVersion + NEWLINE);
        sb.append("productname = " + productName + NEWLINE);
        sb.append("vmHwVersion = " + vmHardwareVersion + NEWLINE);
        sb.append("uuid = " + uuid + NEWLINE);
        sb.append("numcpu = " + numberOfCpu + NEWLINE);
        sb.append("memoryMB = " + memoryMB + NEWLINE);
        sb.append("guestid = " + guestId + NEWLINE);
        sb.append("cpushare = " + cpuShareLevel + NEWLINE);
        sb.append("memshare = " + memoryShareLevel + NEWLINE);
        sb.append("istemplate = " + isTemplate + NEWLINE);
        sb.append("cpuReady = " + cpuReady + NEWLINE);
        sb.append("cpuReadyMs = " + cpuReadyMs + NEWLINE);
        sb.append("memoryBalloon = " + memoryBalloon + NEWLINE);
        sb.append("cpushares = " + cpuShares + NEWLINE);
        sb.append("cpulimit = " + cpuLimit + NEWLINE);
        sb.append("cpures = " + cpuReservation + NEWLINE);
        sb.append("memshares = " + cpuShares + NEWLINE);
        sb.append("memlimit = " + cpuLimit + NEWLINE);
        sb.append("memres = " + cpuReservation + NEWLINE);

        sb.append("IPs = " + ipAddressList + NEWLINE);
        sb.append("host = " + host + NEWLINE);
        sb.append("respool = " + resourcePool + NEWLINE);
        sb.append(NEWLINE);

        return(sb.toString());
    }
}
