package com.onaro.commons.exception;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;

/**
 * Exception that happens when server interfaces fail to return result to the caller.
 * Can not have nested exceptions since client code base might not have libraries that contain nested exception.
 *
 * @author Anton Sergeev
 */
public class NestedStackTraceException extends Exception {

    private static final long serialVersionUID = 1L;

    /**
     * Stack trace of the nested exception
     */
    private String nestedStackTrace;

    /**
     * Stack trace of the nested exception
     */
    private String id;


    public NestedStackTraceException() {
        nestedStackTrace = "";
        id = "";
    }

    public NestedStackTraceException(String message) {
        super(message);
        nestedStackTrace = "";
        id = "";
    }

    public NestedStackTraceException(Throwable ex) {
        super();
        buildStackTrace(ex);
    }

    public NestedStackTraceException(String message, Throwable ex) {
        super(message);
        buildStackTrace(ex);
    }

    private void buildStackTrace(Throwable ex) {
        StringWriter stringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stringWriter);
        ex.printStackTrace(printWriter);
        nestedStackTrace = stringWriter.toString();
        printWriter.close();    //this closes both streams

        StringBuilder str = new StringBuilder();
        Throwable cause = ex;
        while (cause != null && !(cause instanceof NestedStackTraceException)) {
            for (StackTraceElement stackTraceElement : cause.getStackTrace()) {
                if (stackTraceElement.getLineNumber() > -1) {
                    str.append(stackTraceElement.getLineNumber());
                    break;
                }
            }
            cause = cause.getCause();
            if (cause != null) {
                str.append('-');
            }
        }
        if (cause != null) {
            str.append(((NestedStackTraceException) cause).getId());
            nestedStackTrace += "\n Caused by:\n" + ((NestedStackTraceException) cause).getNestedStackTrace();
        }
        id = str.toString();
    }

    public String getNestedStackTrace() {
        return nestedStackTrace;
    }

    public String getId() {
        return id;
    }

    @Override
    public String toString() {
        String message = getLocalizedMessage();
        return getClass().getName() +
                (id == null || id.length() == 0 ? "" : " [" + id + "]") +
                (message == null || message.length() == 0 ? "" : "\n" + message);
    }

    @Override
    public void printStackTrace(PrintStream s) {
        s.println(toString());
        s.println(nestedStackTrace);
    }

    @Override
    public void printStackTrace(PrintWriter s) {
        s.println(toString());
        s.println(nestedStackTrace);
    }


}
