package com.onaro.commons.framework.mgmt;

import com.onaro.commons.AuProperties;
import com.onaro.commons.exception.ConfigurationException;
import com.onaro.commons.rest.RestConstants;
import com.onaro.commons.util.HttpURLConnectionUtils;
import org.json.simple.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.HttpsURLConnection;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.HttpURLConnection;
import javax.net.ssl.SSLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Base64;
import java.util.LinkedHashSet;
import java.util.Properties;
import java.util.Set;

import static com.onaro.commons.rest.RestConstants.ACCEPT_ENCODING;
import static com.onaro.commons.rest.RestConstants.ACCEPT_HEADER;
import static com.onaro.commons.rest.RestConstants.CONNECTION;
import static com.onaro.commons.rest.RestConstants.CONTENT_LENGTH;
import static com.onaro.commons.rest.RestConstants.CONTENT_TYPE_HEADER;
import static com.onaro.commons.rest.RestConstants.CONTENT_TYPE_JSON;
import static com.onaro.commons.rest.RestConstants.KEEP_ALIVE;
import static com.onaro.commons.rest.RestConstants.COOKIE;

/**
 * Acquisition client service to execute REST APIs on ocie/au end.
 * Singleton class to initialize SSL connection
 */
public class AcquisitionHttpClientConnectionManager {

    private static final Logger logger = LoggerFactory.getLogger(AcquisitionHttpClientConnectionManager.class);

    private static final String ACQ_HOME = System.getProperty("acq.home");
    private static final String GENERAL_PROPERTIES_FILE = "general.properties";

    private static final int CONNECTION_TIMEOUT_IN_MS = Integer.parseInt(getValueFromGeneralProperties("CONNECTION_TIMEOUT_IN_MS", "1500"));
    private static final int READ_TIMEOUT_IN_MS = Integer.parseInt(getValueFromGeneralProperties("READ_TIMEOUT_IN_MS", "0"));
    
    private static final int DEFAULT_CERT_DOWNLOAD_TIMEOUT_IN_MS = 60000;

    private static final String DEFAULT_USERNAME = "acquisition";

    private static String jSessionId = "";

    private final BasicHttpBasedUrlConnection httpConnection = BasicHttpBasedUrlConnection.getInstance();
    private static final AcquisitionHttpClientConnectionManager acquisitionHttpClientConnectionManager = new AcquisitionHttpClientConnectionManager();

    private AcquisitionHttpClientConnectionManager() {
        try{
            httpConnection.initSSL();
        } catch (ConfigurationException ex) {
            new RuntimeException("AcquisitionHttpClientConnectionManager could not be initialized", ex);
        }
    }

    public static AcquisitionHttpClientConnectionManager getInstance() {
        return acquisitionHttpClientConnectionManager;
    }

    /**
     * Opens an HTTPsURLConnection to specific ip, port etc.
     *
     * @param ip
     * @param port
     * @param resourcePath
     * @param method
     * @param contentType
     * @return
     * @throws IOException
     * @throws com.onaro.commons.exception.ConfigurationException
     */
    private HttpsURLConnection createBasicHTTPSConnection(String ip,
                                                          Integer port,
                                                          String resourcePath,
                                                          String user,
                                                          String password,
                                                          String method,
                                                          String contentType,
                                                          String requestObject) throws IOException, ConfigurationException {

        URL url = null;
        if (ip != null && port != null){
            url = new URL("https", ip, port, resourcePath);
        } else {
            url = new URL(System.getProperty("codebase") + resourcePath);
        }

        HttpsURLConnection con = null;
        con = (HttpsURLConnection)httpConnection.openHttpConnection(url, DEFAULT_CERT_DOWNLOAD_TIMEOUT_IN_MS, false);
        setConnectionProperties(con, user, password, method , contentType, requestObject);
        try {
            con.connect();
        } catch (SSLException ex) {
            //If connection fails, retry after downloading the fresh certificate
	    logger.error(String.format("Certificate error for cluster with host %s %s", url.getHost(), ex.getMessage()));
	    con = (HttpsURLConnection)httpConnection.openHttpConnection(url, DEFAULT_CERT_DOWNLOAD_TIMEOUT_IN_MS, true);
            setConnectionProperties(con, user, password, method , contentType, requestObject);
            con.connect();
        }

        logger.debug(String.format("Https Connection created to host %s and port %d for %s execution ", url.getHost(), url.getPort(), method));
        return con;

    }


    private void setConnectionProperties( HttpsURLConnection con, String user, String password, String method, String contentType, String requestObject) throws IOException{
        String auPropsPasssword = AuProperties.getAuProperties().getProperty(AuProperties.KEY_ACQUISITION_USER_PASSWORD);
        final StringBuilder sb = new StringBuilder(user == null ? DEFAULT_USERNAME : user).append(":").append(password == null ? auPropsPasssword : password);
        String base64UserPass = Base64.getEncoder().encodeToString((sb.toString()).getBytes(StandardCharsets.UTF_8));
        con.addRequestProperty("Authorization", "Basic " + base64UserPass);

        if (!jSessionId.isEmpty()) {
            con.setRequestProperty(COOKIE, jSessionId);
        }

        con.setRequestMethod(method);

        con.setConnectTimeout(CONNECTION_TIMEOUT_IN_MS);
        con.setReadTimeout(READ_TIMEOUT_IN_MS);

        con.setRequestProperty(ACCEPT_HEADER, contentType);
        con.setRequestProperty(CONTENT_TYPE_HEADER, contentType);
        con.setRequestProperty(ACCEPT_ENCODING, contentType);
        con.setRequestProperty(CONNECTION, KEEP_ALIVE);

        con.setUseCaches(false);

        con.setDoInput(true);
        con.setDoOutput(true);

        if ((RestConstants.REQUEST_TYPE_POST.equalsIgnoreCase(method)
                ||  RestConstants.REQUEST_TYPE_PATCH.equalsIgnoreCase(method) )
                && (requestObject == null || requestObject.isBlank())) {
            con.setDoOutput(false);
            con.setRequestProperty(CONTENT_LENGTH, "0");
	    return;
        }

	if (requestObject != null && !requestObject.isBlank()) {
            OutputStream os = con.getOutputStream();
            os.write(requestObject.getBytes(StandardCharsets.UTF_8));
        }

    }

    /**
     * Opens an HTTPsURLConnection to specific ip, port etc for JSONObject request body
     *
     * @param ip
     * @param port
     * @param resourcePath
     * @param method
     *
     * @return
     * @throws IOException
     * @throws com.onaro.commons.exception.ConfigurationException
     */
    private HttpsURLConnection createBasicHTTPSConnection(String ip,
                                                         Integer port,
                                                         String resourcePath,
                                                         String user,
                                                         String password,
                                                         String method,
                                                         JSONObject requestObject ) throws IOException, ConfigurationException {

        return createBasicHTTPSConnection(ip, port, resourcePath, user, password, method, CONTENT_TYPE_JSON, (requestObject == null ? "" : requestObject.toString()));

    }

    /**
     * Executes the REST API and returns the HTTPSURLConnection containing the response inputstream from the execution.
     * Clients are expected to stream this response inputstream and the response status.
     *
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param ip
     * @param port
     * @param resourcePath
     * @param requestObject
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSPOSTAcquisitionAPI(String ip, Integer port, String resourcePath, String user, String password, JSONObject requestObject) throws IOException, ConfigurationException{
        HttpsURLConnection con = createBasicHTTPSConnection(ip, port, resourcePath, user, password, RestConstants.REQUEST_TYPE_POST, requestObject);
        setCurrentJsessionId(con);
        logger.debug(String.format("Executed %s on REST API %s ", RestConstants.REQUEST_TYPE_POST, resourcePath));
        return con;
    }


    public HttpsURLConnection executeHTTPSPOSTAcquisitionAPI(String ip, Integer port, String resourcePath, String user, String password, String requestObject) throws IOException, ConfigurationException{
        HttpsURLConnection con = createBasicHTTPSConnection(ip, port, resourcePath, user, password, RestConstants.REQUEST_TYPE_POST, CONTENT_TYPE_JSON,requestObject);
        setCurrentJsessionId(con);
        logger.debug(String.format("Executed %s on REST API %s ", RestConstants.REQUEST_TYPE_POST, resourcePath));
        return con;
    }

    /**
     * Executes the REST API with default username and password and returns the HTTPSURLConnection containing the response inputstream from the execution.
     * Clients are expected to stream this response inputstream and the response status.
     *
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @param requestObject
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSPOSTAcquisitionAPI(String resourcePath, JSONObject requestObject) throws IOException, ConfigurationException{
        return executeHTTPSPOSTAcquisitionAPI(null, null, resourcePath, null, null, requestObject);
    }

    public HttpsURLConnection executeHTTPSPOSTAcquisitionAPI(String resourcePath, String requestObject) throws IOException, ConfigurationException{
        return executeHTTPSPOSTAcquisitionAPI(null, null, resourcePath, null, null, requestObject);
    }

    /**
     * Executes the REST API with default username and password and returns the HTTPSURLConnection containing the response inputstream from the execution.
     * Clients are expected to stream this response inputstream and the response status.
     *
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @param requestObject
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSPOSTAcquisitionAPI(String ip, Integer port, String resourcePath, JSONObject requestObject) throws IOException, ConfigurationException{
        return executeHTTPSPOSTAcquisitionAPI(ip, port, resourcePath, null, null, requestObject);
    }

    public HttpsURLConnection executeHTTPSPOSTAcquisitionAPI(String ip, Integer port, String resourcePath, String requestObject) throws IOException, ConfigurationException{
        return executeHTTPSPOSTAcquisitionAPI(ip, port, resourcePath, null, null, requestObject);
    }


    public HttpsURLConnection executeHTTPSPOSTAcquisitionAPI(String resourcePath) throws IOException, ConfigurationException{
        return executeHTTPSPOSTAcquisitionAPI(null, null, resourcePath, null, null, "");
    }

    /**
     * Execute GET rest APis. Returns HttpSSurlConnection with the response and response inputstream for parsing
      * @implNote
      *
      * Usage of the returned HTTPSUrlConnection by client
      * InputStream is;
      * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
      *      is = connection.getInputStream();
      *  } else {
      *      is = connection.getErrorStream();
      *  }
     * @param ip
     * @param port
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSGETAcquisitionAPI(String ip, Integer port, String resourcePath, String user, String password) throws IOException, ConfigurationException{
        HttpsURLConnection con = createBasicHTTPSConnection(ip, port, resourcePath, user, password, RestConstants.REQUEST_TYPE_GET, CONTENT_TYPE_JSON, "");
        setCurrentJsessionId(con);
        logger.debug(String.format("Executed %s on REST API %s ", RestConstants.REQUEST_TYPE_GET, resourcePath));
        return con;
    }

    /**
     * Executes GET rest APis with default AU credentials. Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSGETAcquisitionAPI(String resourcePath) throws IOException, ConfigurationException{
        return executeHTTPSGETAcquisitionAPI(null, null, resourcePath, null, null);
    }


    /**
     * Executes GET rest APis with default AU credentials. Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSGETAcquisitionAPI(String ip, Integer port, String resourcePath) throws IOException, ConfigurationException{
        return executeHTTPSGETAcquisitionAPI(ip, port, resourcePath, null, null);
    }

    /**
     *  Execute DELETE rest APis. Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param ip
     * @param port
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSDELETEAcquisitionAPI(String ip, Integer port, String resourcePath, String user, String password) throws IOException, ConfigurationException{
        HttpsURLConnection con = createBasicHTTPSConnection(ip, port, resourcePath, user, password, RestConstants.REQUEST_TYPE_DELETE, CONTENT_TYPE_JSON, "");
        setCurrentJsessionId(con);
        logger.debug(String.format("Executed %s on REST API %s ", RestConstants.REQUEST_TYPE_DELETE, resourcePath));
        return con;
    }

    /**
     *  Execute DELETE rest APis with default creds. Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSDELETEAcquisitionAPI(String resourcePath) throws IOException, ConfigurationException{
        return executeHTTPSDELETEAcquisitionAPI(null, null, resourcePath, null, null);

    }

    /**
     *  Execute DELETE rest APis with default creds. Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSDELETEAcquisitionAPI(String ip, Integer port, String resourcePath) throws IOException, ConfigurationException{
        return executeHTTPSDELETEAcquisitionAPI(ip, port, resourcePath, null, null);

    }

    /**
     *  Execute PATCH rest APis . Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param ip
     * @param port
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSPATCHAcquisitionAPI(String ip, Integer port, String resourcePath, String user, String password, JSONObject requestObject) throws IOException, ConfigurationException{
        HttpsURLConnection con = createBasicHTTPSConnection(ip, port, resourcePath, user, password, RestConstants.REQUEST_TYPE_PATCH, requestObject);
        setCurrentJsessionId(con);
        logger.debug(String.format("Executed %s on REST API %s ", RestConstants.REQUEST_TYPE_PATCH, resourcePath));
        return con;
    }

    public HttpsURLConnection executeHTTPSPATCHAcquisitionAPI(String ip, Integer port, String resourcePath, String user, String password, String requestObject) throws IOException, ConfigurationException{
        HttpsURLConnection con = createBasicHTTPSConnection(ip, port, resourcePath, user, password, RestConstants.REQUEST_TYPE_PATCH, CONTENT_TYPE_JSON,requestObject);
        setCurrentJsessionId(con);
        logger.debug(String.format("Executed %s on REST API %s ", RestConstants.REQUEST_TYPE_PATCH, resourcePath));
        return con;
    }

    /**
     *  Execute PATCH rest APis with default creds. Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSPATCHAcquisitionAPI(String resourcePath, JSONObject requestObject) throws IOException, ConfigurationException{
        return executeHTTPSPATCHAcquisitionAPI(null, null,  resourcePath, null, null, requestObject);
    }

    public HttpsURLConnection executeHTTPSPATCHAcquisitionAPI(String resourcePath, String requestObject) throws IOException, ConfigurationException{
        return executeHTTPSPATCHAcquisitionAPI(null, null, resourcePath, null, null, requestObject);
    }

    /**
     *  Execute PATCH rest APis with default creds. Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSPATCHAcquisitionAPI(String ip, Integer port, String resourcePath, JSONObject requestObject) throws IOException, ConfigurationException{
        return executeHTTPSPATCHAcquisitionAPI(ip, port,  resourcePath, null, null, requestObject);
    }

    /**
     *  Execute PATCH rest APis with default creds. Returns HttpSSurlConnection with the response and response inputstream for parsing
     * @implNote
     *
     * Usage of the returned HTTPSUrlConnection by client
     * InputStream is;
     * if (connection.getResponseCode() < HttpURLConnection.HTTP_BAD_REQUEST) {
     *      is = connection.getInputStream();
     *  } else {
     *      is = connection.getErrorStream();
     *  }
     * @param resourcePath
     * @return
     * @throws IOException
     * @throws ConfigurationException
     */
    public HttpsURLConnection executeHTTPSPATCHAcquisitionAPI(String resourcePath) throws IOException, ConfigurationException{
        return executeHTTPSPATCHAcquisitionAPI(null, null,  resourcePath, null, null, "");
    }

    private void setCurrentJsessionId(HttpsURLConnection urlConnection) throws IOException{
        String currJessionId = HttpURLConnectionUtils.getJsesionIdCookie(urlConnection);

        if (urlConnection.getResponseCode() == HttpURLConnection.HTTP_OK  && currJessionId != null) {
            jSessionId = currJessionId;
        }
    }

    public static String getValueFromGeneralProperties(String key, String defaultValue) {
        String value = defaultValue;
        Properties prop = new Properties();
        String generalPropertiesFilePath = ((ACQ_HOME == null || ACQ_HOME.isEmpty()) ?  "/opt/netapp/essentials/au" : ACQ_HOME)
                .concat(File.separator)
                .concat("conf")
                .concat(File.separator)
                .concat("preferences")
                .concat(File.separator)
                .concat(GENERAL_PROPERTIES_FILE);

        try {
            prop.load(new FileInputStream(generalPropertiesFilePath));
            if (prop.getProperty(key) != null) {
                value = prop.getProperty(key);
            }
        } catch (Exception e) {
            logger.error("Unable to load value of {} from file {}: {} ", key, GENERAL_PROPERTIES_FILE, e);
        }
        return value;
    }
}
