package com.onaro.commons.framework.mgmt;

import com.onaro.commons.exception.ConfigurationException;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Properties;
import java.util.Set;

/**
 * This class should be treated as singleton and only used via CommunicationManager
 * because it sets up and works with the only certificate store file used in acquisition
 */
public class BasicHttpBasedUrlConnection {
    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(BasicHttpBasedUrlConnection.class);

    private static final String SERVER_PROPERTIES_KEY_FOR_TRUSTSTORE_PASSWORD = "truststore.password";
    private static final char[] DEFAULT_TRUSTSTORE_PASSWORD_VALUE = { 'c', 'h', 'a', 'n', 'g', 'e', 'i', 't' };

    private SSLSocketFactory sslSocketFactory = null;
    private String trustStorePath;

    private static BasicHttpBasedUrlConnection instance = new BasicHttpBasedUrlConnection();

    private BasicHttpBasedUrlConnection() {
    }

    public static BasicHttpBasedUrlConnection getInstance() {
        return instance;
    }

    /**
     * Manual addition of PATCH method since PATCH is not supported by Java HttpURLConnection though PATCH is a
     * valid verb.
     * Referring to myself from
     * @See /bard_main/modules/dfm-api-gateway/src/main/java/com/netapp/dfm/api/gateway/services/impl/GatewayConnectionServiceImpl.java
     */
    static {
        try {
            Field modifiersField = Field.class.getDeclaredField("modifiers");
            modifiersField.setAccessible(true);

            Field methodsField = HttpURLConnection.class.getDeclaredField("methods");
            modifiersField.setInt(methodsField, methodsField.getModifiers() & ~Modifier.FINAL);
            methodsField.setAccessible(true);

            String[] existingMethods = (String[]) methodsField.get(null);
            Set<String> methodsSet = new LinkedHashSet<>(Arrays.asList(existingMethods));
            methodsSet.addAll(Arrays.asList("PATCH"));
            Object[] newMethodObjects = methodsSet.toArray();
            int size = newMethodObjects == null ? 0 : newMethodObjects.length;
            String[] newMethods = new String[size];
            if (newMethodObjects != null) {
                for (int i = 0; i < newMethodObjects.length; i++) {
                    newMethods[i] = (String)newMethodObjects[i];
                }
            }
            methodsField.set(null, newMethods);

        } catch (NoSuchFieldException | IllegalAccessException e) {
            //Do not cause the server to fail in case the PATCH method cannot be added. Suppressing the exception
            //since the client will simply get an "invalid method 'PATCH'" but will not fail the server.
            logger.error("Runtime exception occured whie modifying HttpURLConnection to support PATCH method", e);
        }
    }

    /**
     * Initialize SSL communication including downloading of certificates from the server
     *
     * @throws com.onaro.commons.exception.ConfigurationException
     *          When having certificate issues
     */
    public void initSSL() throws ConfigurationException {
        if (logger.isDebugEnabled()) {
            logger.debug("HttpBasedUrlConnection - Initializing SSL");
        }

        // Configure truststore location for everybody
        final String essentialsDir = System.getProperty("onaro.home");
        if (essentialsDir == null) {
            throw new ConfigurationException("onaro.home system property not set!");
        }

        synchronized (this) {
            trustStorePath = essentialsDir + "/jboss/server/onaro/cert/server.truststore";
            System.setProperty("javax.net.ssl.trustStore", trustStorePath);
        }

        // HEADSUP: This violates HTTPS host verification, since we arnt setting up SSL certs right.
        final HostnameVerifier hv = new HostnameVerifier() {
            @Override
            public boolean verify(final String urlHostName, final SSLSession session) {
                return true;
            }
        };

        HttpsURLConnection.setDefaultHostnameVerifier(hv);

        // Prepare SSL socket factory with all available certificates
        loadCertificates();
    }

    private void loadCertificates() throws ConfigurationException {
        final File trustStore = new File(trustStorePath);
        if (!trustStore.exists() || !trustStore.isFile()) {
            final String defaultKeyStorePath = System.getProperties().getProperty("java.home") + File.separator + "lib" + File.separator + "security" + File.separator + "cacerts";

            if (logger.isDebugEnabled()) {
                logger.debug("HttpBasedUrlConnection - Copying default truststore from " + defaultKeyStorePath);
            }

            try {
                //noinspection ResultOfMethodCallIgnored
                trustStore.getParentFile().mkdirs();

                // Simply copy the file
                final InputStream is = new BufferedInputStream(new FileInputStream(new File(defaultKeyStorePath)));
                final OutputStream os = new BufferedOutputStream(new FileOutputStream(trustStore));
                final byte[] buffer = new byte[100000];

                try {
                    while (true) {
                        final int l = is.read(buffer);
                        if (l == -1) {
                            break;
                        }

                        os.write(buffer, 0, l);
                    }
                } finally {
                    //noinspection EmptyCatchBlock
                    try {
                        os.close();
                    } catch (IOException e) {
                    }
                    //noinspection EmptyCatchBlock
                    try {
                        is.close();
                    } catch (IOException e) {
                    }
                }

                // Now need to change the permission of the file to 600, only
                // for NIX devices for now
                //
                String systemOS = System.getProperty("os.name");
                if ( !systemOS.contains("Window") ) {
                    Set<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
                    // Add owners permission
                    perms.add(PosixFilePermission.OWNER_READ);
                    perms.add(PosixFilePermission.OWNER_WRITE);

                    Files.setPosixFilePermissions(Paths.get(trustStorePath), perms);
                }

            } catch (IOException e) {
                throw new ConfigurationException("Failed to copy default key store", e);
            }
        }

        final CertificateDownloader downloader = new CertificateDownloader(trustStorePath);
        synchronized (this) {
            sslSocketFactory = downloader.loadCertificates();
        }

        if (logger.isDebugEnabled()) {
            logger.debug("HttpBasedUrlConnection - Loaded certificates!");
        }
    }

    /**
     * Downloads the SSL certificate from the given URL and updates the local TrustStore.
     * This will also modify the default SSL socket factory to reflect the newly downloaded certificate
     *
     * @param url     URL of server to download certificates from
     * @param timeout In millis
     * @throws ConfigurationException When failed to download
     */
    public synchronized void downloadCertificate(final URL url, final int timeout, final boolean override) throws ConfigurationException {
        downloadCertificate(url.getHost(), url.getPort(), timeout, override);
    }

    public synchronized void downloadCertificate(final String host, final int port, final int timeout, final boolean override) throws ConfigurationException {
        if (logger.isDebugEnabled()) {
            logger.debug("HttpBasedUrlConnection - Downloading certificate from host - " + host + ":" + port);
        }

        final CertificateDownloader downloader = new CertificateDownloader(trustStorePath);
        downloader.downloadIfNeeded(host, port, timeout, override);

        sslSocketFactory = downloader.loadCertificates();

        // Modify the default SSL socket factory to reflect the downloaded certificate!
        HttpsURLConnection.setDefaultSSLSocketFactory(sslSocketFactory);
    }

    /**
     * Returnes an initialized {@link URLConnection} setup to work.
     * <p/>
     * This includes setting up the correct truststore (for HTTPS certificates) if neccessary.
     *
     * @param url     URL to open
     * @param timeout in millis
     * @return An initialized connection to this URL
     * @throws ConfigurationException
     *
     */
    public HttpURLConnection openHttpConnection(final URL url, final int timeout, boolean checkCertificate) throws IOException, ConfigurationException {
        if (logger.isDebugEnabled()) {
            logger.debug("HttpBasedUrlConnection - setting up http connection to url : " + url.toString());
        }

        final URLConnection urlConnection = url.openConnection();
        if (!(urlConnection instanceof HttpURLConnection)) {
            throw new ConfigurationException("Not http based url : " + url.toString());
        }

        if (urlConnection instanceof HttpsURLConnection) {
            if (logger.isDebugEnabled()) {
                logger.debug("HttpBasedUrlConnection - enabling SSL for url : " + url.toString());
            }

            // Make sure we download the certificate from the server if none is installed (and update the SSL factory)
            if ((null == sslSocketFactory) || checkCertificate) {
                downloadCertificate(url, timeout, false);
            }

            // enable SSL on this connection
            synchronized (this) {
                ((HttpsURLConnection) urlConnection).setSSLSocketFactory(sslSocketFactory);
            }
        }

        return (HttpURLConnection) urlConnection;
    }

    public synchronized SSLSocketFactory getSslSocketFactory() {
        return sslSocketFactory;
    }

    public static String getServerTrustStorePassword() {
        Properties prop = new Properties();
        String serverPropertiesFilePath = System.getProperty("onaro.home", "/opt/netapp/essentials") + "/conf/server.properties";
        try {
            prop.load(new FileInputStream(serverPropertiesFilePath));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        String password = prop.getProperty(SERVER_PROPERTIES_KEY_FOR_TRUSTSTORE_PASSWORD, String.copyValueOf(DEFAULT_TRUSTSTORE_PASSWORD_VALUE));
        return password;
    }
}
