package com.onaro.commons.framework.mgmt;


import com.onaro.commons.exception.ConfigurationException;
import com.onaro.commons.rest.RestConstants;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;

import javax.net.ssl.HttpsURLConnection;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * Base class for all AU client managers. This is the basic type of all the client HTTP manager types.
 */
public abstract class HTTPClientManager {

    private final static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(HTTPClientManager.class);

    public JSONObject parse(HttpsURLConnection connResponse) throws IOException, ConfigurationException {
        int responseCode = connResponse.getResponseCode();
        String responseMessage = connResponse.getResponseMessage();
        String contentType = connResponse.getHeaderField(RestConstants.CONTENT_TYPE_HEADER);
        logger.debug(String.format("REST %s %s Response status : %d %s", connResponse.getURL().getProtocol(),  connResponse.getURL().toString(), responseCode, responseMessage ));
        JSONObject jsonObject = new JSONObject();
        InputStream is = null;
        if (responseCode >= HttpURLConnection.HTTP_BAD_REQUEST) {
            is = connResponse.getErrorStream();
        } else {
            is = connResponse.getInputStream();
        }

        try {
            if (contentType != null && contentType.equalsIgnoreCase(RestConstants.CONTENT_TYPE_JSON) && is != null) {
                jsonObject = (JSONObject)new JSONParser().parse(new InputStreamReader(is, StandardCharsets.UTF_8));
                if (jsonObject != null) {
                    //Only throw an exception with the errorMessage if the response code is an error code
                    if (responseCode >= HttpURLConnection.HTTP_BAD_REQUEST) {
                        //Default Spring JSON error response has a "message" field with error message
                        String errorMessage = (String)jsonObject.get("message");
                        throw new ConfigurationException(String.format("Error Response %d %s %s ", responseCode, responseMessage, errorMessage));
                    }
                }
            }
        } catch (ParseException ex) {
            logger.error(String.format("Parse exception while reading REST response %s", ex.getMessage()));
            throw new ConfigurationException(ex.getMessage());
        } finally {
            if (is != null) {
                is.close();
            }
        }
        return jsonObject;
    }

    public String buildResourcePath(String baseResourcePath, String pathParam, Map<String, String> requestParamMap) {
        return buildResourcePath(baseResourcePath, pathParam, null, requestParamMap);
    }

    public String buildResourcePath(String baseResourcePath, String pathParam, String extendedResourcePath, Map<String, String> requestParamMap) {
        StringBuilder sb = new StringBuilder(baseResourcePath);

        if (pathParam != null) {
            sb.append("/").append(pathParam);
        }
        if (extendedResourcePath != null) {
            if (extendedResourcePath.contains("/")) {
                sb.append(extendedResourcePath);
            } else {
                sb.append("/").append(extendedResourcePath);
            }

        }
        if (requestParamMap != null && !requestParamMap.isEmpty()) {
            sb.append("?");
            int i = 1;
            int requestParamMapSize = requestParamMap.size();
            for (Map.Entry<String, String> mapElement : requestParamMap.entrySet()) {
                String key = mapElement.getKey();
                String value = mapElement.getValue();
                if (key == null) {
                    continue;
                }
                if (value == null) {
                    value = "";
                }
                String encodedRequestparam = URLEncoder.encode(value, StandardCharsets.UTF_8);
                //sb.append(key + "=" + "\"" + value + "\"");
                sb.append(key + "=" + encodedRequestparam);
                i++;
                if (i <= requestParamMapSize) {
                    sb.append("&");
                }
            }
        }
        return sb.toString();
    }

    public void logHTTPErrorStatus(int statusCode) throws ConfigurationException {
        StringBuilder builder = new StringBuilder();
        builder.append("Server returned HTTP status ").append(statusCode).append(".");
        logger.error(builder.toString());
        throw new ConfigurationException(builder.toString());
    }


    public void logHTTPErrorStatusWithMessage(final InputStream stream, int statusCode) throws ConfigurationException {
        StringBuilder builder = new StringBuilder();
        String errorMessage = getErrorStringFromStream(stream);
        builder.append("Server returned HTTP status ").append(statusCode).append(".");
        builder.append("HTTP error message :").append(errorMessage);
        logger.error(builder.toString());
        throw new ConfigurationException(builder.toString());
    }

    private String getErrorStringFromStream(final InputStream stream) {
        if (stream == null) {
            return null;
        }
        BufferedReader streamReader = null;
        StringBuilder stringBuilder = new StringBuilder();
        String line;
        try {
            streamReader = new BufferedReader(new InputStreamReader(stream));
            while ((line = streamReader.readLine()) != null) {
                stringBuilder.append(line).append('\n');
            }
        } catch (Exception ignore) {
        } finally {
            if (streamReader != null) {
                try {
                    streamReader.close();
                } catch (Exception ignore) {
                    logger.error("Error closing the stream: " + ignore.getMessage());
                }
            }
        }
        return stringBuilder.toString();
    }

}

