/*
 * Copyright (c) 2010 NetApp
 * All rights reserved
 */

package com.onaro.commons.table;

import java.util.prefs.Preferences;

/**
 * Abstracts configuration of mySql server host and the database names (sanscreen, performance, etc.).
 * Extracts the information from the System Properties (Registry)
 * <p/>
 * Date: Apr 13, 2007
 * Time: 3:31:24 PM
 *
 * @author yarom.gabay
 */
public class DbConfig {

    private static final String DB_SYSTEM_NODE = "/com/onaro/sanscreen/server/db";


    /**
     * Defaults Values
     */
    enum Schema {
        SANSCREEN,
        ACQUISITION,
        MANAGEMENT,
    }
    public static final String MYSQL_HOST_DEFAULT_VALUE = "localhost";

    /**
     * Key Names
     */
    public static final String NAME_KEY = ".name";
    private static final String MYSQL_HOST_KEY = "mysql.host";

    public static String getSanscreenDbName() {
        String schema = Schema.SANSCREEN.name().toLowerCase();
        return getRegistryNode().get(schema + NAME_KEY, schema);
    }

    public static String getAcquisitionDbName() {
        String schema = Schema.ACQUISITION.name().toLowerCase();
        return getRegistryNode().get(schema + NAME_KEY, schema);
    }

    public static String getManagementDbName() {
        String schema = Schema.MANAGEMENT.name().toLowerCase();
        return getRegistryNode().get(schema + NAME_KEY, schema);
    }

    public static String getMySqlHost() {
        return getRegistryNode().get(MYSQL_HOST_KEY, MYSQL_HOST_DEFAULT_VALUE);
    }

    /**
     * Modify the host where the database resides.
     *
     * @param mySqlHost
     */
    public static void setMySqlHost(String mySqlHost) {
        getRegistryNode().put(MYSQL_HOST_KEY, mySqlHost);
    }

    /**
     * Set prefix to all databases. {@code null} means no prefix.
     *
     * @param prefix Could be {@code null}
     */
    public static void setDatabasePrefix(String prefix) {
        prefix = (prefix == null || prefix.length() == 0) ? "" : prefix + "_";

        for (Schema schema : Schema.values()) {
            String schemaName = prefix + schema.name().toLowerCase();
            String schemaKey = schemaName + NAME_KEY;
            getRegistryNode().put(schemaKey, schemaName);
        }
    }

    private static Preferences getRegistryNode() {
        return Preferences.systemRoot().node(DB_SYSTEM_NODE);
    }
}
