package com.onaro.commons.util;

import java.io.Serializable;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.text.MessageFormat;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;


public class MessageManager {
    private static final MessageManager INSTANCE = new MessageManager();
    /**
     * The logger for this class.
     */
    static final org.slf4j.Logger logger =org.slf4j.LoggerFactory.getLogger(MessageManager.class);

    public static MessageManager getInstance() {
        return INSTANCE;
    }

    protected MessageManager() {
        // do nothing
    }

    public static <T> T getMessages(Class<? extends T> messagesClass) {
        Locale locale = Locale.getDefault();

        return getMessages(messagesClass, locale);
    }

    public static <T> T getMessages(Class<? extends T> messagesClass, Locale locale) {
        MessageManager messageManager = getInstance();

        return messageManager.getMessagesImpl(messagesClass, locale);
    }

    protected ResourceBundle getResourceBundle(Method method, Locale locale) {
        Class<?> clazz = method.getDeclaringClass();

        String rbBaseName = clazz.getName();

        ClassLoader classLoader = clazz.getClassLoader();

        return ResourceBundle.getBundle(rbBaseName, locale, classLoader);
    }

    private final <T> T getMessagesImpl(Class<? extends T> messagesClass, Locale locale) {
        if (messagesClass == null) throw new IllegalArgumentException();

        ClassLoader classLoader = messagesClass.getClassLoader();

        InvocationHandler invocationHandler = newInvocationHandler(locale);

        Class<?>[] interfaces = { messagesClass };

        Object proxyObj = Proxy.newProxyInstance(classLoader, interfaces, invocationHandler);

        T proxy = messagesClass.cast(proxyObj);

        return proxy;
    }

    protected InvocationHandler newInvocationHandler(Locale locale) {
        return new MsgInvocationHandler(locale);
    }

    protected String getMessageKey(Method method) {
        // Try to get the Message annotation off the method
        Message message = method.getAnnotation(Message.class);
        if (message != null) {
            return getMessageKey(message);
        }
        
        // If no annotation is present, key off the method name
        String messageKey = method.getName();
        
        // If the method name starts with "get", trim off the "get" and use the rest of
        // the method name as the key. If it doesn't start with "get", the entire method name will be used
        if (messageKey.startsWith("get")) { //$NON-NLS-1$
            messageKey = messageKey.substring(3);
        }
        
        return messageKey;
    }

    protected final String getMessageKey(Message message) {
        if (message == null) throw new IllegalArgumentException("message"); //$NON-NLS-1$

        String msgKey = message.value();
        if (msgKey != null && msgKey.length() > 0) return msgKey;

        msgKey = message.key();

        if (msgKey == null || msgKey.trim().length() == 0) {
            throw new IllegalArgumentException("Message key not specified in annotation"); //$NON-NLS-1$
        }

        return msgKey;
    }
    
    protected boolean isFormat(Method method) {
        Message message = method.getAnnotation(Message.class);
        
        // If the method does not have a Message annotation, the message isn't a format
        if (message == null) {
            return false;
        }

        return isFormat(message);
    }
    
    protected final boolean isFormat(Message message) {
        if (message == null) throw new IllegalArgumentException("message"); //$NON-NLS-1$
        
        return message.format();   
    }

    protected String getMessage(Method method, Object[] args, Locale locale) {
        String msgKey = getMessageKey(method);
        boolean isFormat = isFormat(method);

        ResourceBundle rb = getResourceBundle(method, locale);

        String msgPattern = rb.getString(msgKey);

        String msg = null;

        if (!isFormat && args != null && args.length > 0) {
            msg = MessageFormat.format(msgPattern, args);
        }
        else {
            msg = msgPattern;
        }

        return msg;
    }

    protected static class MsgInvocationHandler implements InvocationHandler, Serializable {
        private static final long serialVersionUID = 1L;

        protected final Locale locale;

        public MsgInvocationHandler(Locale locale) {
            if (locale == null) throw new IllegalArgumentException();

            this.locale = locale;
        }

        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String msg = getInstance().getMessage(method, args, locale);

            return msg;
        }

    }
    
    /*
     * Unit Testing Utility Methods 
     *
     */
    public void checkMessageAnnotations(Class<?> messagesClass) {
        for (Method method : messagesClass.getMethods()) {
            getMessageKey(method);
        }
    }
    
    public void checkMissingMessages(Class<?> messagesClass) {
        checkMissingMessages(messagesClass, Locale.getDefault());
    }
    
    public void checkMissingMessages(Class<?> messagesClass, Locale locale) {
        for (Method method : messagesClass.getMethods()) {
            String msgKey = getMessageKey(method);

            ResourceBundle rb = getResourceBundle(method, locale);

            rb.getString(msgKey);
        }
    }
    
    public void checkUnusedMessages(Class<?> messagesClass) {
        checkUnusedMessages(messagesClass, Locale.getDefault());
    }
    
    public void checkUnusedMessages(Class<?> messagesClass, Locale locale) {
        String rbBaseName = messagesClass.getName();
        ClassLoader classLoader = messagesClass.getClassLoader();
        
        ResourceBundle rb = ResourceBundle.getBundle(rbBaseName, locale, classLoader);
        
        Enumeration<String> keyEnumeration = rb.getKeys();
        Set<String> unusedKeys = new HashSet<String>();
        while (keyEnumeration.hasMoreElements()) {
            String key = keyEnumeration.nextElement();
            unusedKeys.add(key);
        }
        
        for (Method method : messagesClass.getMethods()) {
            String msgKey = getMessageKey(method);
            unusedKeys.remove(msgKey);
        }
        
        if (!unusedKeys.isEmpty()) {
            String msg = "Resource bundle for message class {0} has the following unused keys: {1}"; //$NON-NLS-1$
            throw new IllegalArgumentException(MessageFormat.format(msg, messagesClass.getName(), unusedKeys.toString()));
        }
    }
    
    public void checkMalformedParameters(Class<?> messagesClass) {
        checkMalformedParameters(messagesClass, Locale.getDefault());
    }
    
    public void checkMalformedParameters(Class<?> messagesClass, Locale locale) {
        for (Method method : messagesClass.getMethods()) {
            String msgKey = getMessageKey(method);
            boolean isFormat = isFormat(method);

            ResourceBundle rb = getResourceBundle(method, locale);

            String msgPattern = rb.getString(msgKey);

            int numMethodParams = method.getParameterTypes().length;

            Pattern pattern = Pattern.compile("\\{(\\d+).*?\\}"); //$NON-NLS-1$

            Matcher matcher = pattern.matcher(msgPattern);

            int groupCount = 0;
            
            int maxPattern = 0;

            while (matcher.find()) {
                groupCount++;
                
                String group = matcher.group(1);
                Integer value = Integer.valueOf(group);
                if (value.intValue() > maxPattern) {
                    maxPattern = value.intValue();
                }
            }
            
            if (isFormat) {
                if (numMethodParams != 0) {
                    String msg = "Message method specifies parameters but method is declared as a format {0}"; //$NON-NLS-1$
                    throw new IllegalArgumentException(MessageFormat.format(msg, method.getName()));
                }
            } else if (numMethodParams != 0 && groupCount == 0) {
                String msg = "Message method does not specify parameters for method {0} pattern {1}"; //$NON-NLS-1$
                throw new IllegalArgumentException(MessageFormat.format(msg, method.getName(), msgPattern));
            } else if (groupCount == 0 && numMethodParams != 0) {
                String msg = "Message method specifies parameters that are not used by pattern for method {0} pattern {1}"; //$NON-NLS-1$
                throw new IllegalArgumentException(MessageFormat.format(msg, method.getName(), msgPattern));
            } else if (groupCount > 0 && numMethodParams != maxPattern + 1) {
                String msg = "Incorrect number of method arguments, {0} != {1} for method {2} pattern {3}"; //$NON-NLS-1$
                throw new IllegalArgumentException(MessageFormat.format(msg, numMethodParams, (maxPattern + 1), method.getName(), msgPattern));
            }
        }
    }
}
