package com.onaro.commons.util;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.IdentityHashMap;
import java.util.Map;

public final class ObjectUtils {
	private static final int SEED_HASH_CODE = (2 << 7) - 1;

	private static final int NULL_HASH_CODE = (2 << 3) - 1;

	private ObjectUtils() {
		// do nothing
	}

	/**
	 * @see org.apache.commons.lang.ObjectUtils#equals(Object, Object)
	 */
	@Deprecated
	public static boolean isEqual(Object o1, Object o2) {
		if (o1 == null) return o2 == null;

		return o1.equals(o2);
	}

	public static int hashCode(Iterable<?> iter) {
		int hashCode = SEED_HASH_CODE;

		for (Object obj : iter) {
			int subHashCode = NULL_HASH_CODE;

			if (obj != null) {
				subHashCode = obj.hashCode();
			}

			hashCode = ((2 << 5) - 1) * hashCode + subHashCode;
		}

		return hashCode;
	}

	public static int hashCode(Object... args) {
		return hashCode(Arrays.asList(args));
	}
	
	/**
	 * Perform a safe call of valueOf on an enum. If the given name is not valid, the IllegalArgumentException
	 * will be caught and the defaultValue will be returned.
	 * 
	 * @param name the name of the enum constant to retrieve
	 * @param defaultValue the default value if the name is not valid (should not be null)
	 * @return the enum constant, default value if not valid
	 */
	public static <T extends Enum<T>> T safeEnumValueOf(String name, T defaultValue) {
		T result = defaultValue;
		Class<T> clazz = defaultValue.getDeclaringClass();
		try {
			result = T.valueOf(clazz, name);
		} catch (IllegalArgumentException e) { 
		    // Try again with the upper case version of the name
		    if (name != null) {
    		    try {
    		        result = T.valueOf(clazz, name.toUpperCase());
    		    } catch (IllegalArgumentException e2) { 
    		    }
		    }
		}
		return result;
	}

	public static <T> T deepClone(T srcObj) throws CloneNotSupportedException {
		if (srcObj == null) return null;

		IdentityHashMap<Object,Object> cachedObjects = new IdentityHashMap<Object,Object>();
		return deepClone(srcObj, cachedObjects);
	}
	
	@SuppressWarnings("unchecked") //$NON-NLS-1$
	public static <T> T shallowClone(T srcObj) throws CloneNotSupportedException {
        if (srcObj == null) {
            return null;
        }
		try {
			Class<? extends Object> srcObjClass = srcObj.getClass();
			Method cloneMethod = srcObjClass.getMethod("clone"); //$NON-NLS-1$
			if (cloneMethod.getDeclaringClass().equals(Object.class)) {
				// The source object is not Cloneable.
				throw new CloneNotSupportedException();
			}

			T clonedObj = (T) cloneMethod.invoke(srcObj);
			return clonedObj;
		}
		catch (Exception e) {
			CloneNotSupportedException cnse = new CloneNotSupportedException();
			cnse.initCause(e);
			throw cnse;
		}
	}
	
	private static <T,C extends Collection<T>> C deepCloneCollection(C collection, IdentityHashMap<Object,Object> cachedObjects) throws CloneNotSupportedException {
		if (collection == null) {
			return null;
		}

		try {
			C newCollection = shallowClone(collection);
			newCollection.clear();

			for (T element : collection) {
				if (element == null) {
					newCollection.add(null);
				}
				else {
					newCollection.add(deepClone(element, cachedObjects));
				}
			}
			return newCollection;
		}
		catch (Exception e) {
			CloneNotSupportedException ex = new CloneNotSupportedException();
			ex.initCause(e);
			throw ex;
		}
	}
	
	private static <K,V,M extends Map<K,V>> M deepCloneMap(M map, IdentityHashMap<Object,Object> cachedObjects) throws CloneNotSupportedException {
		if (map == null) {
			return null;
		}

		try {
			M newMap = shallowClone(map);
			newMap.clear();

			for (Map.Entry<K,V> entry : map.entrySet()) {
				K clonedKey = deepClone(entry.getKey(), cachedObjects);
				V clonedValue = deepClone(entry.getValue(), cachedObjects);
				newMap.put(clonedKey, clonedValue);
			}
			return newMap;
		}
		catch (Exception e) {
			CloneNotSupportedException ex = new CloneNotSupportedException();
			ex.initCause(e);
			throw ex;
		}
	}
	
	@SuppressWarnings("unchecked") //$NON-NLS-1$
	private static <T> T[] deepCloneArray(T[] srcArray, IdentityHashMap<Object,Object> cachedObjects) throws CloneNotSupportedException {
		if (srcArray == null) {
			return null;
		}

		try {
			T[] newArray = (T[]) Array.newInstance(srcArray.getClass().getComponentType(), srcArray.length);

			for (int i = 0; i < srcArray.length; i++) {
				T element = srcArray[i];
				if (element != null) {
					newArray[i] = deepClone(element, cachedObjects);
				}
			}
			return newArray;
		}
		catch (Exception e) {
			CloneNotSupportedException ex = new CloneNotSupportedException();
			ex.initCause(e);
			throw ex;
		}
	}

	@SuppressWarnings("unchecked") //$NON-NLS-1$
	private static <T> T deepClone(T srcObj, IdentityHashMap<Object,Object> cachedObjects) throws CloneNotSupportedException {
		
		if (srcObj == null) return null;
		
		if (cachedObjects.containsKey(srcObj)) {
			return (T) cachedObjects.get(srcObj);
		}
		
		if (isImmutable(srcObj.getClass())) return srcObj;
		
		T clonedObj = null;
		
		if (srcObj instanceof Collection) {
			clonedObj = (T) deepCloneCollection((Collection) srcObj, cachedObjects);
			cachedObjects.put(srcObj, clonedObj);
			return clonedObj;
		}
		
		if (srcObj instanceof Map) {
			clonedObj = (T) deepCloneMap((Map) srcObj, cachedObjects);
			cachedObjects.put(srcObj, clonedObj);
			return clonedObj;
		}
		
		Class<? extends Object> srcObjClass = srcObj.getClass();
		
		if (srcObjClass.isArray()) {
			clonedObj = (T) deepCloneArray((Object[])srcObj, cachedObjects);
			cachedObjects.put(srcObj, clonedObj);
			return clonedObj;
		}
		
		try {
			clonedObj = shallowClone(srcObj);
			cachedObjects.put(srcObj, clonedObj);

			Class<? extends Object> cloneObjClass = clonedObj.getClass();

			for (Class<?> c = cloneObjClass; c != null; c = c.getSuperclass()) {
				Field[] fields = c.getDeclaredFields();

				for (Field field : fields) {
					
					if (Modifier.isStatic(field.getModifiers())) continue;
					Class<?> fieldType = field.getType();
					
					if (isImmutable(fieldType)) continue;

					if (!field.isAccessible()) {
						field.setAccessible(true);
					}

					Object clonedFieldValue = field.get(clonedObj);

					if (clonedFieldValue == null) continue;
					
					try
					{
						Object srcFieldValue = field.get(srcObj);
						
						// they already made a deep copy
						if (srcFieldValue != clonedFieldValue) continue;
					}
					catch (Exception e)
					{
						// ignore
					}
					
					try {
						clonedFieldValue = deepClone(clonedFieldValue, cachedObjects);

						field.set(clonedObj, clonedFieldValue);
					}
					catch (CloneNotSupportedException cnse) {
						// oh well, we tried
					}
				}
			}
			return clonedObj;
		}

		catch (Exception e) {
			CloneNotSupportedException cnse = new CloneNotSupportedException();
			cnse.initCause(e);
			throw cnse;
		}
	}

	public static boolean isImmutable(Class<?> clazz) {
		return clazz.isPrimitive() || clazz.isEnum() || String.class.equals(clazz) || Number.class.isAssignableFrom(clazz)
				|| Date.class.equals(clazz);
	}
	
	public static void main(String[] args) {
/*		Point[] points = new Point[2];
		points[0] = new Point(1,2);
		
		try {
			Point[] cloned = deepClone(points);
			cloned[0].x = 3;
			
			System.out.println(System.identityHashCode(points[0]) + " - " + points[0]);
			System.out.println(System.identityHashCode(cloned[0]) + " - " + cloned[0]);
		}
		catch (CloneNotSupportedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}*/
	}
}
