package com.onaro.commons.util;

import java.lang.ref.SoftReference;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * A map implementation with soft values. An entry in a Map will automatically be removed when its value is no
 * longer in ordinary use.
 */
public final class SoftHashMap<K,V> extends AbstractMap<K,V>
{
	private static final int DEFAULT_INITIAL_CAPACITY = 128;
	
	private final HashMap<K,SoftReference<V>> map;
	
	public SoftHashMap()
	{
		this(DEFAULT_INITIAL_CAPACITY);
	}
	
	public SoftHashMap(int initialCapacity)
	{
		map = new LinkedHashMap<K,SoftReference<V>>(initialCapacity, 2.0f, true)
		{
			private static final long serialVersionUID = 1L;

			@Override
			protected boolean removeEldestEntry(java.util.Map.Entry<K,SoftReference<V>> eldest)
			{
				SoftReference<V> ref = eldest.getValue();
				
				if (ref == null) return false;
				
				return ref.get() == null;
			}
			
			
		};
	}
	
	public SoftHashMap(Map<? extends K,? extends V> t)
	{
		this(t.size());
		
		putAll(t);
	}
	
	@Override
	public Set<java.util.Map.Entry<K,V>> entrySet()
	{
		throw new UnsupportedOperationException();
	}

	@Override
	public void clear()
	{
		map.clear();
	}

	@Override
	public boolean containsKey(Object key)
	{
		if (!map.containsKey(key)) return false;
		
		SoftReference<V> ref = map.get(key);
		
		if (ref == null) return true;
		
		V value = ref.get();
		
		if (value == null)
		{
			remove(key);
		}
		
		return value != null;
	}

	@Override
	public V get(Object key)
	{
		SoftReference<V> ref = map.get(key);
		
		if (ref == null) return null;
		
		V value = ref.get();
		
		if (value == null)
		{
			map.remove(key);
		}
		
		return value;
	}

	@Override
	public boolean isEmpty()
	{
		return map.isEmpty();
	}

	@Override
	public Set<K> keySet()
	{
		return map.keySet();
	}

	@Override
	public V put(K key, V value)
	{
		SoftReference<V> newRef = (value != null) ? new SoftReference<V>(value) : null;
		
		SoftReference<V> oldRef = map.put(key, newRef);
		
		if (oldRef == null) return null;
		
		V oldValue = oldRef.get();
		
		return oldValue;
	}

	@Override
	public V remove(Object key)
	{
		SoftReference<V> oldRef = map.remove(key);
		
		if (oldRef == null) return null;
		
		V oldValue = oldRef.get();
		
		return oldValue;
	}

	@Override
	public int size()
	{
		return map.size();
	}

	@Override
	public boolean containsValue(Object value)
	{
		if (value == null)
		{
			return map.containsValue(null);
		}
		
		for (SoftReference<V> ref : map.values())
		{
			if (ref == null) continue;
			
			V refValue = ref.get();
			
			if (refValue == null) continue;
			
			if (value.equals(refValue)) return true;
		}
		
		return false;
	}
	
	
}
