package com.netapp.oci.profiler;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.*;

/**
 * CSV file implementation for the <code>ProfilerDataStore</code>.
 *
 * @author Shan Ponnusamy
 */
public class CsvFileDataStore implements ProfilerDataStore {

    /**
     * CSV file headers.
     */
    private Map<String, Integer> headerIndexMap;

    private int noOfHeaders;

    /**
     * Writer to the output file.
     */
    private BufferedWriter writer;

    public CsvFileDataStore(File outputFile, String[] headers) throws IOException {

        // add "Time" column to the list of column headers.  This should be the first column.
        List<String> headersList = new ArrayList<String>(headers.length + 1);
        headersList.add(ProfilerService.TIME);
        headersList.addAll(Arrays.asList(headers));
               
        // initialize the headerIndexMap;
        headerIndexMap = new HashMap<String, Integer>();
        int index = 0;
        for (String header: headersList) {
            headerIndexMap.put(header, index++);
        }
        noOfHeaders = index;
        
        writer = new BufferedWriter(new FileWriter(outputFile));
        writeRow(headersList.toArray());   // write column names to the output file.
    }

    @Override
    public void store(Map<String, Object> events) throws IOException {
        if (!events.containsKey(ProfilerService.TIME)) {
            events.put(ProfilerService.TIME, System.currentTimeMillis());
        }
        Object[] orderedValues = orderByColumns(events);
        writeRow(orderedValues);
    }

    private synchronized void writeRow(Object[] orderedValues) throws IOException {
        writer.write(toCsvString(orderedValues));
        writer.newLine();
        writer.flush();
    }

    private String toCsvString(Object[] values) {
        assert values.length > 0;
        
        StringBuilder str =  new StringBuilder();
        for (Object value: values) {
            str.append(value == null ? "": value);
            str.append(',');
        }
        str.deleteCharAt(str.length() - 1);
        return str.toString();
    }

    // order the values according to the column header order.
    private Object[] orderByColumns(Map<String, Object> data) {
        Object[] orderedValues = new Object[noOfHeaders];
        for (Map.Entry<String, Object> itemValuePair: data.entrySet()) {
            int index = headerIndexMap.get(itemValuePair.getKey());
            orderedValues[index] = itemValuePair.getValue();
        }
        return orderedValues;
    }

    @Override
    public void store(String event, Object value) throws IOException {
        Object[] orderedValues = new Object[noOfHeaders];
        orderedValues[0] = System.currentTimeMillis();  // set the value for "Time" column.
        int index = headerIndexMap.get(event);
        orderedValues[index] = value;        
        writeRow(orderedValues);
    }

    @Override
    public void close() throws IOException {
        writer.close();
    }
}
