package com.netapp.oci.profiler;

import com.sun.management.OperatingSystemMXBean;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.util.HashMap;
import java.util.Map;

/**
 * CPU usage and memory usage profiler.
 *
 * @author Shan Ponnusamy
 */
public class CpuMemoryProfiler implements Profiler {

    /**
     * Profiled item constants.
     */
    public static final String CPU_USAGE = "Cpu Usage";
    public static final String USED_HEAP_SIZE = "Used Heap Size";
    public static final String CURRENT_HEAP_SIZE = "Current Heap Size";
    public static final String MAX_HEAP_SIZE = "Max Heap Size";

    private long prevUpTime;
    private long prevProcessCpuTime;
    private int nCPUs;

    private OperatingSystemMXBean osbean;
    private RuntimeMXBean runbean;

    public CpuMemoryProfiler() {
        osbean = (OperatingSystemMXBean) ManagementFactory.getOperatingSystemMXBean();
        runbean = ManagementFactory.getRuntimeMXBean();
        nCPUs = osbean.getAvailableProcessors();
        prevUpTime = runbean.getUptime();
        prevProcessCpuTime = osbean.getProcessCpuTime();
    }

    public Map<String, Object> sample() {
        final Runtime runtime = Runtime.getRuntime();
        long freeMemory = runtime.freeMemory();
        long totalMemory = runtime.totalMemory();
        final long usedHeapSize = totalMemory - freeMemory;

        Map<String, Object> profileData = new HashMap<String, Object>();
        profileData.put(CPU_USAGE, getCpuUsage());
        profileData.put(USED_HEAP_SIZE, usedHeapSize);
        profileData.put(CURRENT_HEAP_SIZE, totalMemory);
        profileData.put(MAX_HEAP_SIZE, runtime.maxMemory());
        return profileData;
    }

    /**
     * Returns the CPU usage in percentage.
     * @return CPU usage percent.
     */
    public double getCpuUsage() {
        long upTime = runbean.getUptime();   // in milliseconds
        long processCpuTime = osbean.getProcessCpuTime();  // in nano seconds.
        double cpuUsage;
        if (prevUpTime > 0L && upTime > prevUpTime) {
            long elapsedCpu = processCpuTime - prevProcessCpuTime;
            long elapsedTime = upTime - prevUpTime;

            // since up time and process CPU time is not taken at the same moment, there is a chance that cpu usage may
            //  come more than 100%.  That is why we cap this at 99%.
            cpuUsage = Math.min(99F, elapsedCpu / (elapsedTime * 10000F * nCPUs));
        } else {
            cpuUsage = 0.001;
        }
        prevUpTime = upTime;
        prevProcessCpuTime = processCpuTime;
        return cpuUsage;
    }
}
