package com.netapp.oci.profiler;

import com.onaro.commons.exec.Exec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;

/**
 * @author Shan Ponnusamy
 */
public class ExtProcCpuMemoryUsageProfiler implements Profiler {

    private static final Logger logger = LoggerFactory.getLogger(ExtProcCpuMemoryUsageProfiler.class);
    private static final double[] ERROR_SAMPLE = new double[] {-1, -1};

    private String processName;
    private boolean disabled;
    private String[] profilerDataName;

    public ExtProcCpuMemoryUsageProfiler(String[] profilerDataName, String processName) {

        boolean isWindows = System.getProperty("os.name").startsWith("Windows");
        if (!isWindows) {
            logger.warn("External Process CPU Usage Profiler supports only Windows OS for now.");
            disabled = true;
        }

        this.processName = processName;
        this.profilerDataName = profilerDataName;
    }

    @Override
    public Map<String, Object> sample() {
        double[] cpuUsage;
        if (!disabled) {
           cpuUsage = getWindowsCpuUsage();
        } else {
           cpuUsage = ERROR_SAMPLE;
        }
        Map<String, Object> profileData = new HashMap<String, Object>();
        profileData.put(profilerDataName[0], cpuUsage[0]);
        profileData.put(profilerDataName[1], cpuUsage[1]);
        return profileData;
    }

    private double[] getWindowsCpuUsage() {
        Exec exec = new Exec("typeperf -sc 1 \"\\Process(" + processName + ")\\% processor time\"  \"\\Process(" + processName + ")\\working set\"");
        exec.run(5 * 1000);   // 5 seconds

        if (exec.getExitCode() == 0) {
            final String output = exec.getOutput();

            /**
             * The output is similar to the following:
             * "(PDH-CSV 4.0)","\\SHANP-DEV\Process(mysqld)\% processor time","\\SHANP-DEV\Process(mysqld)\working set - private"
                "11/08/2012 09:56:25.822","0.000000","289013760.000000"
                Exiting, please wait...
                The command completed successfully.

             *  we need to parse the above string to get the cpu usage.
             */
            try {
                String[] tempStrArray = output.split(",");
                String cpuUsageStr = tempStrArray[3];
                String memUsageStr = tempStrArray[4].split("\\s")[0];                                
                double cpuUsage = Double.parseDouble(cpuUsageStr.replace("\"", ""));
                double memUsage = Double.parseDouble(memUsageStr.replace("\"", ""));
                return new double[] {cpuUsage, memUsage};
            } catch (NumberFormatException e) {
                logger.error("",e);
                disabled = true;
            }
        } else {
            logger.error(exec.getError());
            logger.error("Exit Code: " + exec.getExitCode());
            disabled = true;
        }
        return ERROR_SAMPLE;
    }

    public static void main(String[] args) {
        Map<String, Object> sample = new ExtProcCpuMemoryUsageProfiler(new String[] {"MySQL CPU Usage", "MySQL Mem Usage"}, "mysqld").sample();
        System.out.println(sample);
    }
}
