package com.netapp.oci.profiler.utils;

import java.io.*;
import java.util.*;

/**
 * This is a utility useful while analyzing the profiling data generated by the built-in profiler.  This utility
 * accepts an CSV file generated by the profiler and it generates two CSV files with prefix "MEM_" and "OrigUpdate_".
 * The CSV file with prefix "OrigUpdate_" will contain the originator update duration data.  The other generated CSV
 * file will contain the rest of the profiler data.
 *
 * @author Shan Ponnusamy
 */
public class ProfilerDataProcessor {

    private BufferedReader br;
    private BufferedWriter bwMem;
    private BufferedWriter bwOrig;
    private Map<String, Long> originatorToStartTimeMap;

    public ProfilerDataProcessor(File profilerCsvFile) throws Exception {
        br = new BufferedReader(new FileReader(profilerCsvFile));
        bwMem = new BufferedWriter(new FileWriter(profilerCsvFile.getParent() + File.separator + "Mem_" + profilerCsvFile.getName()));
        bwOrig = new BufferedWriter(new FileWriter(profilerCsvFile.getParent() + File.separator + "OrigUpdate_" + profilerCsvFile.getName()));
        originatorToStartTimeMap = new HashMap<String, Long>();
    }

    public void process() throws Exception {
        try {

            bwOrig.write(constructNewRow("OriginId", "StartTime", "EndTime", "Duration"));
            bwOrig.newLine();

            String line = br.readLine();
            String[] splits = line.split(",");
            int noOfHeaders = splits.length;
            int originStartIndex = noOfHeaders - 2;
            int originEndIndex = noOfHeaders - 1;

            while (line != null && (line.trim().length() != 0)) {
                splits = line.split(",");
                if (!splits[1].equals("")) {
                    bwMem.write(constructNewRow(Arrays.copyOf(splits, originStartIndex)));
                    bwMem.newLine();
                } else {
                    String originId;
                    long timeInMillis = Long.parseLong(splits[0]);
                    if (splits[originStartIndex].equals("")) {
                        originId = splits[originEndIndex];
                    } else {
                        originId = splits[originStartIndex];
                    }

                    if (originatorToStartTimeMap.containsKey(originId)) {
                        long startTime = originatorToStartTimeMap.get(originId);
                        bwOrig.write(constructNewRow(String.valueOf(originId), String.valueOf(startTime), String.valueOf(timeInMillis), String.valueOf(timeInMillis - startTime)));
                        bwOrig.newLine();
                        originatorToStartTimeMap.remove(originId);
                    } else {
                        originatorToStartTimeMap.put(originId, timeInMillis);
                    }
                }
                line = br.readLine();
            }
        } finally {
            br.close();
            bwMem.close();
            bwOrig.close();
        }
    }

    private String constructNewRow(String... str) {
        StringBuilder newLine = new StringBuilder(str[0]);
        for (int i = 1; i < str.length; i++) {
            newLine.append(',');
            newLine.append(str[i]);
        }
        return newLine.toString();
    }

    public static void main(String[] args) throws Exception {
        new ProfilerDataProcessor(new File(args[0])).process();
    }
}
