/*
 * Copyright (c) 2008 Akorri Networks, Inc. All Rights Reserved. This software is provided
 * under license and is not sold by Akorri Networks. This software may be used only in accordance
 * with the terms of an express written license agreement with Akorri Networks, and cannot be used,
 * copied, modified, or in any way distributed without the express permission of Akorri Networks.
 *
 */

package com.netapp.collectors.vmware;

import java.util.Calendar;
import java.util.concurrent.TimeUnit;

import com.netapp.collectors.vmware.logger.LogsManager;
import com.netapp.collectors.vmware.util.TimeUtil;
import org.slf4j.Logger;


import com.vmware.vim25.ServiceContent;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.XMLGregorianCalendar;

/**
 * Provides access to the VMWare connection.
 * 
 * @author dhetti
 *
 */
public class VMWare25Context implements IVMWareContext {
	
    /** Logger. */
    private static final Logger logger = LogsManager.getLogger(VMWare25Context.class);

    /** String indicating the vim package name. */
    private String vimPackage;

    /** boolean value telling if this is an embedded ESX. */
    private boolean isEmbeddedEsx;

    /** boolean value telling if this is an ESX. */
    private boolean isEsx;


    /** VMWare service connection. */
    private ServiceConnection25 serviceConnection;

    /** VMWare service content. */
    private ServiceContent serviceContent;	
    
    /** Virtual center instance id. */
    private Integer vcInstanceId;
    
    /** Current server time*/
    private XMLGregorianCalendar currentTime = null;

    /**
     * Sample start time and end time for collection of stats. These fields are different from startTime and endTime. They are used for poll status
     */
    private XMLGregorianCalendar sampleStartTime = null;
    private XMLGregorianCalendar sampleEndTime = null;


    
    /** Access handle id of the target device. */
    private long accessHandleId;

    private long startTime, endTime;

    private String vcUrl;

    private String vCenterHostNameOrIp;



    /**
     * Constructor to initialize the context. 
     * 
     * @param serviceConnection - connection.
     * @param serviceContent - content object.
     * @param vimPackage - vmware package information.
     * @param isEmbeddedEsx - is it a embedded ESX.
     * @param isEsx - is it an ESX.
     * @param accessHandleId id of the access handle of the target device
     * @param now - current time from connection.
     */
    public VMWare25Context(ServiceConnection25 serviceConnection, ServiceContent serviceContent, 
    		String vimPackage, boolean isEmbeddedEsx, boolean isEsx, long accessHandleId, XMLGregorianCalendar now) {
    	
    	this.serviceConnection = serviceConnection;
    	this.serviceContent = serviceContent;
    	this.vimPackage = vimPackage;
    	this.isEmbeddedEsx = isEmbeddedEsx;
    	this.isEsx = isEsx;
    	this.accessHandleId = accessHandleId;
    	this.currentTime = now;
        initializeStartAndEndTime();
	}
    
    /**
     * Constructor.
     * 
     * @param serviceConnection - Connection.
     * @param serviceContent - Content.
     * @param accessHandleId id of the access handle of the target device
     */
    public VMWare25Context(ServiceConnection25 serviceConnection, ServiceContent serviceContent, long accessHandleId) {
        this.serviceConnection = serviceConnection;
        this.serviceContent = serviceContent;
        this.accessHandleId = accessHandleId;
    }    
    
    /**
     * Constructor to initialize the context.
     * 
     * @param serviceConnection - connection.
     * @param serviceContent - content object.
     * @param vimPackage - vmware package information.
     * @param isEmbeddedEsx - is it a embedded ESX.
     * @param isEsx - is it an ESX.
     * @param vcInstanceId - virtual center instance id
     * @param accessHandleId id of the access handle of the target device
     * @param now - current time from connection.
     */
    public VMWare25Context(ServiceConnection25 serviceConnection, ServiceContent serviceContent,
                           String vimPackage, boolean isEmbeddedEsx, boolean isEsx, Integer vcInstanceId, long accessHandleId,
                           XMLGregorianCalendar now, String url, String vCenterHostNameOrIp){
        this.serviceConnection = serviceConnection;
        this.serviceContent = serviceContent;
        this.vimPackage = vimPackage;
        this.isEmbeddedEsx = isEmbeddedEsx;
        this.isEsx = isEsx;
        this.vcInstanceId = vcInstanceId;
        this.accessHandleId = accessHandleId;
    	this.currentTime = now;
    	this.vcUrl = url;
    	this.vCenterHostNameOrIp = vCenterHostNameOrIp;
    	initializeStartAndEndTime();
    }

    /**
     * Sample start time is calculated as follows
     * Current servertime  --> convert to epoch time
     * (currentServerTimeInEpoch % 300) --> Gives the reminder
     *  subtract the reminder from currentServerTimeInEpoch --> We will arrive at latest server time in multiple of 5 minutes. Lets call it A
     *  subtract 300 seconds from A to arrive at t-1th sample's end time
     *  subtract 600 seconds from A to arrive at t-1th sample's start time
     *
     */
    private void initializeStartAndEndTime() {
        //Step 1: Convert server start time to epoch
        if(this.currentTime == null) {
            logger.error("Vcenter server time is unavailable, unable to intialize the start and end time");
            return;
        }
        long epoch = this.currentTime.toGregorianCalendar().getTimeInMillis();
        //Step 2: Arrive at modulo 5
        long serverModFive = epoch - (epoch % TimeUnit.SECONDS.toMillis(300));
        //step 3: deduct 300 and 600 seconds from epoch to arrive at t-1 sample endtime and start time respectively
        long sampleEndTimeEpoch = serverModFive - TimeUnit.SECONDS.toMillis(300);
        long sampleStartTimeEpoch = serverModFive - TimeUnit.SECONDS.toMillis(600);
        try {
            this.sampleStartTime = TimeUtil.toXMLGregorianCalendar(sampleStartTimeEpoch);
            this.sampleEndTime = TimeUtil.toXMLGregorianCalendar(sampleEndTimeEpoch);
        } catch (DatatypeConfigurationException e) {
            logger.error("Error while initializing the start and end time for collection of samples");
        }
    }

    /**
     * Constructor to initialize the context.
     * 
     * @param serviceConnection - connection.
     * @param serviceContent - content object.
     * @param vcInstanceId - virtual center instance id
     * @param accessHandleId id of the access handle of the target device
     * @param now - current time from connection.
     */
    public VMWare25Context(ServiceConnection25 serviceConnection, ServiceContent serviceContent, Integer vcInstanceId, long accessHandleId, 
            XMLGregorianCalendar now){
        this.serviceConnection = serviceConnection;
        this.serviceContent = serviceContent;
        this.vcInstanceId = vcInstanceId;
        this.accessHandleId = accessHandleId;
        this.currentTime = now;
    }

    /**
     * {@inheritDoc}
     */ 
    public long getAccessHandleId(){
        return this.accessHandleId;
    }
    
    /**
     * {@inheritDoc}
     */    
    public XMLGregorianCalendar getCurrentTime() {
        return this.currentTime;
    }

    /**
     * {@inheritDoc}
     */
	public Object getConnection() {
		return this.serviceConnection;
	}

    public String getVcURL() {
        return this.vcUrl;
    }

    @Override
    public String getvCenterHostNameOrIP() {
        return vCenterHostNameOrIp;
    }

    /**
     * {@inheritDoc}
     */
	public boolean isEmbeddedEsx() {
		return this.isEmbeddedEsx;
	}

    /**
     * {@inheritDoc}
     */
	public boolean isEsx() {
		return this.isEsx;
	}

    /**
     * {@inheritDoc}
     */
	public Object getPerformanceManager() {
		return this.serviceContent.getPerfManager();
	}

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getOptionManager() {
        return this.serviceContent.getSetting();
    }

    /**
     * {@inheritDoc}
     */
	public Object getPropertyCollector() {
		return serviceConnection.getPropCol();
	}

    /**
     * {@inheritDoc}
     */
	public Object getServiceContent() {
		return this.serviceContent;
	}

    /**
     * {@inheritDoc}
     */
	public String getVimPackage() {
		return this.vimPackage;
	}
	
	/**
     * {@inheritDoc}
     */
	public Integer getVcInstanceId(){
	    return this.vcInstanceId;
	}
	
	/**
	 * {@inheritDoc} 
	 */
    public void destroy() {
        try {
            if(serviceConnection != null) {
                serviceConnection.disconnect();
            }
        } catch (Exception e) {
            if (serviceConnection != null) {
                logger.warn(DEBUG_MSG_ERROR_OCCURRED_WHILE_DISCONNECTING, e);
            } else {
                logger.warn(DEBUG_MSG_ERROR_OCCURRED_WHILE_DISCONNECTING, e);
            }
        }
	}

    public long getStartTime() {
        return startTime;
    }

    public void setStartTime(long startTime) {
        this.startTime = startTime;
    }

    public long getEndTime() {
        return endTime;
    }

    public void setEndTime(long endTime) {
        this.endTime = endTime;
    }

    @Override
    public XMLGregorianCalendar getSampleStartTime() {
        return this.sampleStartTime;
    }

    @Override
    public XMLGregorianCalendar getSampleEndTime() {
        return this.sampleEndTime;
    }
}
