package com.netapp.collectors.vmware.util;

import com.netapp.collectors.vmware.VMWareConsts;
import java.util.ArrayList;
import java.util.List;

import com.netapp.collectors.vmware.IVMWareContext;
import com.netapp.collectors.vmware.builders.PropertySpecBuilder;
import com.netapp.collectors.vmware.builders.SelectionSpecBuilder;
import com.netapp.collectors.vmware.builders.TraversalSpecBuilder;
import com.netapp.collectors.vmware.logger.LogsManager;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.ObjectSpec;
import com.vmware.vim25.PerfMetricId;
import com.vmware.vim25.PerfQuerySpec;
import com.vmware.vim25.PropertySpec;
import com.vmware.vim25.SelectionSpec;
import com.vmware.vim25.TraversalSpec;

import org.slf4j.Logger;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.XMLGregorianCalendar;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class SpecBuilderUtil {
    private static final Logger LOG = LogsManager.getLogger(SpecBuilderUtil.class);
    public static List<SelectionSpec> buildFullTraversal() {
        // Recurse through all ResourcePools
        TraversalSpec rpToRp = new TraversalSpecBuilder()
                .type("ResourcePool")
                .path("resourcePool")
                .selectSet()
                .name("rpToRp");
        // Recurse through all ResourcePools
        TraversalSpec rpToVm = new TraversalSpecBuilder()
                .type("ResourcePool")
                .path("vm")
                .selectSet(new SelectionSpecBuilder().name("rpToRp"), new SelectionSpecBuilder().name("rpToVm"))
                .name("rpToVm");
        // Traversal through ResourcePool branch
        TraversalSpec crToRp = new TraversalSpecBuilder()
                .type("ComputeResource")
                .path("resourcePool")
                .selectSet(new SelectionSpecBuilder().name("rpToRp"), new SelectionSpecBuilder().name("rpToVm"))
                .name("crToRp");
        // Traversal through host branch
        TraversalSpec crToH = new TraversalSpecBuilder()
                .type("ComputeResource")
                .path("host")
                .selectSet()
                .name("crToH");
        // Traversal through hostFolder branch
        TraversalSpec dcToHf = new TraversalSpecBuilder()
                .type("Datacenter")
                .path("hostFolder")
                .selectSet(new SelectionSpecBuilder().name("visitFolders"))
                .name("dcToHf");
        // Traversal through vmFolder branch
        TraversalSpec dcToVmf = new TraversalSpecBuilder()
                .type("Datacenter")
                .path("vmFolder")
                .selectSet(new SelectionSpecBuilder().name("visitFolders"))
                .name("dcToVmf");


        // Traversal through all datastores
        TraversalSpec dcToDsf = new TraversalSpecBuilder()
                .type("Datacenter")
                .path("datastore")
                .selectSet(new SelectionSpecBuilder().name("visitFolders"))
                .name("dcToDsf");

        // Recurse through all Hosts
        TraversalSpec hToVm = new TraversalSpecBuilder()
                .type("HostSystem")
                .path("vm")
                .skip(Boolean.FALSE)
                .selectSet(new SelectionSpecBuilder().name("visitFolders"))
                .name("hToVm");
        // Recurse through the folders
        TraversalSpec visitFolders =  new TraversalSpecBuilder()
                .type("Folder")
                .path("childEntity")
                .skip(Boolean.FALSE)
                .selectSet(
                        new SelectionSpecBuilder().name("visitFolders"),
                        new SelectionSpecBuilder().name("dcToHf"),
                        new SelectionSpecBuilder().name("dcToVmf"),
                        new SelectionSpecBuilder().name("dcToDsf"),
                        new SelectionSpecBuilder().name("crToH"),
                        new SelectionSpecBuilder().name("crToRp"),
                        new SelectionSpecBuilder().name("hToVm"),
                        new SelectionSpecBuilder().name("rpToVm"))
                .name("visitFolders");
        List<SelectionSpec> selectionSpecList = new ArrayList<>();
        selectionSpecList.add(rpToRp);
        selectionSpecList.add(rpToVm);
        selectionSpecList.add(crToRp);
        selectionSpecList.add(crToH);
        selectionSpecList.add(dcToHf);
        selectionSpecList.add(dcToVmf);
        selectionSpecList.add(dcToDsf);
        selectionSpecList.add(hToVm);
        selectionSpecList.add(visitFolders);

        return selectionSpecList;
    }

    public static List<PropertySpec> buildPropertySetForVm() {
        List<PropertySpec> vmPropertySpec = new ArrayList<>(); //TODO externalize this to be picked from a file
        PropertySpec pSpec = new PropertySpecBuilder()
                .type(VMWareConsts.VIRTUAL_MACHINE)
                .addToPathSet(PropertyFileUtil.returnPropertiesForVm());
        vmPropertySpec.add(pSpec);
        return vmPropertySpec;
    }


    public static List<ObjectSpec> buildObjectSetForVm(final ManagedObjectReference containerViewReference) {
        List<ObjectSpec> objectSpecList = new ArrayList<>();
        ObjectSpec vmObjectSpec = new ObjectSpec();
        vmObjectSpec.getSelectSet().addAll(buildSelectionSpecForDS(true)); //TODO parameterize this
        vmObjectSpec.setObj(containerViewReference);
        vmObjectSpec.setSkip(Boolean.TRUE);
        objectSpecList.add(vmObjectSpec);
        return objectSpecList;
    }

    private static List<SelectionSpec> buildSelectionSpecForVm(boolean skipInventoryGraph) {
        List<SelectionSpec> selectionSpecList = new ArrayList<>();
        TraversalSpec traversalSpec = new TraversalSpecBuilder()
                .type(VMWareConsts.CONTAINER_VIEW)
                .path(VMWareConsts.VIEW)
                .skip(Boolean.FALSE)
                .name(VMWareConsts.TRAVERSE_ENTITIES);
        selectionSpecList.add(traversalSpec);
        if (skipInventoryGraph) {
            return selectionSpecList;
        } else {
            selectionSpecList.addAll(buildSelectionSpecForVmWithInventory());
        }
        return selectionSpecList;
    }

    private static List<SelectionSpec> buildSelectionSpecForVmWithInventory() {
        List<SelectionSpec> selectionSpecList = new ArrayList<>();
        selectionSpecList.add(new TraversalSpecBuilder()
                .type(VMWareConsts.VIRTUAL_MACHINE)
                .path(VMWareConsts.RESOURCE_POOL)
                .skip(Boolean.FALSE));
        selectionSpecList.add(new TraversalSpecBuilder()
                .type(VMWareConsts.VIRTUAL_MACHINE)
                .path(VMWareConsts.NETWORK)
                .skip(Boolean.FALSE));
        return selectionSpecList;

    }

    public static List<PropertySpec> buildPropertySetForHost() {
        List<PropertySpec> hostPropertySpec = new ArrayList<>(); //TODO externalize this to be picked from a file
        PropertySpec pSpec = new PropertySpecBuilder()
                .type(VMWareConsts.HOST_SYSTEM)
                .addToPathSet(PropertyFileUtil.returnPropertiesForHosts());
        hostPropertySpec.add(pSpec);
        return hostPropertySpec;
    }



    public static List<ObjectSpec> buildObjectSetForHost(final ManagedObjectReference containerViewReference) {
        List<ObjectSpec> objectSpecList = new ArrayList<>();
        ObjectSpec hostObjectSpec = new ObjectSpec();
        hostObjectSpec.getSelectSet().addAll(buildSelectionSpecForVm(true)); //TODO parameterize this
        hostObjectSpec.setObj(containerViewReference);
        hostObjectSpec.setSkip(Boolean.TRUE);
        objectSpecList.add(hostObjectSpec);
        return objectSpecList;
    }


    public static List<PropertySpec> buildPropertySetForDS() {
        List<PropertySpec> dsPropertySpec = new ArrayList<>(); //TODO externalize this to be picked from a file
        PropertySpec pSpec = new PropertySpecBuilder()
                .type(VMWareConsts.DATASTORE)
                .addToPathSet(PropertyFileUtil.returnPropertiesForDataStore());
        dsPropertySpec.add(pSpec);
        return dsPropertySpec;

    }

    public static List<ObjectSpec> buildObjectSetForDS(final ManagedObjectReference containerViewReference) {
        List<ObjectSpec> objectSpecList = new ArrayList<>();
        ObjectSpec vmObjectSpec = new ObjectSpec();
        vmObjectSpec.getSelectSet().addAll(buildSelectionSpecForDS(true)); //TODO parameterize this
        vmObjectSpec.setObj(containerViewReference);
        vmObjectSpec.setSkip(Boolean.TRUE);
        objectSpecList.add(vmObjectSpec);
        return objectSpecList;
    }



    private static List<SelectionSpec> buildSelectionSpecForHost(boolean skipInventoryGraph) {
        List<SelectionSpec> selectionSpecList = new ArrayList<>();
        TraversalSpec traversalSpec = new TraversalSpecBuilder()
                .type(VMWareConsts.CONTAINER_VIEW)
                .path(VMWareConsts.VIEW)
                .skip(Boolean.FALSE)
                .name(VMWareConsts.TRAVERSE_ENTITIES);
        selectionSpecList.add(traversalSpec);
        if (skipInventoryGraph) {
            return selectionSpecList;
        } else {
            selectionSpecList.addAll(buildSelectionSpecForHostWithInventory());
        }
        return selectionSpecList;
    }


    private static List<SelectionSpec> buildSelectionSpecForDS(boolean skipInventoryGraph) {
        List<SelectionSpec> selectionSpecList = new ArrayList<>();
        TraversalSpec traversalSpec = new TraversalSpecBuilder()
                .type(VMWareConsts.CONTAINER_VIEW)
                .path(VMWareConsts.VIEW)
                .skip(Boolean.FALSE)
                .name(VMWareConsts.TRAVERSE_ENTITIES);
        selectionSpecList.add(traversalSpec);
        if (skipInventoryGraph) {
            return selectionSpecList;
        } else {
            selectionSpecList.addAll(buildSelectionSpecForDSWithInventory());
        }
        return selectionSpecList;
    }


    private static List<SelectionSpec> buildSelectionSpecForDSWithInventory() {
        List<SelectionSpec> selectionSpecList = new ArrayList<>();
        selectionSpecList.add(new TraversalSpecBuilder()
                .type(VMWareConsts.DATASTORE)
                .path("name")
                .skip(Boolean.FALSE));
        selectionSpecList.add(new TraversalSpecBuilder()
                .type(VMWareConsts.DATASTORE)
                .path("summary.accessible")
                .skip(Boolean.FALSE));
        selectionSpecList.add(new TraversalSpecBuilder()
                .type(VMWareConsts.DATASTORE)
                .path("name")
                .skip(Boolean.FALSE));
        selectionSpecList.add(new TraversalSpecBuilder()
                .type(VMWareConsts.DATASTORE)
                .path("summary.accessible")
                .skip(Boolean.FALSE));
        selectionSpecList.add(new TraversalSpecBuilder()
            .type(VMWareConsts.DATASTORE)
            .path("summary.capacity")
            .skip(Boolean.FALSE));
        selectionSpecList.add(new TraversalSpecBuilder()
            .type(VMWareConsts.DATASTORE)
            .path("summary.freespace")
            .skip(Boolean.FALSE));
        selectionSpecList.add(new TraversalSpecBuilder()
            .type(VMWareConsts.DATASTORE)
            .path("summary.multipleHostAccess")
            .skip(Boolean.FALSE));
        return selectionSpecList;
    }



    private static List<SelectionSpec> buildSelectionSpecForHostWithInventory() {
        List<SelectionSpec> selectionSpecList = new ArrayList<>();
        selectionSpecList.add(new TraversalSpecBuilder()
                .type(VMWareConsts.HOST_SYSTEM)
                .path(VMWareConsts.RESOURCE_POOL)
                .skip(Boolean.FALSE));
        selectionSpecList.add(new TraversalSpecBuilder()
                .type(VMWareConsts.HOST_SYSTEM)
                .path(VMWareConsts.NETWORK)
                .skip(Boolean.FALSE));
        return selectionSpecList;
    }

    /**
     * Takes a sampleStartTime and end time from context and builds the perf query specs
     * @param managedObject
     * @param vmWareContext
     * @param instance
     * @param counters
     * @return
     * @throws DatatypeConfigurationException
     */
    public static PerfQuerySpec buildPerfQuerySpec(ManagedObjectReference managedObject, final IVMWareContext vmWareContext, String instance, List<Integer> counters) throws DatatypeConfigurationException{
        XMLGregorianCalendar sampleStartTime = vmWareContext.getSampleStartTime();
        XMLGregorianCalendar sampleEndTime = vmWareContext.getSampleEndTime();
        LOG.trace("Building specs to collect the samples from start {} end {}", sampleStartTime.toString(), sampleEndTime.toString());
        return buildPerfQuerySpec(managedObject, sampleStartTime, sampleEndTime, instance, counters);
    }
    public static PerfQuerySpec buildPerfQuerySpec(ManagedObjectReference managedObject,final XMLGregorianCalendar startTime,final XMLGregorianCalendar endTime, String instance, List<Integer> counters) throws DatatypeConfigurationException {
        PerfQuerySpec perfQuerySpec = new PerfQuerySpec();
        perfQuerySpec.setEntity(managedObject);
        perfQuerySpec.setMaxSample(1); //TODO externalize this value to be fetched from property file
        perfQuerySpec.setFormat("csv");
        perfQuerySpec.setStartTime(startTime);
        perfQuerySpec.setEndTime(endTime);
        perfQuerySpec.setIntervalId(300);
        perfQuerySpec.getMetricId().addAll(buildMetricId(instance, counters));
        return perfQuerySpec;
    }

    public static List<PerfMetricId> buildMetricId(String instance, List<Integer> counters) {
        List<PerfMetricId> perfMetricIds = new ArrayList<>();
            for (Integer counter : counters) {
                PerfMetricId perfMetricId = new PerfMetricId();
                perfMetricId.setInstance(instance);
                perfMetricId.setCounterId(counter);
                perfMetricIds.add(perfMetricId);
            }
        return perfMetricIds;
    }

    public static List<PropertySpec> buildPerfCounterSpec(final ManagedObjectReference perfManager) {
        PropertySpec propertySpec = new PropertySpecBuilder()
                .addToPathSet(Arrays.asList(VMWareConsts.PERF_COUNTER))
                .all(false)
                .type(perfManager.getType());
        return Arrays.asList(propertySpec);
    }

    public static List<PropertySpec> buildPerfHistoricalIntervalSpec(ManagedObjectReference perfManager) {
        PropertySpec propertySpec = new PropertySpecBuilder()
                .addToPathSet(Arrays.asList(VMWareConsts.HISTORICAL_INTERVAL))
                .all(false)
                .type(perfManager.getType());
        return Arrays.asList(propertySpec);
    }

    public static List<Integer> buildCountersFromProperties(final List<String> properties, final Map<String,Integer> metricsCounterMap){
        List<Integer> counters = new ArrayList<>();
        for(String prop : properties) {
            counters.add(metricsCounterMap.get(prop));
        }
        return counters;
    }
}
