package com.onaro.commons.exec;

import java.io.*;

import org.slf4j.LoggerFactory;

/**
 * Date: Aug 31, 2006
 * Time: 1:40:12 PM
 */
public class Exec {

    static transient final org.slf4j.Logger logger = LoggerFactory.getLogger("com.onaro.commons.exec");//$NON-NLS-1$

    /**
     * a specified system command.
     */
    private String command;

    private File dir;

    private String[] envp;

    private int exitCode;

    private String error;

    private String output;

    private Exception exception;

    public Exec(String command) {
        this(command, null, null);
    }


    public Exec(String command, File dir) {
        this(command, dir, null);
    }

    public Exec(String command, File dir, String[] envp) {
        this.command = command;
        this.dir = dir;
        this.envp = envp;
    }

    public void run(long timout) {
        exception = null;
        output = null;
        error = null;
        exitCode = Integer.MIN_VALUE;

        ProcessOutputHandler stdout = new ProcessOutputHandler();
        ProcessOutputHandler stderr = new ProcessOutputHandler();
        try {
            final Process process = Runtime.getRuntime().exec(command, envp, dir);

            if (timout > -1) {
                final long killTime = System.currentTimeMillis() + timout;
                new Thread(new Runnable() {
                    public void run() {
                        try {
                            while (System.currentTimeMillis() < killTime && exitCode == Integer.MIN_VALUE) {
                                Thread.sleep(1000);
                            }
                            if (exitCode == Integer.MIN_VALUE) {
                                logger.info("destroy process");//$NON-NLS-1$
                                process.destroy();
                            }
                        } catch (InterruptedException e) {
                            if (exitCode == Integer.MIN_VALUE) {
                                logger.info("destroy process", e);//$NON-NLS-1$
                                process.destroy();
                            }
                        }
                    }
                }).start();
            }

            Thread currentThread = Thread.currentThread();
            stdout.start(currentThread, process.getInputStream(), "stdout");//$NON-NLS-1$
            stderr.start(currentThread, process.getErrorStream(), "stderr");//$NON-NLS-1$

            exitCode = process.waitFor();
            logger.debug("exitCode: " + exitCode);//$NON-NLS-1$

        } catch (Exception e) {
            exception = e;
            error = "Failed to exec: " + command;//$NON-NLS-1$
            logger.error("Failed to exec: " + command + " - " + e.getMessage(), e);//$NON-NLS-1$ //$NON-NLS-2$
            return;
        }

        error = stderr.getOutput();
        output = stdout.getOutput();

        exception = stderr.exception;
        if (exception == null)
            exception = stdout.exception;
    }

    public String getCommand() {
        return command;
    }

    public int getExitCode() {
        return exitCode;
    }

    public String getError() {
        return error;
    }

    public String getOutput() {
        return output;
    }

    public Exception getException() {
        return exception;
    }

    boolean isError() {
        return exception != null && error != null && error.length() > 0 && exitCode != Integer.MIN_VALUE;
    }

    class ProcessOutputHandler {

        LineNumberReader reader;

        StringBuilder output = new StringBuilder();

        Exception exception = null;

        Thread currentThread;

        String streamName;

        public void start(Thread currentThread, InputStream in, String streamName) {
            try {
                this.currentThread = currentThread;
                reader = new LineNumberReader(new InputStreamReader(in));
                this.streamName = streamName;
            } catch (Exception e) {
                exception = e;
                output.append(streamName).append(" Failed to create LineNumberReader:").append(e.getMessage());//$NON-NLS-1$
                logger.error(streamName + " Exception while execute " + command + " - " + e.getMessage(), e);//$NON-NLS-1$ //$NON-NLS-2$
                return;
            }

            String lineSeparator = System.getProperty("line.separator");//$NON-NLS-1$
            String line;
            try {
                while ((line = reader.readLine()) != null) {
                    output.append(line).append(lineSeparator);
                    if (logger.isDebugEnabled()) {
                        logger.debug(streamName + ": " + line);//$NON-NLS-1$
                    }
                }
                logger.debug(streamName + ": done");//$NON-NLS-1$
            } catch (IOException e) {
                logger.error("Exception while execute " + command + " - " + e.getMessage(), e);//$NON-NLS-1$ //$NON-NLS-2$
                exception = e;
                currentThread.interrupt();
            }
        }

        boolean getException() {
            return exception != null;
        }

        public String getOutput() {
            return output.toString();
        }
    }

    public static void main(String[] args) {
        if (args.length == 0)
            System.out.println("command missing line argument!");//$NON-NLS-1$

        String command = args[0];
        int timeout = 60000;
        if (args.length == 2) {
            timeout = Integer.parseInt(args[1]);
        }

        System.out.println("command:" + command);//$NON-NLS-1$
        System.out.println("timeout:" + timeout);//$NON-NLS-1$

        Exec exec = new Exec(command);
        exec.run(timeout);
    }
}
