/*
 * $Id $
 *
 * Copyright (c) 2020 NetApp, Inc.
 * All rights reserved.
 */
package com.onaro.commons.framework.mgmt;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

//TODO: This class is mostly derived from CertificateDownloader.java in the same package.
// If the retries and deletion can be incorporated in the original class, this can be removed.
class CertificateDownload {

    private static final int MAX_RETRY_COUNT = 2;
    private static final Logger LOG = LoggerFactory.getLogger(CertificateDownload.class);
    private final String truststorePath;
    private final String truststorePassword;

    private static final char[] HEXDIGITS = "0123456789abcdef".toCharArray();

    public CertificateDownload(final String truststorePath) {
        this(truststorePath, BasicHttpBasedUrlConnection.getServerTrustStorePassword());
    }

    public CertificateDownload(final String truststorePath, final String truststorePassword) {
        this.truststorePath = truststorePath;
        this.truststorePassword = truststorePassword;
    }

    /**
     * Download certificate from the given host and update the keystore if needed and writes it to truststore.
     *
     * @param host     Server IP
     * @param port     Server port
     * @param timeout  In millis
     * @param override always download this certificate
     * @throws CertificateException When failed to load certificate
     */
    public SSLSocketFactory downloadIfNeeded(final String host, int port, final int sslConnectionTimeout, final boolean override) throws CertificateException {
        KeyStore keyStore;
        SSLSocketFactory sslSocketFactory = null;

        int retryCount = 1;
        while (retryCount <= MAX_RETRY_COUNT) {
            try {
                if (port < 0) { // in case of "https://localhost"
                    port = 443;
                }

                keyStore = loadKeyStore(truststorePath, truststorePassword);
                System.setProperty("javax.net.ssl.trustStore", truststorePath);

                final TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
                tmf.init(keyStore);

                final X509TrustManager defaultTrustManager = (X509TrustManager) tmf.getTrustManagers()[0];
                final SavingTrustManager tm = new SavingTrustManager(defaultTrustManager);

                final SSLContext context = SSLContext.getInstance("TLS");
                context.init(null, new TrustManager[]{tm}, null);

                sslSocketFactory = context.getSocketFactory();

                LOG.debug("Opening connection to {}:{}", host, port);

                // Try to connect to server. This will check if the certificate exists and valid. If not, we will have to download and install it.
                final SSLSocket socket = (SSLSocket) sslSocketFactory.createSocket();
                socket.connect(new InetSocketAddress(host, port), sslConnectionTimeout);
                try {
                    LOG.debug("Starting SSL handshake");
                    socket.startHandshake();
                    LOG.debug("No errors, certificate is already trusted");
                    if (override && retryCount == 1) {
                        LOG.info("Download and install certificate for IP {}", host);
                    } else {
                        return sslSocketFactory;
                    }

                } catch (SSLException ssle) {
                    LOG.info("Need to download and install certificate for IP {} [SSLException = {}]",
                            host, ssle.getMessage());
                } finally {
                    socket.close();
                }

                retryCount ++;

                // Check what we got from the server
                final X509Certificate[] chain = tm.chain;
                if (chain == null) {
                    throw new CertificateException("Could not obtain server certificate chain from server " + host);
                }

                // Install certificates
                final MessageDigest sha1 = MessageDigest.getInstance("SHA1");
                final MessageDigest md5 = MessageDigest.getInstance("MD5");
                for (int i = 0; i < chain.length; i++) {
                    final X509Certificate cert = chain[i];

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("- {} Subject {}", (i + 1), cert.getSubjectDN());
                        LOG.debug("-   Issuer  {}", cert.getIssuerDN());

                        sha1.update(cert.getEncoded());
                        LOG.debug("-   SHA1    {}", toHexString(sha1.digest()));

                        md5.update(cert.getEncoded());
                        LOG.debug("-   MD5     {}", toHexString(md5.digest()));
                    }
                    final String alias = "aiqum-vc-"+ host + "-" + (i + 1);
                    keyStore.setCertificateEntry(alias, cert);
                    LOG.info("certificate was added to key store using alias: {}", alias);
                }
            } catch (IOException e) {
                throw new CertificateException(e.getMessage(), e);
            } catch (GeneralSecurityException gse) {
                throw new CertificateException("Failed to install certificate", gse);
            }

            writeToTrustStore(keyStore, truststorePath);
        }
        return sslSocketFactory;
    }

    private void writeToTrustStore(KeyStore keyStore, String truststorePath) throws CertificateException {
        OutputStream out = null;
        try {
            // Save key store
            out = new FileOutputStream(truststorePath);
            keyStore.store(out, truststorePassword.toCharArray());
        } catch (IOException ioe) {
            throw new CertificateException("Failed to save key store", ioe);
        } catch (GeneralSecurityException gse) {
            throw new CertificateException("Failed to save key store", gse);
        } finally {
            if (out != null) {
                try {
                    out.close();
                } catch (IOException ignore) {
                    LOG.error("Failed to close the trustStore File");
                }
            }
        }
    }

    public void deleteCertificates(String alias) throws GeneralSecurityException, IOException {
        KeyStore keyStore = loadKeyStore(truststorePath, truststorePassword);
        if(alias != null) {
            LOG.info("Deleting vCenter certificate for alias: {}", alias);
            if(keyStore.containsAlias(alias)) {
                keyStore.deleteEntry(alias);
                LOG.info("Deleted certificates for alias: {}", alias);
                writeToTrustStore(keyStore, truststorePath);
            } else {
                LOG.info("Can't delete the certificate for alias [{}], as there is no entry for it in the trust store", alias);
            }
        }
    }

    private static String toHexString(final byte[] bytes) {
        final StringBuilder sb = new StringBuilder(bytes.length * 3);
        for (int b : bytes) {
            b &= 0xff;
            sb.append(HEXDIGITS[b >> 4]);
            sb.append(HEXDIGITS[b & 15]);
            sb.append(' ');
        }
        return sb.toString();
    }

    /**
     * Loads and returns a {@link KeyStore}.
     *
     * @param path
     * @param password
     * @return
     * @throws GeneralSecurityException
     * @throws IOException
     */
    private KeyStore loadKeyStore(String path, String password)
            throws GeneralSecurityException, IOException
    {
        InputStream fis = new FileInputStream(path);
        try {
            final String type = KeyStore.getDefaultType();
            LOG.debug("loading {} w/ keystore.type={}", path, type);
            KeyStore keyStore = KeyStore.getInstance(type);
            keyStore.load(fis, password.toCharArray());
            return keyStore;
        } finally {
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException ignore) {}
            }
        }
    }

    private static class SavingTrustManager implements X509TrustManager {
        private final X509TrustManager tm;
        private X509Certificate[] chain;

        SavingTrustManager(final X509TrustManager tm) {
            this.tm = tm;
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return tm.getAcceptedIssuers();
        }

        @Override
        public void checkClientTrusted(final X509Certificate[] chain, final String authType)
                throws CertificateException {
            throw new UnsupportedOperationException();
        }

        @Override
        public void checkServerTrusted(final X509Certificate[] chain, final String authType)
                throws CertificateException {
            this.chain = chain;
            tm.checkServerTrusted(chain, authType);
        }
    }
}
