package com.onaro.commons.framework.mgmt;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.*;
import java.io.*;
import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Properties;
import com.onaro.commons.exception.ConfigurationException;

class CertificateDownloader {

    private static final Logger LOG = LoggerFactory.getLogger(CertificateDownloader.class);

    private final String truststorePath;
    private final String truststorePassword;
    private static final char[] HEXDIGITS = "0123456789abcdef".toCharArray();

    public CertificateDownloader(final String truststorePath) {
        this(truststorePath, BasicHttpBasedUrlConnection.getServerTrustStorePassword());
    }

    public CertificateDownloader(final String truststorePath, final String truststorePassword) {
        this.truststorePath = truststorePath;
        this.truststorePassword = truststorePassword;
    }

    /**
     * Download certificate from the given host and update the keystore if needed
     *
     * @param host     Server IP
     * @param port     Server port
     * @param timeout  In millis
     * @param override always download this certificate
     * @throws Exception When failed to load certificate
     */
    public void downloadIfNeeded(final String host, int port, final int timeout, final boolean override) throws ConfigurationException {
        final KeyStore serverTruststore;
        try {
            if (port < 0) { // in case of "https://localhost"
                port = 443;
            }

            serverTruststore = loadKeyStore(truststorePath, truststorePassword);

            final TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            tmf.init(serverTruststore);

            final X509TrustManager defaultTrustManager = (X509TrustManager) tmf.getTrustManagers()[0];
            final SavingTrustManager tm = new SavingTrustManager(defaultTrustManager);

            final SSLContext context = SSLContext.getInstance("TLS");
            context.init(null, new TrustManager[]{tm}, null);

            final SSLSocketFactory factory = context.getSocketFactory();

            LOG.debug("Opening connection to {}:{}", host, port);

            // Try to connect to server. This will check if the certificate exists and valid. If not, we will have to download and install it.
            final SSLSocket socket = (SSLSocket) factory.createSocket();
            socket.setSoTimeout(timeout);
            socket.connect(new InetSocketAddress(host, port), timeout);

            try {
                LOG.debug("Starting SSL handshake");
                socket.startHandshake();
                LOG.debug("No errors, certificate is already trusted");

                if (override) {
                    LOG.info("Download and install certificate for IP {}", host);
                } else {
                    return; // Certificate is valid. We're done here!
                }
            } catch (SSLException ssle) {
                LOG.info("Need to download and install certificate for IP {} [SSLException = {}]",
                        host, ssle.getMessage());
            } finally {
                socket.close();
            }

            // Check what we got from the server
            final X509Certificate[] chain = tm.chain;
            if (chain == null) {
                throw new ConfigurationException("Could not obtain server certificate chain from server " + host);
            }

            // Install certificates
            final MessageDigest sha1 = MessageDigest.getInstance("SHA1");
            final MessageDigest md5 = MessageDigest.getInstance("MD5");
            for (int i = 0; i < chain.length; i++) {
                final X509Certificate cert = chain[i];

                if (LOG.isDebugEnabled()) {
                    LOG.debug("- {} Subject {}", (i + 1), cert.getSubjectDN());
                    LOG.debug("-   Issuer  {}", cert.getIssuerDN());

                    sha1.update(cert.getEncoded());
                    LOG.debug("-   SHA1    {}", toHexString(sha1.digest()));

                    md5.update(cert.getEncoded());
                    LOG.debug("-   MD5     {}", toHexString(md5.digest()));
                }

                final String alias = host + "-" + (i + 1);
                serverTruststore.setCertificateEntry(alias, cert);

                LOG.info("certificate was added to key store using alias '{}", alias);
            }
        } catch (IOException e) {
            throw new ConfigurationException(e.getMessage(), e);
        } catch (GeneralSecurityException gse) {
            throw new ConfigurationException("Failed to install certificate", gse);
        }

        OutputStream out = null;
        try {
            // Save key store
            out = new FileOutputStream(truststorePath);
            serverTruststore.store(out, truststorePassword.toCharArray());
        } catch (IOException ioe) {
            throw new ConfigurationException("Failed to save key store", ioe);
        } catch (GeneralSecurityException gse) {
            throw new ConfigurationException("Failed to save key store", gse);
        } finally {
            if (out != null) {
                try {
                    out.close();
                } catch (IOException ignore) {}
            }
        }
    }

    public SSLSocketFactory loadCertificates() throws ConfigurationException {
        LOG.debug("Loading certificates!");

        try {
            KeyStore trustkeyStore = loadKeyStore(truststorePath, truststorePassword);
            final TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            tmf.init(trustkeyStore);

            final X509TrustManager defaultTrustManager = (X509TrustManager) tmf.getTrustManagers()[0];
            final SavingTrustManager tm = new SavingTrustManager(defaultTrustManager);

            final SSLContext context = SSLContext.getInstance("TLS");
            context.init(null, new TrustManager[]{tm}, null);

            return context.getSocketFactory();
        } catch (GeneralSecurityException gse) {
            throw new ConfigurationException("Failed to load certificates", gse);
        } catch (IOException ioe) {
            throw new ConfigurationException("Failed to load certificates", ioe);
        }
    }

    private static String toHexString(final byte[] bytes) {
        final StringBuilder sb = new StringBuilder(bytes.length * 3);
        for (int b : bytes) {
            b &= 0xff;
            sb.append(HEXDIGITS[b >> 4]);
            sb.append(HEXDIGITS[b & 15]);
            sb.append(' ');
        }
        return sb.toString();
    }

    public Properties buildJdbcConnectionProps() {
        final Properties jdbcProperties = new Properties();
        jdbcProperties.setProperty("javax.net.ssl.trustStoreType", KeyStore.getDefaultType());
        jdbcProperties.setProperty("javax.net.ssl.trustStorePassword", truststorePassword);
        jdbcProperties.setProperty("javax.net.ssl.keyStorePassword", truststorePassword);
        jdbcProperties.put("javax.net.ssl.trustStore", truststorePath);
        jdbcProperties.put("javax.net.ssl.keyStore", truststorePath);
        return jdbcProperties;
    }

    /**
     * Loads and returns a {@link KeyStore}.
     *
     * @param path
     * @param password
     * @return
     * @throws GeneralSecurityException
     * @throws IOException
     */
    private KeyStore loadKeyStore(String path, String password)
            throws GeneralSecurityException, IOException
    {
        InputStream fis = new FileInputStream(path);
        try {
            final String type = KeyStore.getDefaultType();
            LOG.debug("loading {} w/ keystore.type={}", path, type);
            KeyStore keyStore = KeyStore.getInstance(type);
            keyStore.load(fis, password.toCharArray());
            return keyStore;
        } finally {
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException ignore) {}
            }
        }
    }

    private static class SavingTrustManager implements X509TrustManager {
        private final X509TrustManager tm;
        private X509Certificate[] chain;

        SavingTrustManager(final X509TrustManager tm) {
            this.tm = tm;
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return tm.getAcceptedIssuers();
        }

        @Override
        public void checkClientTrusted(final X509Certificate[] chain, final String authType)
                throws CertificateException {
            throw new UnsupportedOperationException();
        }

        @Override
        public void checkServerTrusted(final X509Certificate[] chain, final String authType)
                throws CertificateException {
            this.chain = chain;
            tm.checkServerTrusted(chain, authType);
        }
    }
}
