/*
 * $Id: $
 *
 * Copyright (c) 2020 NetApp, Inc.
 * All rights reserved.
 */
package com.onaro.commons.framework.mgmt;


import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.security.GeneralSecurityException;
import java.security.cert.CertificateException;
import java.util.HashSet;
import java.util.Set;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;

/**
 * Download/install/delete certificate to/from the AU client key store.
 */
public class CertificateDownloadingService {
    private static final org.slf4j.Logger LOGGER = org.slf4j.LoggerFactory.getLogger(CertificateDownloadingService.class);

    private static CertificateDownloadingService instance = new CertificateDownloadingService();
    private CertificateDownload certificateDownloader;
    private String trustStorePath;

    private CertificateDownloadingService() {
        try {
            // Configure truststore location
            final String essentialsDir = System.getProperty("onaro.home");
            if (essentialsDir == null) {
                throw new CertificateException("onaro.home system property not set!");
            }
            trustStorePath = essentialsDir + "/jboss/server/onaro/cert/server.truststore";
            certificateDownloader = new CertificateDownload(trustStorePath);
            System.setProperty("javax.net.ssl.trustStore", trustStorePath);
            loadCertificates();
        } catch (CertificateException e) {
            LOGGER.error(e.getMessage());
            throw new RuntimeException(e.getMessage());
        }
    }

    public static CertificateDownloadingService getInstance() {
        return instance;
    }

    /**
     * Load all the certificates from java trust store to AU trust store path (/opt/net/essentials/au/conf/cert/client.keystore)
     * @throws CertificateException
     */
    private void loadCertificates() throws CertificateException {
        final File trustStore = new File(trustStorePath);
        if (!trustStore.exists() || !trustStore.isFile()) {
            final String defaultKeyStorePath = System.getProperties().getProperty("java.home") +
                    File.separator + "lib" + File.separator + "security" + File.separator + "cacerts";

            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("HttpBasedUrlConnection - Copying default truststore from " + defaultKeyStorePath);
            }

            try (final InputStream is = new BufferedInputStream(new FileInputStream(new File(defaultKeyStorePath)));
                 final OutputStream os = new BufferedOutputStream(new FileOutputStream(trustStore))) {
                trustStore.getParentFile().mkdirs();
                final byte[] buffer = new byte[100000];
                while (true) {
                    final int l = is.read(buffer);
                    if (l == -1) {
                        break;
                    }
                    os.write(buffer, 0, l);
                }

                String systemOS = System.getProperty("os.name").toLowerCase();
                if ( !systemOS.contains("window") ) {
                    Set<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
                    // Add owners permission
                    perms.add(PosixFilePermission.OWNER_READ);
                    perms.add(PosixFilePermission.OWNER_WRITE);

                    Files.setPosixFilePermissions(Paths.get(trustStorePath), perms);
                }
            } catch (IOException e) {
                throw new CertificateException("Failed to copy default key store", e);
            }
        }
    }

    /**
     * Download SSL server certificates and installs it to the client truststore (only if client truststore doesn't have the SSL server certificates)
     *
     * @param url     URL to open
     * @param timeout in millis
     * @return An initialized connection to this URL
     * @throws CertificateException
     *
     */
    public synchronized void downloadCertificates(final URL url, final int sslConnectionTimeout) throws CertificateException {
        if (!url.getProtocol().equalsIgnoreCase("HTTPS")) {
            throw new CertificateException("Not http based url : " + url.toString());
        }
        // Make sure we download the certificate from the server if none is installed (and update the SSL factory)
        downloadCertificate(url, sslConnectionTimeout, false);
    }

    /**
     * Download SSL server certificates  and installs it to the client truststore.
     * "override" param  overrides client trust store with latest SSL server certificates.
     *
     *
     * @param url     URL of server to download certificates from
     * @param timeout In millis
     * @throws CertificateException When failed to download
     */
    private void downloadCertificate(final URL url, final int sslConnectionTimeout, final boolean override) throws CertificateException {
        SSLSocketFactory sslSocketFactory = certificateDownloader.downloadIfNeeded(url.getHost(), url.getPort(), sslConnectionTimeout, override);
        if (sslSocketFactory != null) {
            HttpsURLConnection.setDefaultSSLSocketFactory(sslSocketFactory);
            // Just like the connection to Cluster, ignore the host names.
            HttpsURLConnection.setDefaultHostnameVerifier((s, sslSession) -> true);
        }
    }

    /**
     * Delete the certificate from trust store once connection is closed
     * @param alias
     */
    public synchronized void deleteCertificates(String alias) throws GeneralSecurityException, IOException {
        certificateDownloader.deleteCertificates(alias);
    }
}

