package com.onaro.commons.metrics;

import java.util.TreeMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.json.MetricsModule;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.onaro.commons.executor.ExecutorNameConstant;
import com.onaro.commons.metrics.executor.ScheduledThreadPoolMonitorExecutor;
import com.onaro.commons.metrics.executor.ThreadPoolScope;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is used to log all metrics
 */

public final class MetricMonitorLogger {
    private static final String LOGGER_NAME = "metrics";
    private static final Logger logger = LoggerFactory.getLogger(LOGGER_NAME);
    private static final Logger errorLogger = LoggerFactory.getLogger(MetricMonitorLogger.class);

    private static final ObjectMapper JSONMAPPER = new ObjectMapper();
    private static final ScheduledExecutorService metricScheduler = ScheduledThreadPoolMonitorExecutor.newSingleThreadScheduledExecutor(ExecutorNameConstant.METRIC_LOGGER_POOL, ThreadPoolScope.GLOBAL);

    public static void init() {
        JSONMAPPER.registerModule(new MetricsModule(TimeUnit.MILLISECONDS, TimeUnit.MILLISECONDS, false));
        // run every hour
        metricScheduler.scheduleAtFixedRate(() -> {
            reportThreadPoolMetricStats();
        }, 0, 1, TimeUnit.HOURS);
    }

    /**
     * This method logs all metrics stored inside METRIC_REGISTRY to aumetrics.log
     */
    public static void reportThreadPoolMetricStats() {
        String metricsJson = metricAsJson();
        logger.info(metricsJson);
    }

    /**
     * This method will convert metricRegistry to JSON and return all metrics data stored inside all METRIC_REGISTRY.
     */
    public static String metricAsJson() {
        var map = MetricsRegistryProvider.getMetricRegistryNames().stream().collect(Collectors.toMap(Function.identity(), MetricsRegistryProvider::getMetricRegistry));
        TreeMap<String, MetricRegistry> registryMetricDataMap = new TreeMap<>(map);
        try {
            return JSONMAPPER.writerWithDefaultPrettyPrinter().writeValueAsString(registryMetricDataMap);
        } catch (JsonProcessingException e) {
            errorLogger.error("Error while fetching registry metric data", e);
        }
        return "";
    }

}
