package com.onaro.commons.metrics;

import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.Histogram;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.SharedMetricRegistries;
import com.codahale.metrics.Timer;
import com.codahale.metrics.jvm.FileDescriptorRatioGauge;
import com.codahale.metrics.jvm.ThreadStatesGaugeSet;
import com.onaro.commons.metrics.executor.ThreadPoolScope;


/**
 * This class is used to instantiate a application wide registry and register common metrics to it
 */

public final class MetricsRegistryProvider {

    private static final Object MUTEX = new Object();

    private MetricsRegistryProvider() {

    }

    static {
        initRegistries();
    }

    /**
     * Initialize Registers
     */
    private static void initRegistries() {
        // Init thread pool registry
        MetricRegistry threadPoolMetricRegistry = SharedMetricRegistries.getOrCreate(RegistryKey.THREAD_POOL);
        registerJVMMetrics(threadPoolMetricRegistry);
        registerCurrentTime(threadPoolMetricRegistry);
    }

    public static void registerJVMMetrics(MetricRegistry metricRegistry) {
        metricRegistry.register("jvm.fd_usage", new FileDescriptorRatioGauge());
        metricRegistry.register("jvm.thread-states", new ThreadStatesGaugeSet());
    }

    public static void registerCurrentTime(MetricRegistry metricRegistry) {
        metricRegistry.register("current.time", (Gauge<Long>) () -> System.currentTimeMillis());
    }

    public static MetricRegistry getMetricRegistry(String registryKey) {
        return SharedMetricRegistries.getOrCreate(registryKey);
    }

    public static Set<String> getMetricRegistryNames() {
        return SharedMetricRegistries.names();
    }

    /**
     * Wrapper around MetricRegistry.register
     *
     * @param <T>
     * @param metricRegistry
     * @param metricsPrefix
     * @param monitoringKey
     * @param metric
     */
    public static <T extends Metric> T register(MetricRegistry metricRegistry, String metricsPrefix, String monitoringKey, T metric) {
        return metricRegistry.register(MetricRegistry.name(metricsPrefix, monitoringKey), metric);
    }

    /**
     * Used for registering threadpools with scope
     *
     * @param <T>
     * @param metricRegistry
     * @param metricsPrefix
     * @param monitoringKey
     * @param metric
     * @param threadPoolScope
     * @return
     */
    public static <T extends Metric> T registerThreadPool(MetricRegistry metricRegistry, String metricsPrefix, String monitoringKey, T metric, ThreadPoolScope threadPoolScope) {
        String key = MetricRegistry.name(metricsPrefix, monitoringKey);
        synchronized (MUTEX) {
            if (threadPoolScope == ThreadPoolScope.METHOD) {
                metricRegistry.remove(key);
            }
            return metricRegistry.register(key, metric);
        }
    }

    /**
     * Update duration of a running task
     *
     * @param metricRegistry
     * @param metricsPrefix
     * @param monitoringKey
     * @param startTime
     */
    public static void updateDurationTimer(MetricRegistry metricRegistry, String metricsPrefix, String monitoringKey, long startTime) {
        final long duration = System.currentTimeMillis() - startTime;
        Timer timer = MetricsRegistryProvider.timer(metricRegistry, metricsPrefix, monitoringKey);
        timer.update(duration, TimeUnit.MILLISECONDS);
    }

    /**
     * Wrapper around MetricRegistry.timer
     *
     * @param metricRegistry
     * @param metricsPrefix
     * @param monitoringKey
     */
    public static Timer timer(MetricRegistry metricRegistry, String metricsPrefix, String monitoringKey) {
        return metricRegistry.timer(MetricRegistry.name(metricsPrefix, monitoringKey));
    }

    /**
     * Wrapper around MetricRegistry.histogram
     *
     * @param metricRegistry
     * @param metricsPrefix
     * @param monitoringKey
     */
    public static Histogram histogram(MetricRegistry metricRegistry, String metricsPrefix, String monitoringKey) {
        return metricRegistry.histogram(MetricRegistry.name(metricsPrefix, monitoringKey));
    }

    /**
     * Resets the metric registry and removes all the metrics.
     */
    public static void clearAllMetrics(MetricRegistry metricRegistry) {
        for (String name : metricRegistry.getNames()) {
            metricRegistry.remove(name);
        }
    }

   /** Return value of specific gauge from a given metricRegistry in string format
     *
     * @param metricRegistry
     * @param metricsPrefix
     * @param monitoringKey
     * @return
     */
    public static String getGauge(MetricRegistry metricRegistry, String metricsPrefix, String monitoringKey) {
        return metricRegistry.getGauges().get(MetricRegistry.name(metricsPrefix, monitoringKey)).getValue().toString();
    }

    /**
     * Wrapper around MetricRegistry.remove
     *
     * @param <T>
     * @param metricRegistry
     * @param metricsPrefix
     * @param monitoringKey
     */
    public static void remove(MetricRegistry metricRegistry, String metricsPrefix, String monitoringKey) {
        metricRegistry.remove(MetricRegistry.name(metricsPrefix, monitoringKey));
    }

    public static class ThreadPoolMonitoringKey {
        public static final String TASK_EXECUTION = "task-execution";
        public static final String TASK_QUEUE_WAIT_TIME = "task-queue-wait-time";
        public static final String CORE_POOL_SIZE = "corePoolSize";
        public static final String MAX_POOL_SIZE = "maxPoolSize";
        public static final String QUEUE_SIZE = "queueSize";
        public static final String ACTIVE_THREADS = "activeThreads";
        public static final String SUBMITTED_TASKS = "submittedTasks";
        public static final String COMPLETED_TASKS = "completedTasks";
        public static final String NOT_COMPLETED_TASKS = "notCompletedasks";
        public static final String PARALLELISM_LEVEL = "parallelismLevel";
        public static final String ASYNC_MODE = "asyncMode";
        public static final String STEAL_COUNT = "stealCount";
        public static final String THREAD_POOL_SCOPE = "threadPoolScope";
    }

    public static class RegistryKey {
        public static final String THREAD_POOL = "threadPool";
    }
}
