/*
 * $Id: //depot/prod/acquisition/main/modules/commons-commons/src/main/java/com/onaro/commons/rest/NetAppPostPerformanceData.java#1 $
 *
 *
 * Copyright (c) 2020 NetApp, Inc. All rights reserved.
 */
package com.onaro.commons.rest;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.onaro.commons.exception.NetappInterfaceException;
import com.onaro.commons.util.HttpURLConnectionUtils;
import com.ontap.commons.data.dto.NetappPerformanceDto;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;


public class NetAppPostPerformanceData {

    private static final Logger logger = LoggerFactory.getLogger(NetAppPostPerformanceData.class.getName());

    private static final String COOKIE_STRING = "Cookie";
    private static final String SESSION_STRING = "JSESSIONID=";
    private static String jSessionId = "";


    private NetAppPostPerformanceData() {
    }


    /*
     * For additional use cases where more complex classes have to be converted to Json,
     * there might be need to use additional methods from jackson. Right now, this only
     * caters to the purposes of Netapp performance Dto.
     */
    public static <T> void postNetappPerformanceData(final HttpURLConnection urlConnection, final String restResource, NetappPerformanceDto<T> netappPerformanceDto)
            throws NetappInterfaceException {

        OutputStreamWriter streamWriter = null;

        try {
            if (logger.isDebugEnabled()) {
                logger.info("Sending  data to :" + urlConnection.getURL().toString());
            }

            urlConnection.setRequestMethod(RestConstants.REQUEST_TYPE_POST);
            urlConnection.setRequestProperty(RestConstants.CONTENT_TYPE_HEADER,
                    RestConstants.CONTENT_TYPE_JSON + "; " + RestConstants.UTF_ENCODING);
            urlConnection.setRequestProperty(RestConstants.ACCEPT_HEADER, RestConstants.CONTENT_TYPE_JSON);
            urlConnection.setDoOutput(true);

            /**
             * At first call, authentication will happen via username and password and jessionId here will be empty.
             * if already populated , set in the header.
             */

            if (!jSessionId.isEmpty()) {
                urlConnection.setRequestProperty(COOKIE_STRING, jSessionId);
            }


            // Get the output stream for the connection.
            streamWriter = new OutputStreamWriter(urlConnection.getOutputStream(), RestConstants.UTF_ENCODING);

            ObjectMapper mapper = new ObjectMapper();

            if (logger.isDebugEnabled()) {
                String jsonString = mapper.writeValueAsString(netappPerformanceDto);
                logger.debug("Size of samples in bytes: " + jsonString.length());
                logger.debug("Samples sent: " + jsonString);
            }

            // The method converts the Dto passed into json and streams to the specified writer
            // Mapper closes the stream unless specified otherwise
            mapper.writerWithDefaultPrettyPrinter().writeValue(streamWriter, netappPerformanceDto);

            final String errorString = getStringFromStream(urlConnection.getErrorStream());
            final String inputString = getStringFromStream(urlConnection.getInputStream());
            final String statusMessage = urlConnection.getResponseMessage();
            final int statusCode = urlConnection.getResponseCode();
            if (logger.isDebugEnabled()) {
                StringBuilder builder = new StringBuilder();
                builder.append("Status Code ").append(statusCode).append(" , status Message ").append(statusMessage).append(".");
                if (null != inputString) {
                    builder.append("Input String ").append(inputString).append(".");
                }
                if (null != errorString) {
                    builder.append("Error String ").append(errorString).append(".");
                }
                logger.debug(builder.toString());
            }

            /**
             *  Get the new jessionid:
             * 1. If jessionId is null here: same session and currentJessionId is valid one.
             * 2. If not null: new session and jesssionId needs to be set again.
             */
            String currJessionId = HttpURLConnectionUtils.getJsesionIdCookie(urlConnection);

            if (statusCode == HttpURLConnection.HTTP_OK  && currJessionId != null) {
                jSessionId = currJessionId;
            }

            // TODO: This is copied from OriginatorUpdateClientUtils. All these need to be reused
            // Also, keeping these blocks same as the earlier web client. May need to revisit this later.
            if(statusCode < 200 || statusCode > 299){
                StringBuilder builder = new StringBuilder();
                builder.append("Server returned HTTP status ").append(statusCode).append(".");
                if (errorString != null) {
                    builder.append("Error message received: ").append(errorString);
                }
                logger.error("Error while sending performance data to server:" + builder.toString());
                throw new RuntimeException(builder.toString());
            }

        } catch (Exception e) {
            // catch all block
            logger.error("Error while sending performance data to server: " + e.getMessage());
            throw new NetappInterfaceException(e.getMessage(), e);
        }
    }

    //TODO: This is copied from NetAppFoundationListener. All these need to be reused.
    private static String getStringFromStream(final InputStream stream) {
        if (stream == null) {
            return null;
        }
        BufferedReader streamReader = null;
        StringBuilder stringBuilder = new StringBuilder();
        String line;
        try {
            streamReader = new BufferedReader(new InputStreamReader(stream));
            while ((line = streamReader.readLine()) != null) {
                stringBuilder.append(line).append('\n');
            }
        } catch (Exception ignore) {
        } finally {
            if (streamReader != null) {
                try {
                    streamReader.close();
                } catch (Exception ignore) {
                    logger.error("Error closing the stream: " + ignore.getMessage());
                }
            }
        }
        return stringBuilder.toString();
    }

}
