/*
 * Copyright (c) 2020 NetApp, Inc.
 * All rights reserved.
 */

package com.onaro.commons.rest;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.text.MessageFormat;
import java.util.stream.Collectors;
import com.onaro.commons.framework.mgmt.AcquisitionHttpClientConnectionManager;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.onaro.commons.exception.NetappInterfaceException;

/**
 * ParamUtil helps to manage params stored in database, using in mega/bard
 */

public class ParamUtil {
    public static final String PARAM_REST_URL = RestConstants.ACQUISITION_API + RestConstants.PARAM;
    private static final Logger LOG = LoggerFactory.getLogger(ParamUtil.class);
    private static final String queryParams = PARAM_REST_URL + "?paramName={0}&paramType={1}&defaultValue={2}";

    private ParamUtil(){
        // This is util so making the constructor private
    }

    /**
     *
     * Get param value for the key. If no value is found, returns defaultValue
     * @param paramKey param key
     * @param defaultValue default value to return if no value is found for the paramKey
     * @return
     */
    public static String get(String paramKey, String defaultValue) {
        String value  = getParam(paramKey, String.class.getSimpleName(), defaultValue);
        return value;
    }

    /**
     *
     * Get param value for the key. If no value is found, returns defaultValue
     * @param paramKey param key
     * @param defaultValue default value to return if no value is found for the paramKey
     * @return
     */
    public static  boolean get(String paramKey, boolean defaultValue) {
        String value =  getParam(paramKey, boolean.class.getSimpleName(), defaultValue + "");
        return Boolean.parseBoolean(value);
    }

    /**
     *
     * Get param value for the key. If no value is found, returns defaultValue
     * @param paramKey param key
     * @param defaultValue default value to return if no value is found for the paramKey
     * @return
     */
    public static int get(String paramKey, int defaultValue) {
        String value =  getParam(paramKey, boolean.class.getSimpleName(), defaultValue + "");
        return Integer.parseInt(value);
    }

    /**
     *
     * Get param value for the key. If no value is found, returns defaultValue
     * @param paramKey param key
     * @param defaultValue default value to return if no value is found for the paramKey
     * @return
     */
    public static long get(String paramKey, long defaultValue) {
        String value =  getParam(paramKey, boolean.class.getSimpleName(), defaultValue + "");
        return Long.parseLong(value);
    }

    /**
     *
     *
     * Set param value for the key.
     * @param paramKey param key
     * @param value value to set for the paramKey
     * @return
     */
    public static void set(String paramKey, String value) throws NetappInterfaceException {
        saveParam(paramKey, String.class.getSimpleName(), value);
    }

    /**
     *
     * Set param value for the key.
     * @param paramKey param key
     * @param value value to set for the paramKey
     * @return
     */
    public static void set(String paramKey, boolean value) throws NetappInterfaceException {
        saveParam(paramKey, boolean.class.getSimpleName(), value + "");
    }

    /**
     *
     * Set param value for the key.
     * @param paramKey param key
     * @param value value to set for the paramKey
     * @return
     */
    public static void set(String paramKey, int value) throws NetappInterfaceException {
        saveParam(paramKey, int.class.getSimpleName(), value + "");
    }

    /**
     *
     * Set param value for the key.
     * @param paramKey param key
     * @param value value to set for the paramKey
     * @return
     */
    public static void set(String paramKey, long value) throws NetappInterfaceException {
        saveParam(paramKey, long.class.getSimpleName(), value + "");
    }

    public static String getParam(String paramName, String paramType, String defautlValue) {

        try {
            String stringUrl = MessageFormat.format(queryParams, paramName, paramType, defautlValue);
            AcquisitionHttpClientConnectionManager acquisitionHttpClientConnectionManager = AcquisitionHttpClientConnectionManager.getInstance();
            HttpURLConnection connection = acquisitionHttpClientConnectionManager.executeHTTPSGETAcquisitionAPI(stringUrl);
            int statusCode = connection.getResponseCode();

            if (statusCode == 200) {
                try(BufferedReader in = new BufferedReader(new InputStreamReader(connection.getInputStream()))) {
                    String response = in.lines().collect(Collectors.joining());
                    LOG.debug("GET Response for param " + connection.getURL().toString() + " ", response);
                    JSONParser parser = new JSONParser();
                    JSONObject json = (JSONObject) parser.parse(response);
                    return (String) json.get("paramValue");
                } catch (IOException e) {
                    String errorMessage = " - Failed to get param " + e.getMessage();
                    LOG.error(errorMessage, e);
                    return defautlValue;
                }
            }

            if (statusCode < 200 || statusCode > 299) {
                StringBuilder builder = new StringBuilder();
                builder.append("Server returned HTTP status ").append(statusCode).append(".");
                LOG.error(builder.toString());
                return defautlValue;
            }

        } catch (Exception exceptionInstance ) {
            String errorMessage = " - Failed to get param" + exceptionInstance.getMessage();
            LOG.error(errorMessage, exceptionInstance);
            return defautlValue;
        }
        return defautlValue;
    }

    public static void saveParam(String paramName, String paramType, String paramValue)  throws NetappInterfaceException {
        try {
            String resourceUrl = PARAM_REST_URL;
            AcquisitionHttpClientConnectionManager acquisitionHttpClientConnectionManager = AcquisitionHttpClientConnectionManager.getInstance();

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("paramName", paramName);
            jsonObject.put("paramType", paramType);
            jsonObject.put("paramValue", paramValue);

            HttpURLConnection connection = acquisitionHttpClientConnectionManager.executeHTTPSPOSTAcquisitionAPI(resourceUrl, jsonObject);
            int statusCode = connection.getResponseCode();

            if (statusCode == 200) {
                try(BufferedReader in = new BufferedReader(new InputStreamReader(connection.getInputStream()))) {
                    String response = in.lines().collect(Collectors.joining());
                    LOG.info("Param saved, Response for param " + connection.getURL().toString() + " ", response);
                } catch (IOException e) {
                    String errorMessage = " - Failed to save param " + e.getMessage();
                    LOG.error(errorMessage, e);
                    throw new NetappInterfaceException(errorMessage);
                }
            }

            if (statusCode < 200 || statusCode > 299) {
                StringBuilder builder = new StringBuilder();
                builder.append("Server returned HTTP status ").append(statusCode).append(".");
                LOG.error(builder.toString());
                throw new NetappInterfaceException(builder.toString());
            }

        } catch (Exception exceptionInstance ) {
            String errorMessage = " - Failed to save param" + exceptionInstance.getMessage();
            LOG.error(errorMessage, exceptionInstance);
            throw new NetappInterfaceException(errorMessage);
        }
    }
}
