/* 
 * Copyright (c) 2008 NetApp 
 * All rights reserved 
 */

package com.onaro.commons.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import javax.swing.SwingConstants;

public class CollectionPrinter <E> {
    public static final String COMMA_SEPARATED = ", "; //$NON-NLS-1$
    public static final CollectionPrinter<String> STRING = new CollectionPrinter<String>();
    public static final CollectionPrinter<String> STRING_DISPLAY = new CollectionPrinter<String>(COMMA_SEPARATED);
    public static final CollectionPrinter<Long> LONG = getComparableInstance(Long.class);
    public static final CollectionPrinter<Long> LONG_DISPLAY = getComparableInstance(COMMA_SEPARATED, null, Long.class);
    public static final CollectionPrinter<Integer> INTEGER = getComparableInstance(Integer.class);
    public static final CollectionPrinter<Integer> INTEGER_DISPLAY = getComparableInstance(COMMA_SEPARATED, null, Integer.class);

    private String separator;
    private String blankValue;
    
    private String prefix;// Any text we want to prefix the output with (might be html start tags for example)
    private String suffix;// Any text we want to prefix the output with (might be html end tags for example)
    
    private Comparator<E> comparator;
    /**
     * Preferred alignment of text. Not currently used internally by CollectionPrinter,
     * but may be useful when text is displayed or printed.
     */
    private int alignment;

    /**
     * Formats a collection of values.
     * 
     * @param separator the string to use between values
     * @param blankValue the string to use when a value is null or blank.  If null, ignore nulls/blanks.
     * @param prefix the prefix string for the output.  Can be null.  Usually used if you want to add html start tags to the
     *      output.
     * @param suffix the suffix string for the output.  Can be null.  Usually used if you want to add html end tags to the
     *      output.
     * @param comparator the comparator to use to sort values.  Can be null if no sorting is necessary.
     * @param alignment the alignment of the text
     */
    public CollectionPrinter(String separator, String blankValue, String prefix, String suffix, Comparator<E> comparator, int alignment) {
        assert (alignment == SwingConstants.LEADING || alignment == SwingConstants.TRAILING
                || alignment == SwingConstants.LEFT || alignment == SwingConstants.CENTER || alignment == SwingConstants.RIGHT);
        this.separator = separator;
        this.blankValue = blankValue;
        this.prefix = prefix;
        this.suffix = suffix;
        this.comparator = comparator;        
        this.setAlignment(alignment);
    }

    /**
     * Formats a collection of values.
     * 
     * @param separator the string to use between values
     * @param blankValue the string to use when a value is null or blank.  If null, ignore nulls/blanks.
     * @param comparator the comparator to use to sort values.  Can be null if no sorting is necessary.
     * @param alignment the alignment of the text
     */
    public CollectionPrinter(String separator, String blankValue, Comparator<E> comparator, int alignment) {
        this(separator, blankValue, null, null, comparator, alignment);
    }

    public CollectionPrinter(String separator, String blankValue, Comparator<E> comparator) {
        this (separator, blankValue, comparator, SwingConstants.LEADING);
    }

    public CollectionPrinter(String separator, Comparator<E> comparator) {
        this (separator, null, comparator);
    }

    public CollectionPrinter(String separator) {
    	this(separator, null);
    	this.comparator = new DefaultComperator();
    }

    public CollectionPrinter() {
        this(","); //$NON-NLS-1$
    }
    
    /**
     * Factory method to create a CollectionPrinter for a type that implements Comparable.
     * @param <C> the printer's target object type
     * @param typeClass the target object type
     * @return a CollectionPrinter for this type with the default separator - ","
     */
    public static <C extends Comparable<C>> CollectionPrinter<C> getComparableInstance(Class<C> typeClass) {
        return getComparableInstance(",", null, typeClass); //$NON-NLS-1$
    }
    
    /**
     * Factory method to create a CollectionPrinter for a type that implements Comparable.
     * @param <C> the printer's target object type
     * @param separator the separator to use
     * @param blankValue the value to use when a value is null or blank.  If null, then skip the value altogether
     * @param typeClass the target object type
     * @return a CollectionPrinter for this type
     */
    public static <C extends Comparable<C>> CollectionPrinter<C> getComparableInstance(String separator, String blankValue, Class<C> typeClass) {
        // Right-align numeric types.
        return Number.class.isAssignableFrom (typeClass)?
                new CollectionPrinter<C>(separator, blankValue, ComparableComparator.getInstance(typeClass), SwingConstants.RIGHT):
                new CollectionPrinter<C>(separator, blankValue, ComparableComparator.getInstance(typeClass), SwingConstants.LEADING);
    }

    public Comparator<E> getComparator() {
		return comparator;
	}

	public void setComparator(Comparator<E> comparator) {
		this.comparator = comparator;
	}

    public String getSeparator() {
        return separator;
    }

    public void setSeparator(String separator) {
        this.separator = separator;
    }

    public String getBlankValue() {
        return blankValue;
    }

    public void setBlankValue(String blankValue) {
        this.blankValue = blankValue;
    }

    public String getPrefix() {
        return prefix;
    }

    public void setPrefix(String prefix) {
        this.prefix = prefix;
    }

    public String getSuffix() {
        return suffix;
    }

    public void setSuffix(String suffix) {
        this.suffix = suffix;
    }

    /**
     * @param alignment the alignment to set. Must be one of the values accepted by JLabel.setHorizontalAlignment(),
     * namely the SwingConstants LEFT, CENTER, RIGHT, LEADING, TRAILING.
     * 
     * @see javax.swing.JLabel#setHorizontalAlignment
     */
    public void setAlignment(int alignment) {
        assert (alignment == SwingConstants.LEADING || alignment == SwingConstants.TRAILING
                || alignment == SwingConstants.LEFT || alignment == SwingConstants.CENTER || alignment == SwingConstants.RIGHT);
        this.alignment = alignment;
    }

    /**
     * @return the alignment, one of the {@link javax.swing.SwingConstants}
     */
    public int getAlignment() {
        return alignment;
    }

	public String toString(E ... array) {
        return toString(Arrays.asList(array));
    }

    /**
     * Note that the iterable may not be sorted.  Note that prefix/suffix values are not included if there are no elements in the iterable.
     * @param iterable
     * @return
     */
    public String toString(Iterable<E> iterable) {
        if (iterable != null && iterable.iterator().hasNext()) {
            StringBuilder builder = new StringBuilder();
            boolean addedValue = false;

            for (Iterator<E> iterator = iterable.iterator(); iterator.hasNext();) {
                builder.append(separator);
                
                String s = text(iterator.next());

                // If not null or an empty value, add value
                if(s != null && s.trim().length() > 0) {
                    builder.append(s);
                    addedValue = true;
                }
                else if(blankValue != null) {
                    // Do we have a blank value?  If so, add it instead
                    builder.append(blankValue);
                }
            }
            // Sometimes the text() call to get the text will return an empty string for each element of the iterator.  This would leave us with
            // a string of separators only.  If we didn't actually get a non-empty value from any text() call then just return an empty string.
            if(!addedValue) {
                return ""; //$NON-NLS-1$
            }
            // Skip the initial separator
            builder.delete(0, separator.length());

            // Add prefix if there
            if(prefix != null) {
                builder.insert(0, prefix);
            }
            // Add suffix if there
            if(suffix != null) {
                builder.append(suffix);
            }
            return(builder.toString());
        } else {
            return ""; //$NON-NLS-1$
        }
    }

    public String toString(Collection<E> collection) {
        // Checks for fast return on empty collections
        if (collection == null || collection.isEmpty()) {
            return ""; //$NON-NLS-1$
        }
        else if (collection.size() == 1 && prefix == null && suffix == null) {
            E element = collection.iterator().next();
            String text = text(element);
            
            if(text != null && text.trim().length() > 0) {
                return(text);
            }
            else if(blankValue != null) {
                return(blankValue);
            }
            return ""; //$NON-NLS-1$
        }
        return toString(sort(collection));
    }

    public String toStringNoSort(Collection<E> collection) {
        return toString((Iterable<E>)collection);
    }

    private Iterable<E> sort(Collection<E> collection) {
        if (comparator != null && collection != null && collection.size() > 1) {
            List<E> list = new ArrayList<E>(collection);
            Collections.sort(list, comparator);
            return list;
        } else {
            return collection;
        }
    }

    public String text(E value) {
        
        // Display blank cell if value is null.
        if (value == null) {
            return null;
        }
        
        // Display special text for N/A values.
        else if (ValueConstants.isNotAvailableValue(value)) {
            return ValueConstants.NOT_AVAILABLE_STRING;
        }
        
        // Do standard string conversion.
        else {
            return value.toString();
        }
    }

    private class DefaultComperator implements Comparator<E> {
        public int compare(E o1, E o2) {
            String text1 = text(o1);
            String text2 = text(o2);
            if(text1 == null) {
                return text2 == null ? 0 : -1;
            } else if (text2 == null) {
            	return 1;
            } else {
                return text1.compareTo(text2);
            }
        }
    }
}


