package com.onaro.commons.util;

import java.io.*;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {

    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);

    // magic number for Windows, 64Mb - 32Kb)
    private static int MAX_COUNT = (64 * 1024 * 1024) - (32 * 1024);
    private static TempDirCount TEMP_DIR_COUNTER = new TempDirCount();
    public static final String UTF_8 = "utf-8";
    private static final String ONARO_HOME_DIR = System.getProperty("onaro.home");
    private static final String LOG_COLLECTION_OBJECTS = System.getProperty("log.collection.objects");
    private static final String SERVER_PROPERTIES_FILE = "server.properties";
    private static final String SERVER_PROPERTIES_CHECK_CLOUDAGENT_ENABLED = "enable.cloudagent";
    private static final String SERVER_PROPERTIES_JBOSS_PASSWORD = "jboss.password";

    public static final String SERVER_PROPERTIES_BASELINE_TIMEOUT = "collection.baseline_timeout";
    public static final String SERVER_PROPERTIES_COUNTERS_TIMEOUT = "collection.counters_timeout";
    public static final String SERVER_PROPERTIES_CLOUD_AGENT_TTL = "cloud_agent.ttl_in_minutes";

    private static String ZAPI_DEBUG_FILE;
    private static String REST_DEBUG_FILE;
    private static Boolean IS_CLOUD_AGENT_ENABLED;

    static {
        if (ONARO_HOME_DIR != null) {
            ZAPI_DEBUG_FILE = ONARO_HOME_DIR.concat(File.separator).concat("..").concat(File.separator).concat("ocum").concat(File.separator).concat("zapiDebug.log");
            REST_DEBUG_FILE = ONARO_HOME_DIR.concat(File.separator).concat("..").concat(File.separator).concat("ocum").concat(File.separator).concat("restDebug.log");
        }
        IS_CLOUD_AGENT_ENABLED = null;
    }

    public static void copyFile(File in, File out) throws IOException {
        FileChannel inChannel = new FileInputStream(in).getChannel();
        FileChannel outChannel = new FileOutputStream(out, true).getChannel();
        try {
            long size = inChannel.size();
            long position = 0;
            while (position < size) {
                position += inChannel.transferTo(position, MAX_COUNT, outChannel);
            }
        } finally {
            if (inChannel != null) inChannel.close();
            if (outChannel != null) outChannel.close();
        }
    }

    /**
     * Tries to delete all the files before deleting the directory,and report on failures
     * @param directory to be deleted.
     * @param logger to log deletion failures (optional)
     * @return true if the directory was deleted
     */
    public static boolean deleteDir(File directory, org.slf4j.Logger logger){
        if(directory != null && directory.exists() && directory.isDirectory()){
            deleteDirContent(directory, logger);

            boolean deleted = directory.delete();
            if (!deleted && logger != null) {
                logger.warn("Failed to delete direcotry '" + directory + "'.");
            }
            return deleted;
        }else {
            return false;
        }
    }

    /**
     * Tries to delete all the files in a directory
     * @param directory all files and directories will be deleted
     * @param logger to log deletion failures (optional)
     */
    public static List<File> deleteDirContent(File directory, org.slf4j.Logger logger) {
        return deleteDirContent(directory, null, logger);
    }


    /**
     * Tries to delete files in a directory, based on a filter
     * @param directory files and directories will be deleted, based on the filter
     * @param filter determines which of te files and directories will be deleted
     * @param logger to log deletion failures (optional)
     */
    public static List<File> deleteDirContent(File directory, FileFilter filter, org.slf4j.Logger logger){
        if(directory != null && directory.exists() && directory.isDirectory()){
            List<File> filesFailedToDelete = new ArrayList<File>();

            //delete child files
            for (File file : directory.listFiles(filter)) {
                if(file.isDirectory()) {
                    filesFailedToDelete.addAll(deleteDirContent(file, filter, null));
                }
                boolean deleted = file.delete();
                if (!deleted) {
                    filesFailedToDelete.add(file);
                }
            }

            if(logger != null && !filesFailedToDelete.isEmpty()){
                logger.warn("Failed to delete content of direcotry '" + directory + "'. The following files inside the directory could not be deleted :" + filesFailedToDelete);
            }


            return filesFailedToDelete;
        }else {
            return Collections.emptyList();
        }
    }

    /**
     * Creates a temporary directory similar to the way File.createTempFile() works.
     * @param parentDir must be specified, or IOException will be thrown. If the directory does not exist, it will be created
     * @param prefix the prefix of the directory name. the rest of name will be a random number.
     * @return a File representing a temporary directory
     * @throws IOException if the parentDir is null or if the directory failed to be created
     */
    public static File createTempDir(File parentDir, String prefix) throws IOException {
        if(parentDir != null) {
            File tempDir = new File(parentDir, prefix + TEMP_DIR_COUNTER.inc());
            tempDir.deleteOnExit();
            boolean created = tempDir.mkdirs();
            if(created) {
                return tempDir;
            }else {
                throw new IOException("Failed to create temporary directory '" + tempDir + "'");
            }
        }else{
            throw new IOException("Failed to create temporary directory. Parent directory is not specified.");
        }
    }

    public static List<String> readLinesFromFile(File file) throws IOException {
        Reader fileReader = new InputStreamReader(new FileInputStream(file));
        try {
            return readLines(fileReader);
        } finally {
            fileReader.close();
        }
    }

    public static List<String> readLines(Reader input) throws IOException {
        BufferedReader reader = new BufferedReader(input);
        List<String> list = new ArrayList<String>();
        String line = reader.readLine();
        while (line != null) {
            list.add(line);
            line = reader.readLine();
        }
        return list;
    }

    /**
     * Read the whole file into a byte array. The method blocks until the whole file is read.
     * This is a convenience method and should be used only wnen appropriate.
     * It is not suitable for large files as their entire content will be loaded to the memory (there are other alternatives)
     * @param file the entire file will be read
     * @return a byte array with the content of the file
     * @throws IOException if the file can not be read correctly for some reason.
     */
    public static byte[] readEntireFile(File file) throws IOException {
        RandomAccessFile randomAccessFile = new RandomAccessFile(file, "r");

        if (randomAccessFile.length() <= Integer.MAX_VALUE) {
            try {
                int fileSize = (int) randomAccessFile.length();
                byte[] bytes = new byte[fileSize];
                int readSoFar = 0;

                while(readSoFar < fileSize){
                    int readThisTime = randomAccessFile.read(bytes, readSoFar, fileSize - readSoFar);
                    if (readThisTime > -1) {
                        readSoFar += readThisTime;
                    }else {
                        throw new IOException("Unexpected EOF. Only "+readSoFar+" bytes read. File size is "+fileSize);
                    }
                }

                return bytes;
            } finally {
                randomAccessFile.close();
            }
        } else {
            throw new IOException("File too big");
        }
    }

    /**
     * Read the entire text file into a String. The method blocks until the whole file is read.
     * This is a convenience method and should be used only wnen appropriate.
     * It is not suitable for large files as their entire content will be loaded to the memory (there are other alternatives)
     * @param file containing text. The entire file will be read
     * @param charset used for decoding the bytes into the string
     * @return a String with the content of the file
     * @throws IOException if the file can not be read correctly for some reason.
     */
    public static String readEntireTextFile(File file, String charset) throws IOException {
        return new String(readEntireFile(file), charset);
    }

    /**
     * Read the entire text file into a String. The method blocks until the whole file is read.Using the default charset
     * This is a convenience method and should be used only wnen appropriate.
     * It is not suitable for large files as their entire content will be loaded to the memory (there are other alternatives)
     * @param file containing text. The entire file will be read
     * @return a String with the content of the file
     * @throws IOException if the file can not be read correctly for some reason.
     */
    public static String readEntireTextFile(File file) throws IOException {
        return readEntireTextFile(file, UTF_8);
    }

    /**
     * Read the entire text stream into a String. The method blocks until the whole stream is read.Using the default charset
     * This is a convenience method and should be used only wnen appropriate.
     * It is not suitable for large files as their entire content will be loaded to the memory (there are other alternatives)
     * @param in containing text. The entire stream will be read
     * @return a String with the content of the stream
     * @throws IOException if the stream can not be read correctly for some reason.
     */
    public static String readEntireTextStream(InputStream in) throws IOException {
        int readLen;
        char[] chars = new char[1024];
        InputStreamReader reader = new InputStreamReader(in, UTF_8);
        StringBuilder sb = new StringBuilder();
        while ((readLen = reader.read(chars)) != -1) {
            sb.append(chars, 0, readLen);
        }
        in.close();
        return sb.toString();
    }


    /**
     * Write the entire string into a file
     * @param content will be written ito the file
     * @param file should exist and have write permission
     * @param append if true the content will be appended to the file
     * @param charset the text will be encoded using this charset
     * @throws IOException if writting failed for some reason
     */
    public static void writeStringIntoAFile(String content, File file, boolean append, String charset) throws IOException{
        Writer writer = new OutputStreamWriter(new FileOutputStream(file, append), charset);
        writer.write(content);
        writer.close();
    }

    /**
     * Write the entire string into a file, using the machine default encoding without appending
     * @param content will be written ito the file
     * @param file should exist and have write permission
     * @throws IOException if writting failed for some reason
     */
    public static void writeStringIntoAFile(String content, File file) throws IOException {
        writeStringIntoAFile(content, file, false, UTF_8);
    }

    private static class TempDirCount {
        private int counter = new Random().nextInt() & 0xffff;

        public synchronized int inc() {
            return counter++;
        }
    }

    /**
     * Creates a Writer to a file.
     * Alternative to FileWriter, which does not allow to specify the charset
     * @param file passed to the writer. Assume to exist and to have write permissions
     * @return writer to file
     * @throws IOException
     */
    public static Writer createUtf8FileWriter(File file) throws IOException {
        return new OutputStreamWriter(new FileOutputStream(file), UTF_8);
    }

    /**
     * Creates a Reader from a file.
     * Alternative to FileReader, which does not allow to specify the charset
     * @param file passed to the reader. Assume to exist and to have read permissions
     * @return reader from the file
     * @throws IOException
     */
    public static Reader createUtf8FileReader(File file) throws IOException {
        return new InputStreamReader(new FileInputStream(file), UTF_8);
    }

    /**
     * Retrieve a property from the server.properties file.  Will return a null
     * String if not found.
     *
     * @param property - the value to look for.
     */
    private static String getServerProperty(String property) {
        Properties prop = new Properties();
        String serverPropertiesFilePath = !ONARO_HOME_DIR.isEmpty() ? ONARO_HOME_DIR.concat(File.separator).concat("conf").concat(File.separator).concat(SERVER_PROPERTIES_FILE) : "";
        String foundProperty = null;
        try {
            prop.load(new FileInputStream(serverPropertiesFilePath));
            foundProperty = prop.getProperty(property);
        } catch (Exception e) {
            logger.error("Unable to load value of {} from file {}: {} ", property, SERVER_PROPERTIES_FILE, e);
        }
        return foundProperty;
    }

    /**
     * Get the cloud agent time to live (ttl) value.  We have a local setting,
     * but it can be overwritten in the server.properties file.
     */
    public static Integer getCloudAgentTTL() {
        String caTTL = getServerProperty(SERVER_PROPERTIES_CLOUD_AGENT_TTL);
        if (caTTL != null) {
            return Integer.valueOf(caTTL);
        }
        return null;
    }

    /**
     * Helper method to verify if cloudAgent is enabled.
     * @return boolean value indicating if cloudAgent is enabled
     */
    public static boolean checkCloudAgentEnabled() {

        // Return cached value if we've called this before
        if (IS_CLOUD_AGENT_ENABLED != null) {
            return IS_CLOUD_AGENT_ENABLED.booleanValue();
        }

        // First time call, determine if cloud agent support is enabled and cache it for subsequent calls
        String cloudAgentEnabled = getServerProperty(SERVER_PROPERTIES_CHECK_CLOUDAGENT_ENABLED);

        if (cloudAgentEnabled != null) {
            if(cloudAgentEnabled.equals("true")) {
                IS_CLOUD_AGENT_ENABLED = Boolean.TRUE;
                return IS_CLOUD_AGENT_ENABLED.booleanValue();
            }
        }

        IS_CLOUD_AGENT_ENABLED = Boolean.FALSE;
        return IS_CLOUD_AGENT_ENABLED.booleanValue();
    }

    /**
     * Helper method to verify if cloudAgent is enabled.
     * @return boolean value indicating if cloudAgent is enabled
     */
    public static String retrievePassword() {

        Properties prop = new Properties();
        String serverPropertiesFilePath = !ONARO_HOME_DIR.isEmpty() ? ONARO_HOME_DIR.concat(File.separator).concat("conf").concat(File.separator).concat(SERVER_PROPERTIES_FILE) : "";
        String jbossPassword = "";
        try {
            prop.load(new FileInputStream(serverPropertiesFilePath));
            jbossPassword = prop.getProperty(SERVER_PROPERTIES_JBOSS_PASSWORD);
        } catch (Exception e) {
            logger.error("Unable to read from file {}: {} ", SERVER_PROPERTIES_FILE, e);
        }

        if (jbossPassword != null) {
            return jbossPassword;
        }

        return "";
    }
    
    /**
     * Helper method to return string server properties.
     * @return boolean value indicating if cloudAgent is enabled
     */
    public static String getStringServerProperty(final String property) {

        Properties prop = new Properties();
        String serverPropertiesFilePath = !ONARO_HOME_DIR.isEmpty() ? ONARO_HOME_DIR.concat(File.separator).concat("conf").concat(File.separator).concat(SERVER_PROPERTIES_FILE) : "";
        String result = "";
        try {
            prop.load(new FileInputStream(serverPropertiesFilePath));
            result = prop.getProperty(property);
        } catch (Exception e) {
            logger.error("Unable to read from file {}: {} ", SERVER_PROPERTIES_FILE, e);
        }

        if (result != null) {
            return result;
        }

        return "";
    }


    /**
     * Check to see if the logging of the collection objects is requested
     */
    public static boolean isObjectLoggingEnabled() {
        if (LOG_COLLECTION_OBJECTS == null) {
            return false;
        } else {
            if (LOG_COLLECTION_OBJECTS.equals("true")) {
                return true;
            }
        }
        return false;
    }

    /**
     * Write to the ZAPI debug log.  Developers should use the method isObjectLoggingEnabled
     * in their code to make sure it is enabled before writing to this file.
     *
     * @param msg - to write.
     */
    // File to write the ZAPI debug.
    public synchronized static void writeToZapiDebug(String msg) {
        writeToFile(msg, ZAPI_DEBUG_FILE);
    }

    /**
     * Write to the REST debug log.  Developers should use the method isObjectLoggingEnabled
     * in their code to make sure it is enabled before writing to this file.
     *
     * @param msg - to write.
     */
    // File to write the REST debug.
    public synchronized static void writeToRestDebug(String msg) {
        writeToFile(msg, REST_DEBUG_FILE);
    }

    /**
     * Private method to write to a specific file.
     *
     * @param msg to write
     * @param fileName to write to.
     */
    private static void writeToFile(String msg, String fileName) {
        try {
            // Make sure that the file exists
            File outFile = new File(fileName);
            if (!outFile.exists()) {
                outFile.createNewFile();
            }

            FileWriter w = new FileWriter(outFile, true);
            w.write(msg);
            w.close();
        } catch (Exception e) {
            logger.error("Failed to write to file", e);
        }

    }

}
