package com.onaro.commons.util;

public enum Unit {

    BYTE,
    KILOBYTE(BYTE, BYTE.conversionFactor * 1024),
    MEGABYTE(BYTE, KILOBYTE.conversionFactor * 1024),
    GIGABYTE(BYTE, MEGABYTE.conversionFactor * 1024),
    TERABYTE(BYTE, GIGABYTE.conversionFactor * 1024),
    
    MILLISECOND,
    SECOND(MILLISECOND, MILLISECOND.conversionFactor * 1000),
    MINUTE(MILLISECOND, SECOND.conversionFactor * 60),
    HOUR(MILLISECOND, MINUTE.conversionFactor * 60),
    DAY(MILLISECOND, HOUR.conversionFactor * 24),
    WEEK(MILLISECOND, DAY.conversionFactor * 7),
    MONTH(MILLISECOND, DAY.conversionFactor * 31), // Assume 31 days in a month for conversion purposes
    QUARTER(MILLISECOND, MONTH.conversionFactor * 3),
    YEAR(MILLISECOND, DAY.conversionFactor * 365), // Assume 365 days in a year for conversion purposes
    
    ;
    
    private final Unit baseUnit;
    private final long conversionFactor;
    
    Unit() {
        this(null, 1l);
    }
    
    Unit(Unit baseUnit, long conversionFactor) {
        this.baseUnit = baseUnit;
        this.conversionFactor = conversionFactor;
    }
    
    public static long convert(long quantity, Unit sourceUnit, Unit targetUnit) {
        return sourceUnit.convertTo(quantity, targetUnit);
    }
    
    public static double convert(double quantity, Unit sourceUnit, Unit targetUnit) {
        return sourceUnit.convertTo(quantity, targetUnit);
    }
    
    public boolean canConvertTo(Unit target) {
        if (this == target) {
            return true;
        }
        
        // Target can't be null
        if (target == null || 
           // If this is a base unit, ensure that the target's base unit is this
           (baseUnit == null && target.baseUnit != this) ||
           // If the target is a base unit, ensure that the target is this's base unit
           (target.baseUnit == null && this.baseUnit != target)) {
            return false;
        }
        return true;
    }
    
    public double convertTo(double quantity, Unit target) {
        if (!canConvertTo(target)) {
            throw new IllegalArgumentException("Incompatible units for conversion");
        }
        if (this == target) {
            return quantity;
        }
        
        if (conversionFactor > target.conversionFactor) {
            double conversionRatio = (conversionFactor / target.conversionFactor);
            return quantity * conversionRatio;            
        } else {
            double conversionRatio = (target.conversionFactor / conversionFactor);
            return quantity / conversionRatio;
        }
    }
    
    public long convertTo(long quantity, Unit target) {
        if (!canConvertTo(target)) {
            throw new IllegalArgumentException("Incompatible units for conversion");
        }
        if (this == target) {
            return quantity;
        }
        
        if (conversionFactor > target.conversionFactor) {
            double conversionRatio = (conversionFactor / target.conversionFactor);
            return Math.round(quantity * conversionRatio);            
        } else {
            double conversionRatio = (target.conversionFactor / conversionFactor);
            return Math.round(quantity / conversionRatio);
        }
    }
    
    public Double convertTo(Number quantity, Unit target) {
        return Double.valueOf(convertTo(quantity.doubleValue(), target));
    }
}
