package com.onaro.commons.util;

import org.slf4j.LoggerFactory;

/**
 * Class used to compare WWN strings using a colon-separated
 * hexadecimal byte format, i.e., "xx:xx:xx:xx:xx:xx:xx:xx".
 */
public final class WwnComparable implements Comparable<WwnComparable> {
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(WwnComparable.class);

    private final String wwn;
    private short[] wwnHex;
    private Boolean isStandardFormat = null;
    private static final int HEX_RADIX = 16;
    private static final int WWN_LENGTH = 8;

    /**
     * Creates a {@link WwnComparable}.
     * @param wwn world-wide name string
     */
    public WwnComparable(String wwn) {
        this.wwn = wwn;
    }

    public int compareTo(WwnComparable o) {
        short[] hex = getWwnHex();
        short[] oHex = o.getWwnHex();
        if (hex == null) {
            if (oHex == null) {
                return 0;
            }
            return -1;
        }
        if (oHex == null) {
            return 1;
        }

        // Shouldn't be different lengths, but will handle:
        int compareLength = Math.min(hex.length, oHex.length);
        for (int i = 0; i < compareLength; i++) {
            int diff = hex[i] - oHex[i];
            if (diff != 0) {
                return diff;
            }
        }

        // Should be zero; otherwise, the longer is last:
        return hex.length - oHex.length;
    }

    /**
     * Gets the WWN as an array of byte values
     * @return array of 8 shorts
     */
    public short[] getWwnHex() {
        maybeInitWwnHex();
        return wwnHex;
    }

    /**
     * Indicates if WWN is in the standard OCI format
     * @return {@code true} if the format is standard format, else {@code false}
     */
    public boolean isStandardFormat() {
        maybeInitWwnHex();
        return isStandardFormat.booleanValue();
    }

    private void maybeInitWwnHex() {
        if (wwnHex == null && isStandardFormat == null) {
            String[] hexStrings = wwn.split(":"); //$NON-NLS-1$
            int length = hexStrings.length;

            // Expecting a length of 8; warn otherwise, but should still function
            if (length != WWN_LENGTH) {
                logger.warn("WWN format not recognized \"" + wwn + "\""); //$NON-NLS-1$ //$NON-NLS-2$
            }

            try {
                short[] newWwnHex = new short[length];
                for (int i=0; i < length; i++) {
                    newWwnHex[i] = Short.parseShort(hexStrings[i], HEX_RADIX);
                }
                wwnHex = newWwnHex;
                isStandardFormat = Boolean.TRUE;
            }
            catch (NumberFormatException e) {
                // Wasn't in proper format:
                isStandardFormat = Boolean.FALSE;
            }
        }

    }

    @Override
    public String toString() {
        return wwn;
    }
}