package com.onaro.commons.util;

import java.io.*;
import java.util.zip.*;

/**
 * The classes in the java.util.zip package support two widespread compression
 * formats: GZIP and ZIP. Both of these are based on the ZLIB compression algorithm,
 * which is discussed in RFC 1950, RFC 1951, and RFC 1952. These documents are
 * available at ftp://ds.internic.net/rfc/.
 */
public class Zip {

    private static int CHUNK = 8192;

    /**
     * create gzFile from file
     *
     * @param fileName   the source file name
     * @param gzFileName the target gz file name
     * @throws IOException if there where IOError File not exist, Failed to create etc.
     */
    static public void toGz(String fileName, String gzFileName) throws IOException {
        GZIPOutputStream zipOut = null;
        InputStream in = null;
        OutputStream out = null;
        try {
            out = new BufferedOutputStream(new FileOutputStream(gzFileName));
            zipOut = new GZIPOutputStream(out);
            byte[] buffer = new byte[CHUNK];
            // Compress the file.

            in = new BufferedInputStream(new FileInputStream(fileName));
            int length;
            while ((length = in.read(buffer, 0, CHUNK)) != -1)
                zipOut.write(buffer, 0, length);
        } finally {
            if (in != null)
                in.close();
            if (zipOut != null)
                zipOut.close();
            if (out != null)
                out.close();
        }
    }

    /**
     * restore file from gzFile
     *
     * @param gzFileName the source gz file name
     * @param fileName   the target file name
     * @throws IOException if there where IOError File not exist, Failed to create etc.
     */
    static public void fromGz(String gzFileName, String fileName) throws IOException {
        OutputStream out = null;
        InputStream in = null;
        GZIPInputStream zipIn = null;
        try {
            in = new BufferedInputStream(new FileInputStream(gzFileName));
            zipIn = new GZIPInputStream(in);
            // Decompress the file.
            byte[] buffer = new byte[CHUNK];
            out = new BufferedOutputStream(new FileOutputStream(fileName));
            int length;
            while ((length = zipIn.read(buffer, 0, CHUNK)) != -1)
                out.write(buffer, 0, length);
        } finally {
            if (out != null)
                out.close();
            if (zipIn != null)
                zipIn.close();
            if (in != null)
                in.close();
        }
    }


    static public void toZip(String[] fileNames, ByteArrayOutputStream[] streamArray, OutputStream outputStream) throws IOException {
        ZipOutputStream zipOut = null;
        try {
            zipOut = new ZipOutputStream(outputStream);
            zipOut.setMethod(ZipOutputStream.DEFLATED);

            for (int i = 0; i < fileNames.length; i++) {
                CRC32 crc = new CRC32();
                crc.reset();
                ByteArrayOutputStream byteArrayOutputStream = streamArray[i];
                byte[] byteArray = byteArrayOutputStream.toByteArray();
                crc.update(byteArray);
                String fileName = fileNames[i];
                ZipEntry zipEntry = new ZipEntry(fileName);
                zipEntry.setCrc(crc.getValue());
                zipEntry.setSize(byteArrayOutputStream.size());

                zipOut.putNextEntry(zipEntry);
                zipOut.write(byteArray);
                zipOut.closeEntry();
            }
        } finally {
            if (zipOut != null)
                zipOut.close();

        }
    }

    static public void toZip(String[] fileNames, String zipFileName) throws IOException {
        toZip(fileNames, new FileOutputStream(zipFileName), true, "");
    }

    static public void toZip(String[] fileNames, OutputStream outputStream, boolean flatDir) throws IOException {
        toZip(fileNames, outputStream, flatDir, "");
    }

    static public void toZip(String[] fileNames, OutputStream outputStream, String relativeDir) throws IOException {
        toZip(fileNames, outputStream, false, relativeDir);
    }

    /**
     * create gzFile from file
     *
     * @param fileNames    the source file name
     * @param outputStream the target gz file name
     * @throws IOException if there where IOError File not exist, Failed to create etc.
     */
    static private void toZip(String[] fileNames, OutputStream outputStream, boolean flatDir, String relativeDir) throws IOException {
        ZipOutputStream zipOut = null;
        InputStream fileInputStream = null;
        try {
            zipOut = new ZipOutputStream(outputStream);
            zipOut.setMethod(ZipOutputStream.DEFLATED);
            byte[] buffer = new byte[CHUNK];
            // Compress the file.
            for (String fileName : fileNames) {
                File file = new File(fileName);
                if (!file.exists())
                    continue;

                fileInputStream = new BufferedInputStream(new FileInputStream(file));
                CRC32 crc = new CRC32();
                crc.reset();
                int length;
                while ((length = fileInputStream.read(buffer)) != -1) {
                    crc.update(buffer, 0, length);
                }
                fileInputStream.close();
                fileInputStream = null;

                if (flatDir) {
                    int idx = fileName.lastIndexOf('/');
                    if (idx == -1)
                        idx = fileName.lastIndexOf('\\');
                    if (idx > -1)
                        fileName = fileName.substring(idx, fileName.length());
                } else if (relativeDir.length() > 0) {
                    if (fileName.indexOf(relativeDir) == 0) {
                        fileName = fileName.substring(relativeDir.length());
                    }
                }

                ZipEntry zipEntry = new ZipEntry(fileName);
                zipEntry.setCrc(crc.getValue());
                zipEntry.setSize(file.length());
                zipEntry.setTime(file.lastModified());

                zipOut.putNextEntry(zipEntry);
                fileInputStream = new BufferedInputStream(new FileInputStream(file));

                while ((length = fileInputStream.read(buffer)) > -1) {
                    zipOut.write(buffer, 0, length);
                }
                fileInputStream.close();
                zipOut.closeEntry();
            }
        } finally {
            if (fileInputStream != null)
                fileInputStream.close();
            if (zipOut != null)
                zipOut.close();

        }
    }

    static public void fromZip(String zipFileName, File dir) throws IOException {
        fromZip(new BufferedInputStream(new FileInputStream(zipFileName)), dir);
    }

    static public void fromZip(InputStream inputStream, File dir) throws IOException {
        ZipInputStream zipIn = null;
        OutputStream bufferedOutputStream = null;
        try {
            if (!dir.exists())
                dir.mkdir();

            zipIn = new ZipInputStream(inputStream);
            ZipEntry entry;
            while ((entry = zipIn.getNextEntry()) != null) {
                int count;
                byte data[] = new byte[CHUNK];
                // write the files to the disk
                File outputFile = new File(dir.getPath() + "\\" + entry.getName());
                outputFile.getParentFile().mkdirs();
                FileOutputStream fos = new FileOutputStream(outputFile); //$NON-NLS-1$
                bufferedOutputStream = new BufferedOutputStream(fos, CHUNK);
                while ((count = zipIn.read(data, 0, CHUNK)) != -1) {
                    bufferedOutputStream.write(data, 0, count);
                }
                bufferedOutputStream.flush();
                bufferedOutputStream.close();
                fos.close();
            }
            zipIn.close();
        } finally {
            if (zipIn != null)
                zipIn.close();
            if (bufferedOutputStream != null)
                bufferedOutputStream.close();
        }
    }

    /**
     * Decompresses array of bytes using zip algorithm
     *
     * @param compressed - array of bytes, compressed data
     * @return initial array of bytes
     * @throws java.io.IOException if failed to decompress
     */
    public static byte [] decompress(byte[] compressed) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ByteArrayInputStream in = new ByteArrayInputStream(compressed);
        ZipInputStream zin = new ZipInputStream(in);
        try {
            zin.getNextEntry();
            byte[] buffer = new byte[1024];
            int offset;
            while((offset = zin.read(buffer)) != -1) {
                out.write(buffer, 0, offset);
            }
        } finally {
            out.close();
            zin.close();
        }

        return out.toByteArray();
    }

    /**
     * Compresses string using zip algorithm
     *
     * @param bytes - bytes array to compress
     * @return array of bytes as a result of compression
     * @throws java.io.IOException if failed to compress
     */
    public static byte[] compress(byte [] bytes) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ZipOutputStream zout = new ZipOutputStream(out);
        try {
            zout.putNextEntry(new ZipEntry("0")); //$NON-NLS-1$
            zout.write(bytes);
            zout.closeEntry();
        } finally {
            zout.close();
        }

        return out.toByteArray();
    }


    public static void main(String[] args) throws IOException {
        Zip.toZip(new String[]{"c:\\TEMP\\XXX\\New_Sym_Test3_oData.xml", "c:\\TEMP\\XXX\\Sym_Test3_oData.xml"}, "c:\\TEMP\\XXX\\test.zip"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        Zip.fromZip("c:\\TEMP\\XXX\\test.zip", new File("c:\\TEMP\\XXX\\ddd2")); //$NON-NLS-1$ //$NON-NLS-2$
    }
}