/*
 * Copyright (c) 2008 Akorri Networks, Inc. All Rights Reserved. This software is provided
 * under license and is not sold by Akorri Networks. This software may be used only in accordance
 * with the terms of an express written license agreement with Akorri Networks, and cannot be used,
 * copied, modified, or in any way distributed without the express permission of Akorri Networks.
 *
 */

package com.netapp.collectors.vmware;

import com.netapp.collectors.vmware.exceptions.VMWareCertificateException;
import com.netapp.collectors.vmware.exceptions.VMWareConnectionException;
import com.netapp.collectors.vmware.util.certificates.TrustStoreKey;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.util.Map;
import java.io.IOException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import javax.xml.ws.BindingProvider;

import com.vmware.vim25.InvalidLocaleFaultMsg;
import com.vmware.vim25.InvalidLoginFaultMsg;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.ServiceContent;
import com.vmware.vim25.VimPortType;
import com.vmware.vim25.VimService;

import com.netapp.collectors.vmware.logger.LogsManager;
import com.onaro.commons.framework.mgmt.CertificateDownloadingService;
import org.slf4j.Logger;

/**
 * This class is used to make the connection to the VirtualCenter or ESX host.
 * 
 * @author dhetti
 *
 */
public class ServiceConnection25 {

    /** Connection State.*/
    public static int ConnectionState_Connected = 0;
    /** Connection State.*/
    public static int ConnectionState_Disconnected = 1;

    /** Service.*/
    protected VimPortType service;
    /**Service State.*/
    protected int serviceState;
    /**Service Content.*/
    protected ServiceContent serviceContent;
    /**Service Reference. */
    protected ManagedObjectReference serviceReference;

    /** Logger. */
    private static final Logger LOGGER = LogsManager.getLogger(ServiceConnection25.class);

    /**
     * Constructor.
     */
    public ServiceConnection25() {
       serviceState = ConnectionState_Disconnected;

       serviceReference = new ManagedObjectReference();
       serviceReference.setType("ServiceInstance");
       serviceReference.setValue("ServiceInstance");
    }
    
    /**
     * Creates an instance of the VMA proxy and establishes a connection. 
     *
     * @param urlStr web service url.
     * @param username authorized user.
     * @param password authorized password for user.
     * @param ignoreCerts certificate.
     */
    public void connect(String urlStr, String username, String password, boolean ignoreCerts) throws NoSuchAlgorithmException, KeyManagementException, VMWareConnectionException, VMWareCertificateException{
        this.connect(urlStr, username, password, 0, ignoreCerts);
    }
    
    /**
     * Creates an instance of the VMA proxy and establishes a connection. The timeout can be set in milliseconds.
     * Recommended read timeout is 1800000 milliseconds which is 30min. If a value of 0 is set, no read timeout set.
     *
     * @param urlStr web service url.
     * @param username authorized user.
     * @param password authorized password for user.
     * @param readTimeout read time out setting in milliseconds.
     * @param ignoreCerts certificate.
     */
    public void connect(String urlStr, String username, String password, int sslConnectionTimeout, boolean ignoreCerts) throws VMWareConnectionException, VMWareCertificateException {
        if (service != null) {
            disconnect();
        }
        try {
            // Try connection to the VCenter without any certificate handling
            connectToVCenter(urlStr, username, password, false);
        } catch ( VMWareConnectionException ex) {
            /*
             * vim25 catches all the exceptions and returns only a few back
             * So we cannot catch a specific SSLException and handle it alone
             * Instead, we just re-try once no matter what the error is.
             */
            handleCertificates(urlStr, ignoreCerts, sslConnectionTimeout);
            connectToVCenter(urlStr, username, password, true);
        }
    }

    // The logError indicates if the exception needs to be logged with traces

    private void connectToVCenter(String urlStr, String username, String password, boolean logError) throws VMWareConnectionException {

        try {
            VimService vimService = new VimService();
            VimPortType vimPort = vimService.getVimPort();
            this.service = vimPort;
            //this.serviceContent = vimPort.retrieveServiceContent(serviceReference);

            Map<String, Object> ctxt = ((BindingProvider) vimPort).getRequestContext();
//
//// Store the Server URL in the request context and specify true
//// to maintain the connection between the client and server.
//// The client API will include the Server's HTTP cookie in its
//// requests to maintain the session. If you do not set this to true,
//// the Server will start a new session with each request.
            ctxt.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, urlStr);
            ctxt.put(BindingProvider.SESSION_MAINTAIN_PROPERTY, true);

// Retrieve the ServiceContent object and login
            serviceContent = vimPort.retrieveServiceContent(serviceReference);

            vimPort.login(serviceContent.getSessionManager(),
                    username,
                    password,
                    null);

            serviceState = ConnectionState_Connected;

        } catch (RuntimeFaultFaultMsg runtimeFaultFaultMsg) {
            LOGGER.error(String.format("Error while connecting to the VCenter url %s user %s", urlStr, username), runtimeFaultFaultMsg);
            throw new VMWareConnectionException(String.format("Error while connecting to the vcenter url %s user %s", urlStr, username));
        } catch (InvalidLocaleFaultMsg invalidLocaleFaultMsg) {
            LOGGER.error(String.format("Invalid locale information VCenter url %s user %s", urlStr, username), invalidLocaleFaultMsg);
            throw new VMWareConnectionException(String.format("Invalid locale information vcenter url %s user %s", urlStr, username));
        } catch (InvalidLoginFaultMsg invalidLoginFaultMsg) {
            LOGGER.error(String.format("%s VCenter url %s user %s", VMWareConsts.INVALID_LOGIN, urlStr, username), invalidLoginFaultMsg);
            throw new VMWareConnectionException(VMWareConsts.INVALID_LOGIN);
        }  catch (Exception e) {
		if (logError) {

	            LOGGER.error(String.format("connection to  VCenter url %s user %s failed !", urlStr, username), e);
        	    throw new VMWareConnectionException(String.format("connection failure:  %s user %s", urlStr, username));
		} else {
			LOGGER.warn(String.format("Connection to VCenter url %s user %s failed. Retrying with proper certificate...", urlStr,username));
        	        throw new VMWareConnectionException(String.format("connection failure:  %s user %s", urlStr, username));
		}
        }
    }

    private void handleCertificates(String url, boolean ignoreCerts, int sslConnectionTimeout) throws VMWareCertificateException{
        if(ignoreCerts) {
            trustAllCertificates();
        } else {
            downloadCertificatesIfNecessary(url, sslConnectionTimeout);
        }
    }

    /**
     * Trust all the certificates. It Should not be used in production environment.
     */
    private void trustAllCertificates() {
        ignoreCert();
        try {
            trustCertificates();
        } catch (NoSuchAlgorithmException | KeyManagementException e) {
            throw new RuntimeException("Error while ignoring the certificate. Reason - " + e.getMessage());
        }
    }

    private void trustCertificates() throws NoSuchAlgorithmException, KeyManagementException {
        // Create the trust manager.
        javax.net.ssl.TrustManager[] trustAllCerts = new javax.net.ssl.TrustManager[1];
        javax.net.ssl.TrustManager tm = new TrustAllTrustManager();
        trustAllCerts[0] = tm;
        // Create the SSL context
        javax.net.ssl.SSLContext sc = javax.net.ssl.SSLContext.getInstance("SSL");

        // Create the session context
        javax.net.ssl.SSLSessionContext sslsc = sc.getServerSessionContext();

        // Initialize the contexts; the session context takes the trust manager.
        sslsc.setSessionTimeout(0);
        sc.init(null, trustAllCerts, null);

        // Use the default socket factory to create the socket for the secure connection
        javax.net.ssl.HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());

        // Set the default host name verifier to enable the connection.
        HttpsURLConnection.setDefaultHostnameVerifier(hv);

    }

    /**
     * Download and install certificate for VCenter and update the keystore if needed.
     * @param urlString VCenter IP
     * @param timeout
     */
    private void downloadCertificatesIfNecessary(String urlString, int sslConnectionTimeout) throws VMWareCertificateException{
        try {
            URL url = new URL(urlString);
            CertificateDownloadingService certificateDownloadingService = CertificateDownloadingService.getInstance();
            certificateDownloadingService.downloadCertificates(url, sslConnectionTimeout);
        } catch (IOException | CertificateException exception) {
            String errMessage = String.format("Exception during certificate download for URL - %s", urlString);
            LOGGER.error(errMessage,exception);
            throw new VMWareCertificateException(errMessage);
        }
    }

    /**
     * Ignore certificate.
     */
    private void ignoreCert() {
       System.setProperty("org.apache.axis.components.net.SecureSocketFactory",
                             "org.apache.axis.components.net.SunFakeTrustSocketFactory");
    }
    
    /**
     * Check if Connection is established or not.
     * 
     * @return true if connected, false otherwise.
     */
    public boolean isConnected() {
       return serviceState == ConnectionState_Connected;
    }

    /**
     * returns the service. 
     * 
     * @return Service instance.
     */
    public VimPortType getService() {      
       return service;
    }
    
    /**
     * get Service instance reference. 
     * 
     * @return Service MO Reference.
     */
    public ManagedObjectReference getServiceInstanceRef() {
       return serviceReference;
    }

    /**
     * get service content.
     * 
     * @return Service instance content.
     */
    public ServiceContent getServiceContent() {
       return serviceContent;
    }

    /**
     * get property collector.
     * 
     * @return Service property collector.
     */
    public ManagedObjectReference getPropCol() {
       return serviceContent.getPropertyCollector();
    }

    /**
     * get root folder information.
     * 
     * @return Root folder
     */
    public ManagedObjectReference getRootFolder() {
       return serviceContent.getRootFolder();
    }

    /**
     * getService Connection State.
     * 
     * @return Service instance.
     */
    public int getServiceState() {
       return serviceState;
    }

    /**
     * Disconnects from the WebService.
     */
    public void disconnect() {
        try {
            this.getService().logout(this.getServiceContent().getSessionManager());
        } catch (RuntimeFaultFaultMsg runtimeFaultFaultMsg) {
            throw new RuntimeException("Failed while disconnecting from vim service");
        }
    }

    private static class TrustAllTrustManager implements javax.net.ssl.TrustManager,
            javax.net.ssl.X509TrustManager {

        public java.security.cert.X509Certificate[] getAcceptedIssuers() {
            return null;
        }

        public boolean isServerTrusted(java.security.cert.X509Certificate[] certs) {
            return true;
        }

        public boolean isClientTrusted(java.security.cert.X509Certificate[] certs) {
            return true;
        }

        public void checkServerTrusted(java.security.cert.X509Certificate[] certs,
                                       String authType)
                throws java.security.cert.CertificateException {
            return;
        }

        public void checkClientTrusted(java.security.cert.X509Certificate[] certs,
                                       String authType)
                throws java.security.cert.CertificateException {
            return;
        }
    }
    HostnameVerifier hv = new HostnameVerifier() {
        @Override
        public boolean verify(String s, SSLSession sslSession) {
            return false;
        }
    };
}
