/*
 * Copyright (c) 2008 Akorri Networks, Inc. All Rights Reserved. This software is provided
 * under license and is not sold by Akorri Networks. This software may be used only in accordance
 * with the terms of an express written license agreement with Akorri Networks, and cannot be used,
 * copied, modified, or in any way distributed without the express permission of Akorri Networks.
 *
 */

package com.netapp.collectors.vmware;

import java.util.ArrayList;
import java.util.LinkedHashMap;

import com.vmware.vim25.HostMountInfo;

/**
 * Collection domain object for Datastores.
 * 
 * @author slaplante
 */
public class VMWareDatastore {

    /** Name of this datastore. */
    private String name;
    
    /** Datastore class name. Used to determine if using vmfs or nas. */
    private String datastoreClassName;
    
    /** Managed object of this datastore. */
    private Object managedObject;
    
    /** Datastore url. */
    private String url;
    
    /** Vmfs datastore name. */
    private String vmfsName;

    /** Vmfs datastore uuid. */
    private String vmfsUuid;
    
    /** Vmfs datastore capacity. */
    private Long vmfsCapacity;
    
    /** Nas datastore name. */
    private String nasName;
    
    /** Nas datastore capacity. */
    private Long nasCapacity;
    
    /** NAS datastore remote host. */
    private String nasRemoteHost;
    
    /** NAS datastore remote path. */
    private String nasRemotePath;
    
    /** List of virtual machines existing on this datastore. */
    private LinkedHashMap<Object, VMWareVirtualMachine> virtualMachines = new LinkedHashMap<Object, VMWareVirtualMachine>();
    
    /** Hosts existing on this datastore. */
    private LinkedHashMap<Object, VMWareHost> hosts = new LinkedHashMap<Object, VMWareHost>();
    
    /** Host mounts existing on this datastore. */
    private LinkedHashMap<Object, HostMountInfo> hostMountInfos = new LinkedHashMap<Object, HostMountInfo>();
    
    /** Data Store info object. */
    private Object dataStoreInfo;
    
    /** This is a Managed Object Reference to the browser of the datastore.*/
    private Object dataStoreBrowserMor;
    
    /** This is the SCSI LUns for the DS. This is a VMWare API object need to cast it to ScsiLun object in VMWare.*/
    private ArrayList<Object> scsiLUNs = new ArrayList<Object>();    
    
    private Long readIOsAvg;

    private Long writeIOsAvg;

    private Long readDataAvg;

    private Long writeDataAvg;

    private Long readLatencyAvg;

    private Long writeLatencyAvg;

    private int numSamplesReadIOs;

    private int numSamplesWriteIOs;
    
    private int numSamplesReadData;
    
    private int numSamplesWriteData;

    private int numSamplesReadLatency;

    private int numSamplesWriteLatency;

    /**
     * Constructor.
     * 
     * @param managedObject managed object of this datastore
     */
    public VMWareDatastore(Object managedObject){
        this.managedObject = managedObject;
    }
    
    /**
     * Gets the name of this datastore.
     * 
     * @return name name of this datastore
     */
    public String getName() {
        return name;
    }

    /**
     * Sets the name of this datastore.
     * 
     * @param name name of this datastore
     */
    public void setName(String name) {
        this.name = name;
    }
    
    /**
     * Gets the datastore class name.
     * 
     * @return datastoreClassName the datastore's class name
     */
    public String getDatastoreClassName() {
        return datastoreClassName;
    }

    /** 
     * Sets the datastore class name.
     * 
     * @param datastoreClassName the datastore's class name
     */
    public void setDatastoreClassName(String datastoreClassName) {
        this.datastoreClassName = datastoreClassName;
    }

    /**
     * Gets the managed object of this datastore.
     * 
     * @return managedObject managed object of this datastore
     */
    public Object getManagedObject(){
        return this.managedObject;
    }
    
    /**
     * Sets the managed object of this datastore.
     * 
     * @param managedObject managed object of this datastore
     */
    public void setManagedObject(Object managedObject){
        this.managedObject = managedObject;
    }
    
    /**
     * Gets datastore url.
     * 
     * @return url datastore url
     */
    public String getUrl() {
        return url;
    }

    /**
     * Sets datastore url.
     * 
     * @param url datastore url
     */
    public void setUrl(String url) {
        this.url = url;
    }

    /**
     * Gets the vmfs datastore name.
     * 
     * @return vmfsName vmfs datastore name
     */
    public String getVmfsName() {
        return vmfsName;
    }

    /**
     * Sets the vmfs datastore name.
     * 
     * @param vmfsName vmfs datastore name
     */
    public void setVmfsName(String vmfsName) {
        this.vmfsName = vmfsName;
    }

    /**
     * Gets the vmfs datastore uuid.
     * 
     * @return vmfsUuid vmfs datastore uuid
     */
    public String getVmfsUuid(){
        return this.vmfsUuid;
    }
    
    /**
     * Sets the vmfs datastore uuid.
     * 
     * @param vmfsUuid vmfs datastore uuid
     */
    public void setVmfsUuid(String vmfsUuid){
        this.vmfsUuid = vmfsUuid;
    }
    
    /**
     * Gets the vmfs datastore capacity.
     * 
     * @return vmfsCapacity vmfs datastore capacity
     */
    public Long getVmfsCapacity() {
        return vmfsCapacity;
    }

    /**
     * Sets the vmfs datastore capacity.
     * 
     * @param vmfsCapacity vmfs datastore capacity
     */
    public void setVmfsCapacity(long vmfsCapacity) {
        this.vmfsCapacity = vmfsCapacity;
    }

    /**
     * Gets nas datastore name.
     * 
     * @return nasName nas datastore name
     */
    public String getNasName() {
        return nasName;
    }

    /**
     * Sets nas datastore name.
     * 
     * @param nasName nas datastore name
     */
    public void setNasName(String nasName) {
        this.nasName = nasName;
    }

    /**
     * Gets nas datastore capacity.
     * 
     * @return nasCapacity nas datastore capacity
     */
    public Long getNasCapacity() {
        return nasCapacity;
    }

    /**
     * Sets nas datastore capacity.
     * 
     * @param nasCapacity nas datastore capacity
     */
    public void setNasCapacity(long nasCapacity) {
        this.nasCapacity = nasCapacity;
    }

    /**
     * Gets the NAS Remote Host.
     * 
     * @return nasRemoteHost NAS remote host
     */
    public String getNasRemoteHost(){
    	return this.nasRemoteHost;
    }

    /**
     * Sets the NAS Remote Host.
     * 
     * @param nasRemoteHost nasRemoteHost NAS remote host
     */
    public void setNasRemoteHost(String nasRemoteHost){
    	this.nasRemoteHost = nasRemoteHost;
    }
    
    /**
     * Gets the NAS Remote Path. EX. /vol/ESX199_Flex_NFS
     * 
     * @return nasRemotePath NAS remote path
     */
    public String getNasRemotePath(){
    	return this.nasRemotePath;
    }

    /**
     * Sets the NAS Remote Path.
     * 
     * @param nasRemotePath nasRemotePath NAS remote path
     */
    public void setNasRemotePath(String nasRemotePath){
    	this.nasRemotePath = nasRemotePath;
    }
    
    /**
     * Gets a list of all virtual machines existing on this datastore.
     * 
     * @return virtualMachines virtual machines existing on this datastore
     */
    public LinkedHashMap<Object, VMWareVirtualMachine> getAllVirtualMachines() {
        return this.virtualMachines;
    }

    /**
     * Gets a virtual machine existing on this datstore.
     * 
     * @param key managed object of this virtual machine
     * @return virtualMachine virtual machine object existing on this datastore
     */
    public VMWareVirtualMachine getVirtualMachine(Object key) {
        return this.virtualMachines.get(key);
    }
    
    /**
     * Sets a virtual machine existing on this datastore.
     * 
     * @param key managed object of the virtual machine
     * @param virtualMachine virtual machine object existing on this datastore
     */
    public void putVirtualMachine(Object key, VMWareVirtualMachine virtualMachine) {
        this.virtualMachines.put(key, virtualMachine);
    }
    
    /**
     * Sets all the virtual machines associated with this datastore.
     * 
     * @param virtualMachines - list of all the virtual machines.
     */
    public void setAllVirtualMachines(LinkedHashMap<Object, VMWareVirtualMachine> virtualMachines){
    	this.virtualMachines = virtualMachines;
    }
    
    /**
     * Gets a list of all hosts existing on this datastore.
     * 
     * @return hosts hosts existing on this datastore
     */
    public LinkedHashMap<Object, VMWareHost> getAllHosts(){
        return this.hosts;
    }
    
    /**
     * Gets a host existing on this datstore.
     * 
     * @param key managed object of this host
     * @return host host object existing on this datastore
     */
    public VMWareHost getHost(Object key){
        return this.hosts.get(key);
    }
    
    /**
     * Sets a host existing on this datastore.
     * 
     * @param key managed object of the virtual machine
     * @param host host object existing on this datastore
     */
    public void putHost(Object key, VMWareHost host){
        this.hosts.put(key, host);
    }

    /**
     * Gets a list of all host mount infos existing on this datastore.
     * 
     * @return hostMountInfos hostMountInfos existing on this datastore
     */
    public LinkedHashMap<Object, HostMountInfo> getAllHostMountInfos(){
        return this.hostMountInfos;
    }
    
    /**
     * Gets a hostMountInfo existing on this datstore.
     * 
     * @param key managed object of this host
     * @return hostMountInfo hostMountInfo object existing on this datastore
     */
    public HostMountInfo getHostMountInfo(Object key){
        return this.hostMountInfos.get(key);
    }
    
    /**
     * Sets a hostMountInfo existing on this datastore.
     * 
     * @param key managed object of the virtual machine
     * @param hostMountInfo hostMountInfo object existing on this datastore
     */
    public void putHostMountInfo(Object key, HostMountInfo hostMountInfo){
        this.hostMountInfos.put(key, hostMountInfo);
    }

    /**
     * DatastoreInfo object of VMWare APU.
     * 
     * @return - DatastoreInfo.
     */
	public Object getDataStoreInfo() {
		return dataStoreInfo;
	}

	/**
	 * Sets the DatastoreInfo object.
	 *  
	 * @param dataStoreInfo - DatastoreInfo.
	 */
	public void setDataStoreInfo(Object dataStoreInfo) {
		this.dataStoreInfo = dataStoreInfo;
	}

	/**
	 * Datastore browser object.
	 * 
	 * @return mor.
	 */
	public Object getDataStoreBrowserMor() {
		return dataStoreBrowserMor;
	}

	/**
	 * Datastore browser object.
	 * 
	 * @param dataStoreBrowserMor - ManagedObjectReference.
	 */
	public void setDataStoreBrowserMor(Object dataStoreBrowserMor) {
		this.dataStoreBrowserMor = dataStoreBrowserMor;
	}

	/**
	 * Returns the SCSI luns for the DS.
	 * 
	 * @return Array of SCSI Luns.
	 */
    public ArrayList<Object> getAllScsiLUNs() {
		return scsiLUNs;
	}
    
    /**
     * Sets the SCSI LUN Array for the DS.
     * 
     * @param scsiLUN - SCSI Luns.
     */
	public void putScsiLUN(Object scsiLUN) {
		this.scsiLUNs.add(scsiLUN);
	}	

    /**
     * Gets the average number of read IOs.
     * 
     * @return the average number of read IOs
     */
    public Long getReadIOsAvg() {
        return this.readIOsAvg;
    }

    /**
     * Sets the average number of read IOs.
     * 
     * @param readIOsAvg the average number of read IOs to set
     */
    public void setReadIOsAvg(Long readIOsAvg) {
        this.readIOsAvg = readIOsAvg;
    }

    /**
     * Gets the average number of write IOs.
     * 
     * @return the average number of write IOs
     */
    public Long getWriteIOsAvg() {
        return this.writeIOsAvg;
    }

    /**
     * Sets the average number of write IOs.
     * 
     * @param writeIOsAvg the average number of write IOs to set
     */
    public void setWriteIOsAvg(Long writeIOsAvg) {
        this.writeIOsAvg = writeIOsAvg;
    }

    /**
     * Gets the average number of bytes read.
     * 
     * @return the average number of bytes read
     */
    public Long getReadDataAvg() {
        return this.readDataAvg;
    }

    /**
     * Sets the average number of bytes read.
     * 
     * @param readDataAvg the average number of bytes read to set
     */
    public void setReadDataAvg(Long readDataAvg) {
        this.readDataAvg = readDataAvg;
    }

    /**
     * Gets the average number of bytes written.
     * 
     * @return the average number of bytes written
     */
    public Long getWriteDataAvg() {
        return this.writeDataAvg;
    }

    /**
     * Sets the average number of bytes written.
     * 
     * @param writeDataAvg the average number of bytes written to set
     */
    public void setWriteDataAvg(Long writeDataAvg) {
        this.writeDataAvg = writeDataAvg;
    }

    /**
     * Gets the average read latency.
     * 
     * @return the average read latency
     */
    public Long getReadLatencyAvg() {
        return this.readLatencyAvg;
    }

    /**
     * Sets the average read latency.
     * 
     * @param readLatencyAvg the average read latency to set
     */
    public void setReadLatencyAvg(Long readLatencyAvg) {
        this.readLatencyAvg = readLatencyAvg;
    }

    /**
     * Gets the average write latency.
     * 
     * @return the average write latency
     */
    public Long getWriteLatencyAvg() {
        return this.writeLatencyAvg;
    }

    /**
     * Sets the write average latency.
     * 
     * @param writeLatencyAvg the average write latency to set
     */
    public void setWriteLatencyAvg(Long writeLatencyAvg) {
        this.writeLatencyAvg = writeLatencyAvg;
    }

    /**
     * Gets the number of read IO samples.
     * 
     * @return the number of read IO samples
     */
    public int getNumSamplesReadIOs() {
        return this.numSamplesReadIOs;
    }

    /**
     * Sets the number of read IO samples.
     * 
     * @param numSamplesReadIOs the number of read IO samples to set
     */
    public void setNumSamplesReadIOs(int numSamplesReadIOs) {
        this.numSamplesReadIOs = numSamplesReadIOs;
    }

    /**
     * Gets the number of write IO samples.
     * 
     * @return the number of write IO samples
     */
    public int getNumSamplesWriteIOs() {
        return this.numSamplesWriteIOs;
    }

    /**
     * Sets the number of write IO samples.
     * 
     * @param numSamplesWriteIOs the number of write IO samples to set
     */
    public void setNumSamplesWriteIOs(int numSamplesWriteIOs) {
        this.numSamplesWriteIOs = numSamplesWriteIOs;
    }

    /**
     * Gets the number of read data samples.
     * 
     * @return the number of read data samples
     */
    public int getNumSamplesReadData() {
        return this.numSamplesReadData;
    }

    /**
     * Sets the number of read data samples.
     * 
     * @param numSamplesReadData the number of read data samples to set
     */
    public void setNumSamplesReadData(int numSamplesReadData) {
        this.numSamplesReadData = numSamplesReadData;
    }

    /**
     * Gets the number of write data samples.
     * 
     * @return the number of write data samples
     */
    public int getNumSamplesWriteData() {
        return this.numSamplesWriteData;
    }

    /**
     * Sets the number of write data samples.
     * 
     * @param numSamplesWriteData the number of write data samples to set
     */
    public void setNumSamplesWriteData(int numSamplesWriteData) {
        this.numSamplesWriteData = numSamplesWriteData;
    }

    /**
     * Gets the number of read latency samples.
     * 
     * @return the number of read latency samples
     */
    public int getNumSamplesReadLatency() {
        return this.numSamplesReadLatency;
    }

    /**
     * Sets the number of read latency samples.
     * 
     * @param numSamplesReadLatency the number of read latency samples to set
     */
    public void setNumSamplesReadLatency(int numSamplesReadLatency) {
        this.numSamplesReadLatency = numSamplesReadLatency;
    }

    /**
     * Gets the number of write latency samples.
     * 
     * @return the number of write latency samples
     */
    public int getNumSamplesWriteLatency() {
        return this.numSamplesWriteLatency;
    }

    /**
     * Sets the number of write latency samples.
     * 
     * @param numSamplesWriteLatency the number of write latency samples to set
     */
    public void setNumSamplesWriteLatency(int numSamplesWriteLatency) {
        this.numSamplesWriteLatency = numSamplesWriteLatency;
    }

}
