package com.netapp.platform.keystoremanager;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.util.Random;

import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;

/**
 * Password Manager - To encrypt a fixed password and store it in
 * keystore file.
 *
 * If keystore is available, it will retrieve the password.
 * If keystore is not available, it will create and retrieve the password.
 *
 * KeyStore is stored in /data/ocie directory in general on linux
 * KeyStore is stored in C:\Program Files\data directory in general on windows
 *
 * Exit status :
 * 2 - Failed   - no permissions
 * 1 - Failed  - incorrect usage
 * 0 - Success
 */
public class KeystoreManager {

    private static final char[] KEYSTORE_PASSWORD = "changeit".toCharArray();
    private static final String KEYSTORE_TYPE = "jceks";
    private static final String SYMBOLS = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz!@$%^&*()_+=";
    private static final int PASSWORD_LEN = 12;
    private static final String PARTY_METHOD = "PBE";
    private static final String FILE_PATH_SEPARATOR = File.separator ;
    private static final String PASSWORD_FILE_NAME = "platform.keystore";

    private static Random rnd = new Random();
    private String keyStoreDir = "";

    public KeystoreManager(String keyStoreDir) {
        setKeystoreDir(keyStoreDir);
    }

    /**
     * Check the given arguments
     *
     * Valid arguments - 'get' 'Absolute path of data directory'
     * e.g - get /data/ocie
     *
     * @param args
     * @return true if valid arguments are passed
     */
    private static boolean checkSyntax(String[] args) {
        if ( (args.length < 3) || (args.length > 4) ) {
            return false;
        } else if ( (!args[1].equalsIgnoreCase("get")) && (!args[1].equalsIgnoreCase("set")) ) {
            return false;
        } else if ( (args[1].equalsIgnoreCase("get")) && (args.length != 3) ) {
            return false;
        } else if ( (args[1].equalsIgnoreCase("set")) && (args.length != 3) && (args.length != 4) ) {
            return false;
        }
        return true;
    }

    private static void printUsage() {
        System.err.println("Usage : <Full path of keystore dir> get <key name>");
        System.err.println(" or ");
        System.err.println("Usage : <Full path of keystore dir> set <key name>");
        System.err.println(" or ");
        System.err.println("Usage : <Full path of keystore dir> set <key name> <key value>");
    }

    private void setKeystoreDir(String keystoreDir) {
        this.keyStoreDir = keystoreDir;
    }

    private String getKeyStoreDir() {
        return this.keyStoreDir;
    }

    private boolean canRunTool() {
        if ( new File(this.getKeyStoreDir()).isDirectory()) {
            return true;
        }
        return false;
    }

    public String get(String keyName) throws Exception
    {
        FileInputStream getKeyStoreFileForReading = null;
        String keyStoreFilePath = getKeyStoreFilePath();
        if((! Files.exists(Paths.get(keyStoreFilePath))) || (! Files.isReadable(Paths.get(keyStoreFilePath))))  {
            return null;
        } else {
            getKeyStoreFileForReading = new FileInputStream(new File(getKeyStoreFilePath()));
        }

        try {
                return retrieveValueFromKeyStore(keyName, getKeyStoreFileForReading);
        } catch (NoSuchAlgorithmException | InvalidKeySpecException | KeyStoreException | CertificateException | IOException | UnrecoverableKeyException passEx) {
            throw new Exception("Unable to retrieve the password", passEx );
        }
    }

    public void set(String keyName, String keyValue) throws Exception
    {
        String keyStoreFilePathStr = getKeyStoreFilePath();
        Path keyStoreFilePath = Paths.get(keyStoreFilePathStr);
        FileInputStream getKeyStoreFileForReading = null;

        if( Files.exists(keyStoreFilePath)) {
            getKeyStoreFileForReading = new FileInputStream(new File(keyStoreFilePathStr));
        }

        storeValueToKeyStore(keyName, keyValue, getKeyStoreFileForReading);
    }

    public void set(String keyName) throws Exception
    {
        set( keyName, getRandomString(PASSWORD_LEN) );
    }

    private static String getRandomString( int len )
    {
       StringBuilder sb = new StringBuilder( len );
       for( int i = 0; i < len; i++ )
          sb.append( SYMBOLS.charAt( rnd.nextInt(SYMBOLS.length()) ) );
       return sb.toString();
    }

    private boolean doesKeyStoreFileExist() {
       return ( new File(this.getKeyStoreDir() + FILE_PATH_SEPARATOR + PASSWORD_FILE_NAME).exists() );
    }

    private void storeValueToKeyStore(String keyName, String keyValue, FileInputStream keyStoreToRead) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, InvalidKeySpecException {

        KeyStore keyStore = KeyStore.getInstance(KEYSTORE_TYPE);

        if( keyStoreToRead != null ) {
            keyStore.load(keyStoreToRead, KEYSTORE_PASSWORD);
        } else {
            keyStore.load(null, KEYSTORE_PASSWORD);
        }

        SecretKey secretKey = createSecretKey(keyValue);
        keyStore.setKeyEntry(keyName, secretKey, KEYSTORE_PASSWORD, null);

        FileOutputStream keyStoreFile = new FileOutputStream(new File(getKeyStoreFilePath()));
        keyStore.store(keyStoreFile, KEYSTORE_PASSWORD);
        keyStoreFile.flush();
        keyStoreFile.close();

    }

    private String retrieveValueFromKeyStore(String keyName, FileInputStream keyStoreFileForReading) throws NoSuchAlgorithmException, InvalidKeySpecException, UnrecoverableKeyException, KeyStoreException, CertificateException, IOException {

        if( ! doesKeyStoreFileExist() )
            return null;

        SecretKeyFactory factory;
        char[] password = null;
        factory = SecretKeyFactory.getInstance(PARTY_METHOD);
        SecretKey retrievedKey = (SecretKey) retrieveKey(keyName, keyStoreFileForReading);

        if (retrievedKey != null) {
            PBEKeySpec keySpec = (PBEKeySpec) factory.getKeySpec(retrievedKey, PBEKeySpec.class);
            password = keySpec.getPassword();
        } else {
            return null;
        }

        return new String(password);
    }

    private Key retrieveKey(String keyName, FileInputStream keyStoreFileForReading) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException,    UnrecoverableKeyException {

        KeyStore keyStore = KeyStore.getInstance(KEYSTORE_TYPE);
        keyStore.load(keyStoreFileForReading, KEYSTORE_PASSWORD);

        return keyStore.getKey(keyName, KEYSTORE_PASSWORD);
    }

    private SecretKey createSecretKey(String keyValue) throws NoSuchAlgorithmException, InvalidKeySpecException {
        SecretKeyFactory factory = SecretKeyFactory.getInstance(PARTY_METHOD);
        return factory.generateSecret(new PBEKeySpec(keyValue.toCharArray()));
    }

    private String getKeyStoreFilePath() {
        return this.getKeyStoreDir() + FILE_PATH_SEPARATOR + PASSWORD_FILE_NAME;
    }

    /**
     * Main
     * @param args
     * @throws Exception
     * First argument is always data directory
     */
    public static void main(String[] args) throws Exception {

        if (!checkSyntax(args)) {
            printUsage();
            System.exit(1);
        }

        KeystoreManager keyStoreManager = new KeystoreManager(args[0]);

        if (!keyStoreManager.canRunTool()) {
            System.exit(2);
        }

        if (args[1].equalsIgnoreCase("get")) {
            System.out.println(keyStoreManager.get(args[2]));
        } else if (args[1].equalsIgnoreCase("set")) {
            if (args.length == 3 ) {
                keyStoreManager.set(args[2]);
            } else {
                keyStoreManager.set(args[2], args[3]);
            }
        }

    }
}
