"use strict";

var _interopRequireWildcard = require("@babel/runtime/helpers/interopRequireWildcard");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.keyboardImplementation = keyboardImplementation;
exports.releaseAllKeys = releaseAllKeys;

var _dom = require("@testing-library/dom");

var _utils = require("../utils");

var _getNextKeyDef = require("./getNextKeyDef");

var plugins = _interopRequireWildcard(require("./plugins"));

var _getEventProps = require("./getEventProps");

async function keyboardImplementation(text, options, state) {
  const {
    document
  } = options;

  const getCurrentElement = () => getActive(document);

  const {
    keyDef,
    consumedLength,
    releasePrevious,
    releaseSelf
  } = (0, _getNextKeyDef.getNextKeyDef)(text, options);
  const replace = applyPlugins(plugins.replaceBehavior, keyDef, getCurrentElement(), options, state);

  if (!replace) {
    const pressed = state.pressed.find(p => p.keyDef === keyDef);

    if (pressed) {
      keyup(keyDef, getCurrentElement, options, state, pressed.unpreventedDefault);
    }

    if (!releasePrevious) {
      const unpreventedDefault = keydown(keyDef, getCurrentElement, options, state);

      if (unpreventedDefault && hasKeyPress(keyDef, state)) {
        keypress(keyDef, getCurrentElement, options, state);
      }

      if (releaseSelf) {
        keyup(keyDef, getCurrentElement, options, state, unpreventedDefault);
      }
    }
  }

  if (text.length > consumedLength) {
    if (options.delay > 0) {
      await (0, _utils.wait)(options.delay);
    }

    return keyboardImplementation(text.slice(consumedLength), options, state);
  }

  return void undefined;
}

function getActive(document) {
  var _getActiveElement;

  return (_getActiveElement = (0, _utils.getActiveElement)(document)) != null ? _getActiveElement :
  /* istanbul ignore next */
  document.body;
}

function releaseAllKeys(options, state) {
  const getCurrentElement = () => getActive(options.document);

  for (const k of state.pressed) {
    keyup(k.keyDef, getCurrentElement, options, state, k.unpreventedDefault);
  }
}

function keydown(keyDef, getCurrentElement, options, state) {
  const element = getCurrentElement(); // clear carried characters when focus is moved

  if (element !== state.activeElement) {
    state.carryValue = undefined;
    state.carryChar = '';
  }

  state.activeElement = element;
  applyPlugins(plugins.preKeydownBehavior, keyDef, element, options, state);

  const unpreventedDefault = _dom.fireEvent.keyDown(element, (0, _getEventProps.getKeyEventProps)(keyDef, state));

  state.pressed.push({
    keyDef,
    unpreventedDefault
  });

  if (unpreventedDefault) {
    // all default behavior like keypress/submit etc is applied to the currentElement
    applyPlugins(plugins.keydownBehavior, keyDef, getCurrentElement(), options, state);
  }

  return unpreventedDefault;
}

function keypress(keyDef, getCurrentElement, options, state) {
  const element = getCurrentElement();

  const unpreventedDefault = _dom.fireEvent.keyPress(element, (0, _getEventProps.getKeyEventProps)(keyDef, state));

  if (unpreventedDefault) {
    applyPlugins(plugins.keypressBehavior, keyDef, getCurrentElement(), options, state);
  }
}

function keyup(keyDef, getCurrentElement, options, state, unprevented) {
  const element = getCurrentElement();
  applyPlugins(plugins.preKeyupBehavior, keyDef, element, options, state);

  const unpreventedDefault = _dom.fireEvent.keyUp(element, (0, _getEventProps.getKeyEventProps)(keyDef, state));

  if (unprevented && unpreventedDefault) {
    applyPlugins(plugins.keyupBehavior, keyDef, getCurrentElement(), options, state);
  }

  state.pressed = state.pressed.filter(k => k.keyDef !== keyDef);
  applyPlugins(plugins.postKeyupBehavior, keyDef, element, options, state);
}

function applyPlugins(pluginCollection, keyDef, element, options, state) {
  const plugin = pluginCollection.find(p => p.matches(keyDef, element, options, state));

  if (plugin) {
    plugin.handle(keyDef, element, options, state);
  }

  return !!plugin;
}

function hasKeyPress(keyDef, state) {
  var _keyDef$key;

  return (((_keyDef$key = keyDef.key) == null ? void 0 : _keyDef$key.length) === 1 || keyDef.key === 'Enter') && !state.modifiers.ctrl && !state.modifiers.alt;
}