import { Chart } from 'react-chartjs-2';

var SUPPORTED_TYPES = {};
['pie', 'doughnut', 'polarArea', 'bar', 'horizontalBar'].forEach(function (t) {
  SUPPORTED_TYPES[t] = true;
});

function Label() {
  this.renderToDataset = this.renderToDataset.bind(this);
}

Label.prototype.setup = function (chart, options) {
  this.chart = chart;
  this.ctx = chart.ctx;
  this.args = {};
  this.barTotal = {};
  var chartOptions = chart.config.options;
  this.options = Object.assign({
    position: 'default',
    precision: 0,
    fontSize: chartOptions.defaultFontSize,
    fontColor: chartOptions.defaultFontColor,
    fontStyle: chartOptions.defaultFontStyle,
    fontFamily: chartOptions.defaultFontFamily,
    shadowOffsetX: 3,
    shadowOffsetY: 3,
    shadowColor: 'rgba(0,0,0,0.3)',
    shadowBlur: 6,
    images: [],
    outsidePadding: 2,
    textMargin: 2,
    overlap: true
  }, options);

  if (chart.config.type === 'horizontalBar') {
    this.options.position = 'default';
    this.options.arc = false;
    this.options.overlap = true;
  }
};

Label.prototype.render = function () {
  this.labelBounds = [];
  this.chart.data.datasets.forEach(this.renderToDataset);
};

Label.prototype.renderToDataset = function (dataset, index) {
  this.totalPercentage = 0;
  this.total = null;
  var arg = this.args[index];
  arg.meta.data.forEach(function (element, index) {
    this.renderToElement(dataset, arg, element, index);
  }.bind(this));
};

Label.prototype.renderToElement = function (dataset, arg, element, index) {
  if (!this.shouldRenderToElement(arg.meta, element)) {
    return;
  }

  this.percentage = null;
  var label = this.getLabel(dataset, element, index);

  if (!label) {
    return;
  }

  var ctx = this.ctx;
  ctx.save();
  ctx.font = Chart.helpers.fontString(this.options.fontSize, this.options.fontStyle, this.options.fontFamily);
  var renderInfo = this.getRenderInfo(element, label);

  if (!this.drawable(element, label, renderInfo)) {
    ctx.restore();
    return;
  }

  ctx.beginPath();
  ctx.fillStyle = this.getFontColor(dataset, element, index);
  this.renderLabel(label, renderInfo, element);
  this.renderLine(label, element, renderInfo);
  ctx.restore();
};

Label.prototype.renderLabel = function (label, renderInfo, element) {
  return this.options.arc ? this.renderArcLabel(label, renderInfo) : this.renderBaseLabel(label, renderInfo, element);
};

Label.prototype.renderLine = function (label, element, renderInfo) {
  var ctx = this.ctx;
  ctx.beginPath();
  ctx.strokeStyle = element._model.backgroundColor;
  var startPoint = element.getCenterPoint();
  var view = element._view;
  var centreAngle = view.startAngle + (view.endAngle - view.startAngle) / 2;
  ctx.moveTo(startPoint.x + Math.cos(centreAngle) * 7, startPoint.y + Math.sin(centreAngle) * 7);
  ctx.lineTo(renderInfo.x, renderInfo.y);

  if (this.chart.config.type === 'doughnut') {
    var metrics = ctx.measureText(label[0].text);

    if (element._model.x > renderInfo.x) {
      ctx.lineTo(renderInfo.x - metrics.width - 5, renderInfo.y);
    } else {
      ctx.lineTo(renderInfo.x + metrics.width + 5, renderInfo.y);
    }
  }

  ctx.stroke();
};

Label.prototype.renderBaseLabel = function (label, position, element) {
  var ctx = this.ctx;
  ctx.save();
  var extraHeight = 0;

  if (this.chart.config.type === 'horizontalBar') {
    ctx.textBaseline = element._datasetIndex % 2 === 1 ? 'bottom' : 'top';
    ctx.textAlign = 'left';
    extraHeight = element._datasetIndex % 2 === 1 ? -5 : 5;
  } else {
    ctx.textBaseline = 'bottom';
    ctx.textAlign = element._model.x > position.x ? 'right' : 'left';
  }

  if (this.options.textShadow) {
    ctx.shadowOffsetX = this.options.shadowOffsetX;
    ctx.shadowOffsetY = this.options.shadowOffsetY;
    ctx.shadowColor = this.options.shadowColor;
    ctx.shadowBlur = this.options.shadowBlur;
  }

  var lines = label;

  for (var i = 0; i < lines.length; i++) {
    var y = position.y - this.options.fontSize / 2 * lines.length + (this.options.fontSize + 5) * i;

    if (lines[i].fontWeight) {
      ctx.font = Chart.helpers.fontString(this.options.fontSize, lines[i].fontWeight, this.options.fontFamily);
    }

    ctx.fillText(lines[i].text, position.x + (ctx.textAlign === 'right' ? -3 : +3), y + 5 + extraHeight);
  }

  ctx.restore();
};

Label.prototype.renderArcLabel = function (label, renderInfo) {};

Label.prototype.shouldRenderToElement = function (meta, element) {
  return !meta.hidden && !element.hidden && (this.options.showZero || this.chart.config.type === 'polarArea' ? element._view.outerRadius !== 0 : element._view.circumference !== 0);
};

Label.prototype.getLabel = function (dataset, element, index) {
  var label;

  if (typeof this.options.render === 'function') {
    label = this.options.render({
      label: this.chart.config.data.labels[index],
      value: dataset.data[index],
      percentage: this.getPercentage(dataset, element, index),
      dataset: dataset,
      index: index
    });
  } else {
    switch (this.options.render) {
      case 'value':
        label = dataset.data[index];
        break;

      case 'label':
        label = this.chart.config.data.labels[index];
        break;

      case 'image':
        label = this.options.images[index] ? this.loadImage(this.options.images[index]) : '';
        break;

      case 'percentage':
      default:
        label = this.getPercentage(dataset, element, index) + '%';
        break;
    }
  }

  return label;
};

Label.prototype.getFontColor = function (dataset, element, index) {
  var fontColor = this.options.fontColor;

  if (typeof fontColor === 'function') {
    fontColor = fontColor({
      label: this.chart.config.data.labels[index],
      value: dataset.data[index],
      percentage: this.getPercentage(dataset, element, index),
      backgroundColor: dataset.backgroundColor[index],
      dataset: dataset,
      index: index
    });
  } else if (typeof fontColor !== 'string') {
    fontColor = fontColor[index] || this.chart.config.options.defaultFontColor;
  }

  return fontColor;
};

Label.prototype.getPercentage = function (dataset, element, index) {
  if (this.percentage !== null) {
    return this.percentage;
  }

  var percentage;

  if (this.chart.config.type === 'polarArea') {
    if (this.total === null) {
      this.total = 0;

      for (var i = 0; i < dataset.data.length; ++i) {
        this.total += dataset.data[i];
      }
    }

    percentage = dataset.data[index] / this.total * 100;
  } else if (this.chart.config.type === 'horizontalBar') {
    if (this.barTotal[index] === undefined) {
      this.barTotal[index] = 0;

      for (var _i = 0; _i < this.chart.data.datasets.length; ++_i) {
        this.barTotal[index] += this.chart.data.datasets[_i].data[index];
      }
    }

    percentage = dataset.data[index] / this.barTotal[index] * 100;
  } else {
    percentage = element._view.circumference / this.chart.config.options.circumference * 100;
  }

  percentage = parseFloat(percentage.toFixed(this.options.precision));

  if (!this.options.showActualPercentages) {
    if (this.chart.config.type === 'horizontalBar') {
      this.totalPercentage = this.barTotalPercentage[index] || 0;
    }

    this.totalPercentage += percentage;

    if (this.totalPercentage > 100) {
      percentage -= this.totalPercentage - 100;
      percentage = parseFloat(percentage.toFixed(this.options.precision));
    }

    if (this.chart.config.type === 'horizontalBar') {
      this.barTotalPercentage[index] = this.totalPercentage;
    }
  }

  this.percentage = percentage;
  return percentage;
};

Label.prototype.getRenderInfo = function (element, label) {
  if (this.chart.config.type === 'horizontalBar') {
    return this.getBarRenderInfo(element, label);
  } else {
    return this.options.arc ? this.getArcRenderInfo(element, label) : this.getBaseRenderInfo(element, label);
  }
};

Label.prototype.getBaseRenderInfo = function (element, label) {
  if (this.options.position === 'outside' || this.options.position === 'border') {
    var renderInfo;
    var rangeFromCentre;
    var view = element._view;
    var centreAngle = view.startAngle + (view.endAngle - view.startAngle) / 2;
    var innerRadius = view.outerRadius / 2;

    if (this.options.position === 'border') {
      rangeFromCentre = (view.outerRadius - innerRadius) / 2 + innerRadius;
    } else if (this.options.position === 'outside') {
      rangeFromCentre = view.outerRadius - innerRadius + innerRadius + this.options.textMargin;
    }

    renderInfo = {
      x: view.x + Math.cos(centreAngle) * rangeFromCentre,
      y: view.y + Math.sin(centreAngle) * rangeFromCentre
    };

    if (renderInfo.y >= view.y) {
      renderInfo.y += this.options.fontSize + 3;
    }

    if (renderInfo.y < view.y) {
      renderInfo.y -= 3;
    }

    if (view.endAngle - view.startAngle < 0.5) {
      if (element._model.x > renderInfo.x) {
        renderInfo.x -= 10;
      } else {
        renderInfo.x += 10;
      }
    }

    return renderInfo;
  } else {
    return element.tooltipPosition();
  }
};

Label.prototype.getArcRenderInfo = function (element, label) {
  var radius;
  var view = element._view;

  if (this.options.position === 'outside') {
    radius = view.outerRadius + this.options.fontSize + this.options.textMargin;
  } else if (this.options.position === 'border') {
    radius = (view.outerRadius / 2 + view.outerRadius) / 2;
  } else {
    radius = (view.innerRadius + view.outerRadius) / 2;
  }

  var startAngle = view.startAngle;
  var endAngle = view.endAngle;
  var totalAngle = endAngle - startAngle;
  startAngle += Math.PI / 2;
  endAngle += Math.PI / 2;
  var mertrics = this.measureLabel(label);
  startAngle += (endAngle - (mertrics.width / radius + startAngle)) / 2;
  return {
    radius: radius,
    startAngle: startAngle,
    endAngle: endAngle,
    totalAngle: totalAngle,
    view: view
  };
};

Label.prototype.getBarRenderInfo = function (element, label) {
  var renderInfo = element.getCenterPoint();

  if (element._datasetIndex % 2 === 1) {
    renderInfo.y -= this.measureLabel(label).height / 2 + this.options.textMargin;
  } else {
    renderInfo.y += this.measureLabel(label).height / 2 + this.options.textMargin;
  }

  return renderInfo;
};

Label.prototype.drawable = function (element, label, renderInfo) {
  if (this.options.overlap) {
    return true;
  } else if (this.options.arc) {
    return renderInfo.endAngle - renderInfo.startAngle <= renderInfo.totalAngle;
  } else {
    var mertrics = this.measureLabel(label);
    var left = renderInfo.x - mertrics.width / 2;
    var right = renderInfo.x + mertrics.width / 2;
    var top = renderInfo.y - mertrics.height / 2;
    var bottom = renderInfo.y + mertrics.height / 2;

    if (this.options.renderInfo === 'outside') {
      return this.outsideInRange(left, right, top, bottom);
    } else {
      return element.inRange(left, top) && element.inRange(left, bottom) && element.inRange(right, top) && element.inRange(right, bottom);
    }
  }
};

Label.prototype.outsideInRange = function (left, right, top, bottom) {
  var labelBounds = this.labelBounds;

  for (var i = 0; i < labelBounds.length; ++i) {
    var bound = labelBounds[i];
    var potins = [[left, top], [left, bottom], [right, top], [right, bottom]];

    for (var j = 0; j < potins.length; ++j) {
      var x = potins[j][0];
      var y = potins[j][1];

      if (x >= bound.left && x <= bound.right && y >= bound.top && y <= bound.bottom) {
        return false;
      }
    }

    potins = [[bound.left, bound.top], [bound.left, bound.bottom], [bound.right, bound.top], [bound.right, bound.bottom]];

    for (var _j = 0; _j < potins.length; ++_j) {
      var _x = potins[_j][0];
      var _y = potins[_j][1];

      if (_x >= left && _x <= right && _y >= top && _y <= bottom) {
        return false;
      }
    }
  }

  labelBounds.push({
    left: left,
    right: right,
    top: top,
    bottom: bottom
  });
  return true;
};

Label.prototype.measureLabel = function (label) {
  var width = 0;
  var lines = label;

  for (var i = 0; i < lines.length; ++i) {
    var result = this.ctx.measureText(lines[i].text);

    if (result.width > width) {
      width = result.width;
    }
  }

  return {
    width: width,
    height: this.options.fontSize * lines.length
  };
};

Label.prototype.loadImage = function (obj) {
  var image = new Image();
  image.src = obj.src;
  image.width = obj.width;
  image.height = obj.height;
  return image;
};

var useOuterLabelPlugin = function useOuterLabelPlugin() {
  return {
    id: 'labels',
    beforeDatasetsUpdate: function beforeDatasetsUpdate(chart, options) {
      if (!SUPPORTED_TYPES[chart.config.type]) {
        return;
      }

      if (!Array.isArray(options)) {
        options = [options];
      }

      var count = options.length;

      if (!chart._labels || count !== chart._labels.length) {
        chart._labels = options.map(function () {
          return new Label();
        });
      }

      var someOutside = false;
      var maxPadding = 0;

      for (var i = 0; i < count; ++i) {
        var label = chart._labels[i];
        label.setup(chart, options[i]);

        if (label.options.position === 'outside') {
          someOutside = true;
          var padding = label.options.fontSize * 1.5 + label.options.outsidePadding;

          if (padding > maxPadding) {
            maxPadding = padding;
          }
        }
      }

      if (someOutside) {
        chart.chartArea.top += maxPadding;
        chart.chartArea.bottom -= maxPadding;
      }
    },
    afterDatasetUpdate: function afterDatasetUpdate(chart, args, options) {
      if (!SUPPORTED_TYPES[chart.config.type]) {
        return;
      }

      chart._labels.forEach(function (label) {
        label.args[args.index] = args;
      });
    },
    beforeDraw: function beforeDraw(chart) {
      if (!SUPPORTED_TYPES[chart.config.type]) {
        return;
      }

      chart._labels && chart._labels.forEach(function (label) {
        label.barTotalPercentage = {};
      });
    },
    afterDatasetsDraw: function afterDatasetsDraw(chart) {
      if (!SUPPORTED_TYPES[chart.config.type]) {
        return;
      }

      chart._labels && chart._labels.forEach(function (label) {
        label.render();
      });
    }
  };
};

export default useOuterLabelPlugin;
//# sourceMappingURL=OuterLabelPlugin.js.map
