<#
.SYNOPSIS
Split the test dependencies into different project files.

.DESCRIPTION
.NET pipelines will run centain number of test dependencies in each entry of job matrix. 
The Script is to split all test dependencies into difference sets for later use. 
The script is able to test locally using Pester. Please run tests as following every time you made changes to the script:

```
Invoke-Pester -Output Detailed $PSScriptRoot\tests\Generate-Dependency-Test-References-Tests.ps1
```

.PARAMETER ProjectListFilePath
The file path contains all test dependencies. E.g. The file generated by 'eng/service.proj' in dependecny.tests.yml `Find all test dependencies` step

.PARAMETER ProjectFilesOutputFolder
The folder for all test dependency split files e.g. `Project_1.props` which are ready to publish. E.g. $(Build.ArtifactStagingDirectory) 

.PARAMETER NumOfTestProjectsPerJob
The number of test run on each job.

.PARAMETER MatrixConfigsFile
The file path of the matrix job configuration. E.g. "/eng/pipelines/templates/stages/platform-matrix.json"

.PARAMETER ProjectFileConfigName
The config name of the project file name. Hardcoded in "\eng\pipelines\templates\jobs\ci.yml" `PreGenerationSteps` for matix generation. E.g. 'ProjectListOverrideFile' 

.PARAMETER ExcludeTargetTestProjects
The flag to indicate whether exclude target test package or not. 

.PARAMETER ServiceDirectoryToExclude
The service directory which needs to exclude. E.g. "core"

.EXAMPLE
Run script with default parameters.

Generate-Dependency-Test-References.ps1 -ProjectFilesOutputFolder "$(Build.ArtifactStagingDirectory)/projects" -ProjectListFilePath "eng/projects.txt" `
-MatrixConfigsFile "eng/pipelines/templates/stages/platform-matrix.json" -ProjectFileConfigName "ProjectListOverrideFile" -ExcludeTargetTestProjects $true `
-ServiceDirectoryToExclude "core"

#>
Param (
    [ValidateNotNullOrEmpty()]
    [string] $ProjectListFilePath,
    [string] $ProjectFilesOutputFolder="./",
    [int] $NumOfTestProjectsPerJob = 20,
    [ValidateNotNullOrEmpty()]
    [string] $MatrixConfigsFile,
    [ValidateNotNullOrEmpty()]
    [string] $ProjectFileConfigName,
    [boolean] $ExcludeTargetTestProjects=$false,
    [string] $ServiceDirectoryToExclude
)

. (Join-Path $PSScriptRoot generate-dependency-functions.ps1)
Set-StrictMode -Version 3

$projGroups = Split-Project-File-To-Groups `
                -ProjectFile $ProjectListFilePath `
                -NumberOfTestsPerJob $NumOfTestProjectsPerJob `
                -ExcludeService $ExcludeTargetTestProjects `
                -ServiceDirToExclude $ServiceDirectoryToExclude

$projFiles = Write-Test-Dependency-Group-To-Files `
                -ProjectFileConfigName $ProjectFileConfigName `
                -ProjectGroups $projGroups `
                -MatrixOutputFolder $ProjectFilesOutputFolder

Write-Project-Files-To-Matrix `
  -ProjectFiles $projFiles `
  -MatrixJsonPath $MatrixConfigsFile `
  -MatrixOutputFolder $ProjectFilesOutputFolder `
  -ProjectFileConfigName $ProjectFileConfigName