﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;

namespace Microsoft.Extensions.AI;

/// <summary>Represents a chat completion client.</summary>
/// <remarks>
/// <para>
/// Unless otherwise specified, all members of <see cref="IChatClient"/> are thread-safe for concurrent use.
/// It is expected that all implementations of <see cref="IChatClient"/> support being used by multiple requests concurrently.
/// </para>
/// <para>
/// However, implementations of <see cref="IChatClient"/> may mutate the arguments supplied to <see cref="CompleteAsync"/> and
/// <see cref="CompleteStreamingAsync"/>, such as by adding additional messages to the messages list or configuring the options
/// instance. Thus, consumers of the interface either should avoid using shared instances of these arguments for concurrent
/// invocations or should otherwise ensure by construction that no <see cref="IChatClient"/> instances are used which might employ
/// such mutation. For example, the WithChatOptions method be provided with a callback that could mutate the supplied options
/// argument, and that should be avoided if using a singleton options instance.
/// </para>
/// </remarks>
public interface IChatClient : IDisposable
{
    /// <summary>Sends chat messages to the model and returns the response messages.</summary>
    /// <param name="chatMessages">The chat content to send.</param>
    /// <param name="options">The chat options to configure the request.</param>
    /// <param name="cancellationToken">The <see cref="CancellationToken"/> to monitor for cancellation requests. The default is <see cref="CancellationToken.None"/>.</param>
    /// <returns>The response messages generated by the client.</returns>
    /// <remarks>
    /// The returned messages will not have been added to <paramref name="chatMessages"/>. However, any intermediate messages generated implicitly
    /// by the client, including any messages for roundtrips to the model as part of the implementation of this request, will be included.
    /// </remarks>
    Task<ChatCompletion> CompleteAsync(
        IList<ChatMessage> chatMessages,
        ChatOptions? options = null,
        CancellationToken cancellationToken = default);

    /// <summary>Sends chat messages to the model and streams the response messages.</summary>
    /// <param name="chatMessages">The chat content to send.</param>
    /// <param name="options">The chat options to configure the request.</param>
    /// <param name="cancellationToken">The <see cref="CancellationToken"/> to monitor for cancellation requests. The default is <see cref="CancellationToken.None"/>.</param>
    /// <returns>The response messages generated by the client.</returns>
    /// <remarks>
    /// The returned messages will not have been added to <paramref name="chatMessages"/>. However, any intermediate messages generated implicitly
    /// by the client, including any messages for roundtrips to the model as part of the implementation of this request, will be included.
    /// </remarks>
    IAsyncEnumerable<StreamingChatCompletionUpdate> CompleteStreamingAsync(
        IList<ChatMessage> chatMessages,
        ChatOptions? options = null,
        CancellationToken cancellationToken = default);

    /// <summary>Gets metadata that describes the <see cref="IChatClient"/>.</summary>
    ChatClientMetadata Metadata { get; }

    /// <summary>Asks the <see cref="IChatClient"/> for an object of type <typeparamref name="TService"/>.</summary>
    /// <typeparam name="TService">The type of the object to be retrieved.</typeparam>
    /// <param name="key">An optional key that may be used to help identify the target service.</param>
    /// <returns>The found object, otherwise <see langword="null"/>.</returns>
    /// <remarks>
    /// The purpose of this method is to allow for the retrieval of strongly-typed services that may be provided by the <see cref="IChatClient"/>,
    /// including itself or any services it might be wrapping.
    /// </remarks>
    TService? GetService<TService>(object? key = null)
        where TService : class;
}
