require 'fluent/plugin/filter'
require 'net/http'
require 'json'

module Fluent
  module Plugin
    class DimensionHydrator < Filter
      Fluent::Plugin.register_filter('dimension_hydrator', self)

      config_param :storage_endpoint, :string, :default => 'http://cloud-volumes-infrastructure.default.svc.cluster.local/'
      config_param :storage_api_key, :string, :default => nil, secret: true
      config_param :storage_secret_key, :string, :default => nil, secret: true
      config_param :backoff_retries, :integer, :default => 3
      config_param :update_interval, :integer, :default => 3600

      def configure(conf)
        super
        begin
          $log.info 'init hydrator'
          @nodes = {}
          @hosts_UUID = {}
          @hosts_externalUUID = {}
          @stamps = {}
          @clusters = {}
          @zones = {}
          @regions = {}
          @my_mutex = Mutex.new
          populate_resources()
        rescue => e
          $log.info e.to_s
        end
      end

      def synchronize(&block)
        @my_mutex.synchronize(&block)
      end

      def make_request_to_uri(uri, backoff = nil)
        backoff = backoff ||= Backoff.new

        request = Net::HTTP::Get.new(uri)
        request['accept'] = 'application/json'
        request['content-type'] = 'application/json'
        request['api-key'] = @storage_api_key
        request['secret-key'] = @storage_secret_key
        begin
          res = Net::HTTP.start(uri.hostname, uri.port) { |http| http.request(request) }
          if res.code == "429"
            if backoff.count > @backoff_retries
              raise  Net::HTTPServerException.new("Too many requests", res)
            end 
            backoff.increment_and_wait
            puts "Too many requests to #{uri}"
            make_request_to_uri(uri, backoff)
          end
        rescue Errno::ECONNREFUSED, Net::OpenTimeout, Net::ReadTimeout => e
          if backoff.count > @backoff_retries
              backoff.reset
              puts "Failed too many times requesting uri #{uri}, error #{e}"
              raise e
          end
          puts "Problem with call to API #{e}."
          backoff.increment_and_wait
          puts "Retrying..."
          make_request_to_uri(uri, backoff)
        end

        backoff.reset
        res.body
      end

      def make_request(resource_uri)
        begin
          make_request_to_uri(URI.join(@storage_endpoint, resource_uri))
        rescue => e
          $log.error("Failed to make request to uri: #{resource_uri.to_s} error: #{e.to_s}")
          '[]'
        end
      end

      def map_resource_uuids(resource_uri, destination_map, mapping_field='uuid')
        resp = ''
        begin
          resp = make_request(resource_uri)
          JSON.parse(resp).each { |resource_json| destination_map[resource_json[mapping_field]] = resource_json }
        rescue => e
          $log.error("Failed parsing response: #{resp}\n#{e.to_s}")
        end
      end

      def populate_resources()
        populate_nodes()
        populate_hosts_externalUUID()
        populate_hosts_UUID()
        populate_pods()
        populate_clusters()
        populate_zones()
        populate_regions()
        @last_updated = Time.now.to_i
      end

      def populate_regions()
        $log.info('Populating regions...')
        map_resource_uuids('v1/region', @regions)
      end

      def populate_zones()
        $log.info('Populating zones...')
        map_resource_uuids('v1/zone', @zones)
      end

      def populate_clusters()
        $log.info('Populating clusters...')
        map_resource_uuids('v1/cluster', @clusters)
      end

      def populate_pods()
        $log.info('Populating pods...')
        map_resource_uuids('v1/stamp', @stamps)
      end

      def populate_hosts_externalUUID()
        $log.info('Populating hosts with external_UUIDs...')
        map_resource_uuids('v1/host?excludeState=true', @hosts_externalUUID, 'externalUUID')
      end

      def populate_hosts_UUID()
        $log.info('Populating hosts with UUIDs...')
        map_resource_uuids('v1/host?excludeState=true', @hosts_UUID)
      end

      def populate_nodes()
        $log.info('Populating nodes...')
        map_resource_uuids('v1/node', @nodes, 'name')
      end

      # Retrieves the resource (host, stamp, cluster, zone, or region) of specified uuid and type.
      # Resources are obtained via a list-all call to nfsaas-storage, and cached in corresponding hashmaps,
      # with resource uuid as key.
      # get_resources will refresh the caches if the specified resource is not present, or if at least a minute
      # has expired since the last update. 
      def get_resource(resource_uuid, resources_to_search, resource_type, mapping_field='uuid')
        synchronize do
          if !resources_to_search.key?(resource_uuid) || (Time.now.to_i - @last_updated > @update_interval)
            populate_resources()
          end
          if !resources_to_search.key?(resource_uuid)
            $log.warn("Could not find resource of type: #{resource_type} uuid: #{resource_uuid}")
          end
        end
        resources_to_search[resource_uuid]
      end

      def get_node(node_name)
        get_resource(node_name, @nodes, 'node')
      end

      def get_host_by_internal_uuid(host_uuid)
        get_resource(host_uuid, @hosts_UUID, 'host')
      end

      def get_host_by_external_uuid(external_uuid)
        get_resource(external_uuid, @hosts_externalUUID, 'host')
      end

      def get_stamp(stamp_uuid)
        get_resource(stamp_uuid, @stamps, 'stamp')
      end

      def get_cluster(cluster_uuid)
        get_resource(cluster_uuid, @clusters, 'cluster')
      end

      def get_zone(zone_uuid)
        get_resource(zone_uuid, @zones, 'zone')
      end

      def get_region(region_uuid)
        get_resource(region_uuid, @regions, 'region')
      end

      def get_vendor_ids_for_host(host_uuid, record, externalUUID=true)
        if externalUUID
          host = get_host_by_external_uuid(host_uuid)
        else
          host = get_host_by_internal_uuid(host_uuid)
        end

        if !host.nil?
          record['AzureHostId'] = host['vendorID']
          record['sdeHostName'] = host['name']
          stamp = get_stamp(host['stampUUID'])
          if !stamp.nil?
            record['AzureRackId'] = stamp['vendorID']
            cluster = get_cluster(stamp['clusterUUID'])
            if !cluster.nil?
              record['AzureClusterId'] = cluster['vendorID']
              zone = get_zone(cluster['zoneUUID'])
              if !zone.nil?
                record['AzureDatacenter'] = zone['vendorID']
                region = get_region(zone['regionUUID'])
                if !region.nil?
                  record['AzureRegion'] = region['name']
                end
              end
            end
          end
        end
      end

      def process_ems_record(record)
        host_uuid = record['message'].dig('netapp', 'ems_message_origin', 'cluster_uuid')
        if host_uuid.nil?
          record
        else
          get_vendor_ids_for_host(host_uuid, record)
          record
        end
      end

      def process_audit_log_record(record)
        node_name = record['ident']
        if node_name.nil?
          record
        else
          node = get_node(node_name)
          if !node.nil?
            get_vendor_ids_for_host(node['hostUUID'], record, false)
          end
          record
        end
      end

      def filter(_tag, _time, record)
        if _tag.start_with? 'kubernetes.ontap-ems-event'
          process_ems_record(record)
        end
        if _tag.start_with? 'kubernetes.ontap-auditlogs'
          process_audit_log_record(record)
        end
        record
      end

      class Backoff
            def initialize
              @count = 0
            end

            def count
              @count
            end

            def reset
              @count = 0
            end

            def increment_and_wait()
              @count += 1
              wait_timer = ((2 ** @count) * rand(0.2...0.4))
              puts "Backoff reached, waiting for #{wait_timer}..."
              sleep(wait_timer)
            end

      end
    end
  end
end
