require 'fluent/plugin/filter'
require 'json'
require 'socket'

module Fluent
  module Plugin
    # Filter plugin that extracts health metrics from OnTap xml
    class EmsMetricsFilter < Filter
      Fluent::Plugin.register_filter('ems_metric', self)

      config_param :host, :string, :default => nil
      config_param :port, :integer, :default => 8125
      config_param :namespace, :string, :default => 'Demo'

      def configure(conf)
        super
        start_tcpsocket
      end

      def shutdown
        if @tcpsocket != nil
          @tcpsocket.close
        end
      end

      def filter(_tag, _time, record)
        dims = {}
        dims['ontapHostUuid'] = record['message'].dig(
          'netapp', 'ems_message_origin', 'cluster_uuid'
        )
        dims['ontapNodeUuid'] = record['message'].dig(
          'netapp', 'ems_message_origin', 'node_uuid'
        )
        dims['ontapNodeName'] = record['message'].dig('netapp', 'ems_message_info', 'node')
        dims['MessageName'] = record['message'].dig(
          'netapp', 'ems_message_info', 'message_name'
        )
        dims['Source'] = record['message'].dig('netapp', 'ems_message_info', 'source')
        dims['Severity'] = record['message'].dig(
          'netapp', 'ems_message_info', 'severity'
        )
        if record.key?('datacenter')
          dims['DataCenter'] = record['datacenter']
        else
          dims['DataCenter'] = 'attn: Datacenter Variable not set'
          $log.warn 'Warning: Datacenter environment variable not set'
        end
        dims['AzureRegion'] = record['AzureRegion']
        dims['AzureDatacenter'] = record['AzureDatacenter']
        dims['AzureClusterId'] = record['AzureClusterId']
        dims['AzureRackId'] = record['AzureRackId']
        dims['AzureHostId'] = record['AzureHostId']
        dims['sdeHostName'] = record['sdeHostName']

        # Legacy stuff
        dims['Node'] = record['message'].dig('netapp', 'ems_message_info', 'node')
        dims['NodeUuid'] = record['message'].dig('netapp', 'ems_message_origin', 'node_uuid')
        dims['ClusterUuid'] = record['message'].dig('netapp', 'ems_message_origin', 'cluster_uuid')

        # Cleanup any nil dimensions
        dims.delete_if { |k, v| v.nil? }

        statsd_message = {}
        statsd_message['Namespace'] = @namespace
        statsd_message['Metric'] = 'EmsEvent'
        statsd_message['Dims'] = dims
        record = statsd_message.to_json
        statsd_payload = "#{record}:1|g\n"

        if @tcpsocket == nil
          start_tcpsocket
        end

        if @tcpsocket != nil
          begin
            @tcpsocket.write statsd_payload
          rescue => e
            $log.error "Error sending EMS event: #{e.message} (#{e.class})"
            @tcpsocket.close
            @tcpsocket = nil
          end
        end

        $log.info statsd_message
        statsd_message
      end

      def start_tcpsocket
        begin
          $log.info "Opening a TCP connection: #{@host}:#{@port}"
          @tcpsocket = TCPSocket.new(@host,@port)
          @tcpsocket.setsockopt(Socket::SOL_SOCKET, Socket::SO_KEEPALIVE, true)
          @tcpsocket.setsockopt(Socket::SOL_TCP, Socket::TCP_KEEPIDLE, 60)
          @tcpsocket.setsockopt(Socket::SOL_TCP, Socket::TCP_KEEPINTVL, 20)
          @tcpsocket.setsockopt(Socket::SOL_TCP, Socket::TCP_KEEPCNT, 4)
        rescue => e
          $log.error "Can't connect to geneva-metrics service, is the service running and open on #{@host}:#{@port}? Error: #{e.message} (#{e.class})"
          @tcpsocket = nil
        end
      end

    end
  end
end
