require 'helper'
require 'fluent/test/driver/filter'
require 'fluent/plugin/filter_dimension_hydrator'
require 'webmock'

WebMock.disable_net_connect!

class DimensionHydratorFilterTest < Test::Unit::TestCase

  CONFIG = %(
    type dimension_hydrator
  )

  def create_driver(conf = '', &block)
    f = Fluent::Test::Driver::Filter.new(Fluent::Plugin::DimensionHydrator)
    mock_method f.instance, :make_request_to_uri, &block
    f.configure(conf, syntax: :v1)
  end

  def mock_method(instance, method_name, &block)
    class << instance
      attr_accessor :nodes, :hosts_UUID, :hosts_externalUUID, :stamps, :clusters, :zones, :regions
      self
    end
      .class_eval do
        define_method method_name, &block
      end
  end

  def healthy_storage_behavior
    return Proc.new do |uri|
      resource = uri.path.split('/')[-1]
      if resource == 'node' then return $node_response end
      if resource == 'host' then return $host_response end
      if resource == 'stamp' then return $stamp_response end
      if resource == 'cluster' then return $cluster_response end
      if resource == 'zone' then return $zone_response end
      if resource == 'region' then return $region_response end
      return '[]'
    end
  end

  def check_complete_dimensions(filtered_message, host, rack, cluster, zone, region, sde_host_name)
    assert_equal region, filtered_message['AzureRegion']
    assert_equal zone, filtered_message['AzureDatacenter']
    assert_equal cluster, filtered_message['AzureClusterId']
    assert_equal rack, filtered_message['AzureRackId']
    assert_equal host, filtered_message['AzureHostId']
    assert_equal sde_host_name, filtered_message['sdeHostName']
  end

  # Checks that the correct fields get added to the record when all the data is correctly loaded
  # From the start and the incoming record contains the cluster uuid
  def test_add_ems_dimensions_in_optimal_conditions
    d = create_driver CONFIG, &healthy_storage_behavior
    message = $sample_message.clone
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message)
    check_complete_dimensions(filtered_message, 'host1vendorid', 'otip14rack', 'otip14cluster1', 'otip14datacenter', 'otip14region1', 'sdeHost1Name')
    message = $sample_message2.clone
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message)
    check_complete_dimensions(filtered_message, 'host2vendorid', 'otip14rack2', 'otip14cluster2', 'otip14datacenter2', 'otip14region2', 'sdeHost2Name')
  end

  def test_add_ems_dimensions_in_optimal_conditions
    d = create_driver CONFIG, &healthy_storage_behavior
    message = $sample_message.clone
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message)
    check_complete_dimensions(filtered_message, 'host1vendorid', 'otip14rack', 'otip14cluster1', 'otip14datacenter', 'otip14region1', 'sdeHost1Name')
    message = $sample_message2.clone
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message)
    check_complete_dimensions(filtered_message, 'host2vendorid', 'otip14rack2', 'otip14cluster2', 'otip14datacenter2', 'otip14region2', 'sdeHost2Name')
  end

  # Checks that record stays unmodified if the message does not contain the cluster uuid
  def test_add_ems_dimensions_cluster_uuid_not_present
    d = create_driver CONFIG, &healthy_storage_behavior
    expected_message = Marshal.load( Marshal.dump($sample_message))
    expected_message['message']['netapp']['ems_message_origin'].delete('cluster_uuid')
    message_to_filter = Marshal.load( Marshal.dump(expected_message))
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message_to_filter)
    assert_equal expected_message, filtered_message
  end

  # Check that the record stays unmodified if storage returns invalid json
  def test_add_dimensions_invalid_json_from_storage
    d = create_driver CONFIG do |uri|
      return 'not json'
    end
    expected_message = Marshal.load( Marshal.dump($sample_message))
    message_to_filter = Marshal.load( Marshal.dump(expected_message))
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message_to_filter)
    assert_equal expected_message, filtered_message
  end

  def test_too_many_request
    stub_request(:any, /cloud-volumes-infrastructure.default.svc.cluster.local/).
    with(
      headers: {
  	  'User-Agent'=>'Ruby'
      }).to_return(status: 429)

    f = Fluent::Test::Driver::Filter.new(Fluent::Plugin::DimensionHydrator)

    uri = URI('http://cloud-volumes-infrastructure.default.svc.cluster.local/')

    assert_raise Net::HTTPServerException do 
      f.instance.make_request_to_uri(uri)
    end
  end

  def test_backoff_count
    stub_request(:any, /cloud-volumes-infrastructure.default.svc.cluster.local/).
    with(
      headers: {
  	  'User-Agent'=>'Ruby'
      }).to_return(status: 429)

    f = Fluent::Test::Driver::Filter.new(Fluent::Plugin::DimensionHydrator)

    uri = URI('http://cloud-volumes-infrastructure.default.svc.cluster.local/')
    backoff = Fluent::Plugin::DimensionHydrator::Backoff.new()

    assert_raise Net::HTTPServerException do 
      f.instance.make_request_to_uri(uri, backoff)
    end

    assert_equal backoff.count, f.instance.backoff_retries + 1
  end

  def test_read_timeout
    stub_request(:any, /cloud-volumes-infrastructure.default.svc.cluster.local/).
    with(
      headers: {
  	  'User-Agent'=>'Ruby'
      }).to_raise(Net::ReadTimeout)

    f = Fluent::Test::Driver::Filter.new(Fluent::Plugin::DimensionHydrator)

    uri = URI('http://cloud-volumes-infrastructure.default.svc.cluster.local/')

    assert_raise Net::ReadTimeout do 
     f.instance.make_request_to_uri(uri)
    end
  end

  def test_connection_timeout
    stub_request(:any, /cloud-volumes-infrastructure.default.svc.cluster.local/).
    with(
      headers: {
  	  'User-Agent'=>'Ruby'
      }).
    to_timeout

    f = Fluent::Test::Driver::Filter.new(Fluent::Plugin::DimensionHydrator)

    uri = URI('http://cloud-volumes-infrastructure.default.svc.cluster.local/')

    assert_raise Net::OpenTimeout do 
     f.instance.make_request_to_uri(uri)
    end
  end


  def test_connection_refused
    stub_request(:any, /cloud-volumes-infrastructure.default.svc.cluster.local/).
    with(
      headers: {
  	  'User-Agent'=>'Ruby'
      }).
    to_raise(Errno::ECONNREFUSED)

    f = Fluent::Test::Driver::Filter.new(Fluent::Plugin::DimensionHydrator)

    uri = URI('http://cloud-volumes-infrastructure.default.svc.cluster.local/')

    assert_raise Errno::ECONNREFUSED do
      f.instance.make_request_to_uri(uri)
    end
  end

  # Check that the record stays unmodified if getting data from storage throws 
  # exception
  def test_add_dimensions_exception_from_storage
    d = create_driver CONFIG do |uri|
      raise 'oh no'
    end
    expected_message = Marshal.load( Marshal.dump($sample_message))
    message_to_filter = Marshal.load( Marshal.dump(expected_message))
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message_to_filter)
    assert_equal expected_message, filtered_message
  end

  # Check that if a resource is not found, repopulation
  # is tried.
  def test_add_dimensions_repopulate
    # Throw exception first so that initially all the data is empty
    d = create_driver CONFIG do |uri|
      raise 'oh no'
    end
    assert_true d.instance.hosts_externalUUID.empty?
    # Make sure next time we get the right data
    mock_method d.instance, :make_request_to_uri, &healthy_storage_behavior
    expected_message = Marshal.load( Marshal.dump($sample_message))
    message_to_filter = Marshal.load( Marshal.dump(expected_message))
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message_to_filter)
    check_complete_dimensions(filtered_message, 'host1vendorid', 'otip14rack', 'otip14cluster1', 'otip14datacenter', 'otip14region1', 'sdeHost1Name')
  end


  # Check that if a resource is not found, record is unmodifies
  def test_add_dimensions_resource_not_found
    d = create_driver CONFIG do |uri|
      '[]'
    end
    expected_message = Marshal.load( Marshal.dump($sample_message))
    message_to_filter = Marshal.load( Marshal.dump(expected_message))
    filtered_message = d.instance.filter('kubernetes.ontap-ems-event', nil, message_to_filter)
    assert_equal expected_message, filtered_message
  end

  def test_add_ontap_auditlogs_dimensions_in_optimal_conditions
    d = create_driver CONFIG, &healthy_storage_behavior
    message = $sample_ontap_audit_message.clone
    filtered_message = d.instance.filter('kubernetes.ontap-auditlogs.user.info', nil, message)
    check_complete_dimensions(filtered_message, 'host1vendorid', 'otip14rack', 'otip14cluster1', 'otip14datacenter', 'otip14region1', 'sdeHost1Name')
  end

  def test_add_ontap_auditlogs_dimensions_with_missing_node_name
    d = create_driver CONFIG, &healthy_storage_behavior
    message = $sample_ontap_audit_message.clone
    message.delete('ident')
    expected_message = message.clone()
    filtered_message = d.instance.filter('kubernetes.ontap-auditlogs.user.info', nil, message)
    assert_equal expected_message, filtered_message
  end

  def test_add_ontap_auditlogs_dimensions_with_node_not_found
    d = create_driver CONFIG do |uri|
      '[]'
    end
    assert_true d.instance.nodes.empty?
    message = $sample_ontap_audit_message.clone
    filtered_message = d.instance.filter('kubernetes.ontap-auditlogs.user.info', nil, message)
    assert_equal $sample_ontap_audit_message, filtered_message
  end

  def test_add_dimensions_with_invalid_tag
    d = create_driver CONFIG, &healthy_storage_behavior
    message = $sample_ontap_audit_message.clone
    filtered_message = d.instance.filter('some-madeup-tag', nil, message)
    assert_equal $sample_ontap_audit_message, filtered_message
  end

  def setup
    Fluent::Test.setup
    $region_response = 
      '[
        {
          "uuid": "6fa7cef3-d880-cf0a-dad9-afd857dae309",
          "createdAt": "2018-08-28T09:54:59Z",
          "updatedAt": "2018-08-28T10:58:27Z",
          "deletedAt": null,
          "name": "otip14region1"
        },
        {
          "uuid": "0add9438-2163-2384-ae71-c835fe2e9b3b",
          "createdAt": "2018-09-25T14:07:24Z",
          "updatedAt": "2018-09-25T14:07:24Z",
          "deletedAt": null,
          "name": "otip14region2"
        }
      ]'
    $zone_response = 
    '[
      {
        "uuid": "59353951-5e70-7cf9-9827-8c2a4f12a5b6",
        "createdAt": "2018-08-28T09:55:12Z",
        "updatedAt": "2018-08-28T10:58:27Z",
        "deletedAt": null,
        "name": "otip14zone1",
        "vendorID": "otip14datacenter",
        "timezone": "UTC",
        "regionUUID": "6fa7cef3-d880-cf0a-dad9-afd857dae309",
        "regionName": "otip14region1"
      },
      {
        "uuid": "69353951-5e70-7cf9-9827-8c2a4f12a5b6",
        "createdAt": "2018-08-28T09:55:12Z",
        "updatedAt": "2018-08-28T10:58:27Z",
        "deletedAt": null,
        "name": "otip14zone2",
        "vendorID": "otip14datacenter2",
        "timezone": "UTC",
        "regionUUID": "0add9438-2163-2384-ae71-c835fe2e9b3b",
        "regionName": "otipq4region2"
      }
    ]'
    $cluster_response = 
    '[
      {
        "uuid": "469d5312-ba5c-11e8-a0a3-9a871ea9ae80",
        "createdAt": "2018-08-28T09:55:12Z",
        "updatedAt": "2018-08-28T10:58:27Z",
        "deletedAt": null,
        "name": "otip14zone1",
        "vendorID": "otip14cluster1",
        "zoneUUID": "59353951-5e70-7cf9-9827-8c2a4f12a5b6",
        "zoneName": "otip14zone1"
      },
      {
        "uuid": "569d5312-ba5c-11e8-a0a3-9a871ea9ae80",
        "createdAt": "2018-08-28T09:55:12Z",
        "updatedAt": "2018-08-28T10:58:27Z",
        "deletedAt": null,
        "name": "otip14zone1",
        "vendorID": "otip14cluster2",
        "zoneUUID": "69353951-5e70-7cf9-9827-8c2a4f12a5b6",
        "zoneName": "otip14zone2"
      }
    ]'
    $stamp_response = 
    '[
      {
        "uuid": "874b133e-d34c-986a-7b66-f54247ceac64",
        "createdAt": "2018-08-28T09:55:23Z",
        "updatedAt": "2018-08-28T10:58:27Z",
        "deletedAt": null,
        "name": "otip14stamp1",
        "vendorID": "otip14rack",
        "clusterUUID": "469d5312-ba5c-11e8-a0a3-9a871ea9ae80",
        "clusterName": "otip14zone1"
      },
      {
        "uuid": "974b133e-d34c-986a-7b66-f54247ceac64",
        "createdAt": "2018-08-28T09:55:23Z",
        "updatedAt": "2018-08-28T10:58:27Z",
        "deletedAt": null,
        "name": "otip14stamp2",
        "vendorID": "otip14rack2",
        "clusterUUID": "569d5312-ba5c-11e8-a0a3-9a871ea9ae80",
        "clusterName": "otip14zone2"
      }
    ]'
    $node_response = 
    '[
      {
        "uuid": "32eb7213-aa97-11e8-bcc8-005056b085eb",
        "createdAt": "2018-10-11T16:35:17Z",
        "updatedAt": "2018-10-11T16:35:17Z",
        "deletedAt": null,
        "name": "vsim7-node",
        "hostUUID": "88ek2b5e-a397-13e8-b3c8-035336b085eb",
        "hostName": "host1"
      },
      {
        "uuid": "42eb7213-aa97-11e8-bcc8-005056b085eb",
        "createdAt": "2018-10-11T16:35:17Z",
        "updatedAt": "2018-10-11T16:35:17Z",
        "deletedAt": null,
        "name": "vsim7-node2",
        "hostUUID": "5jek8b5e-aw97-11e8-bcc8-005056b085eb",
        "hostName": "host2"
      }
    ]'
    $host_response =
    '[
      {
        "externalUUID": "4be48bce-aa97-11e8-bcc8-005056b085eb",
        "uuid": "88ek2b5e-a397-13e8-b3c8-035336b085eb",
        "createdAt": "2018-08-28T09:57:41Z",
        "updatedAt": "2018-08-28T10:58:27Z",
        "deletedAt": null,
        "name": "sdeHost1Name",
        "type": "ontap",
        "version": "NetApp Release 9.4.1P1: Tue Aug 07 05:39:23 UTC 2018",
        "capabilities": "1.141",
        "ipAddress": "10.68.206.178",
        "port": 443,
        "protocol": "INSECURE_HTTPS",
        "username": "admin",
        "password": "****************",
        "state": "online",
        "stateDetails": "Online",
        "stampUUID": "874b133e-d34c-986a-7b66-f54247ceac64",
        "stampName": "otip14stamp1",
        "thin_provisioning": true,
        "vendorID": "host1vendorid"
      },
        {
        "externalUUID": "5be48bce-aa97-11e8-bcc8-005056b085eb",
        "uuid": "5jek8b5e-aw97-11e8-bcc8-005056b085eb",
        "createdAt": "2018-08-28T09:57:41Z",
        "updatedAt": "2018-08-28T10:58:27Z",
        "deletedAt": null,
        "name": "sdeHost2Name",
        "type": "ontap",
        "version": "NetApp Release 9.4.1P1: Tue Aug 07 05:39:23 UTC 2018",
        "capabilities": "1.141",
        "ipAddress": "10.68.206.178",
        "port": 443,
        "protocol": "INSECURE_HTTPS",
        "username": "admin",
        "password": "****************",
        "state": "online",
        "stateDetails": "Online",
        "stampUUID": "974b133e-d34c-986a-7b66-f54247ceac64",
        "stampName": "otip14stamp2",
        "thin_provisioning": true,
        "vendorID": "host2vendorid"
      }
    ]'
    $sample_message = 
    {
      'datacenter' => 'TestCenter',
      'message' => {
        'netapp' => {
          'ems_message_origin' => {
            'cluster_uuid' => '4be48bce-aa97-11e8-bcc8-005056b085eb',
            'node_uuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc'
          },
          'ems_message_info' => {
            'node' => 'node1',
            'message_name' => 'raid.disk.capacity.info',
            'version' => '1',
            'parameters' => {
              'actual' => '11.45 GB',
              'limit' => '6.00 TB'
            },
            'seq_num' => '420',
            'severity' => 'informational',
            'time' => '1432932082',
            'event' => 'raid.disk.capacity.info: System capacity visible is 11.45 GB of the 6.00 TB maximum. ',
            'num_suppressed_since_last' => '0',
            'ems_severity' => 'info',
            'event_xml_len' => '215',
            'source' => 'statd'
          },
          'version' => '1.0',
          'xmlns' => 'http://www.netapp.com/filer/admin'
        }
      }
    }
    $sample_message2 = Marshal.load( Marshal.dump($sample_message))
    $sample_message2['message']['netapp']['ems_message_origin']['cluster_uuid'] = '5be48bce-aa97-11e8-bcc8-005056b085eb'
    $sample_ontap_audit_message = 
    {
      'datacenter' => 'TestCenter',
      'message' => "this is some audit log",
      'ident' => 'vsim7-node'
    }
  end
end
