require 'helper'
require 'fluent/test/driver/filter'
require 'fluent/plugin/filter_ems_metric'

# Testing the Ems Metric Filter
class EmsMetricsFilterTest < Test::Unit::TestCase
  CONFIG = %(
    type ems_metric
    host '::1'
    port 8125
  )

  def setup
    Fluent::Test.setup
    @time = Fluent::Engine.now
    @sample_message =
    {
      'datacenter' => 'TestCenter',
      'message' => {
        'netapp' => {
          'ems_message_origin' => {
            'cluster_uuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
            'node_uuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc'
          },
          'ems_message_info' => {
            'node' => 'node1',
            'message_name' => 'raid.disk.capacity.info',
            'version' => '1',
            'parameters' => {
              'actual' => '11.45 GB',
              'limit' => '6.00 TB'
            },
            'seq_num' => '420',
            'severity' => 'informational',
            'time' => '1432932082',
            'event' => 'raid.disk.capacity.info: System capacity visible is 11.45 GB of the 6.00 TB maximum. ',
            'num_suppressed_since_last' => '0',
            'ems_severity' => 'info',
            'event_xml_len' => '215',
            'source' => 'statd'
          },
          'version' => '1.0',
          'xmlns' => 'http://www.netapp.com/filer/admin'
        }
      }
    }

  end


  def create_driver(conf = '')
    Fluent::Test::Driver::Filter
      .new(Fluent::Plugin::EmsMetricsFilter).configure(conf, syntax: :v1)
  end

  def test_harvest_metrics
    server = TCPServer.open(8125)
    d = create_driver CONFIG

    d.run(default_tag: 'test') do
      d.feed(@sample_message)
    end

    assert_equal [
      { 'Dims' =>
        { 
          'Node' => 'node1',
          'NodeUuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc',
          'ClusterUuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
          'ontapHostUuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
          'DataCenter' => 'TestCenter',
          'MessageName' => 'raid.disk.capacity.info',
          'ontapNodeName' => 'node1',
          'ontapNodeUuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc',
          'Severity' => 'informational',
          'Source' => 'statd' },
        'Metric' => 'EmsEvent',
        'Namespace' => 'Demo' }
    ], (d.filtered.map { |x| x[1] })
    server.close()
  end

  def test_metrics_without_datacenter_variable
    server = TCPServer.open(8125)
    d = create_driver CONFIG


    d.run(default_tag: 'test') do
      message_without_datacenter = @sample_message.clone()
      message_without_datacenter.delete('datacenter')
      d.feed(message_without_datacenter)
    end

    assert_equal [
      { 'Dims' =>
        { 
          'Node' => 'node1',
          'NodeUuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc',
          'ClusterUuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
          'ontapHostUuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
          'DataCenter' => 'attn: Datacenter Variable not set',
          'MessageName' => 'raid.disk.capacity.info',
          'ontapNodeName' => 'node1',
          'ontapNodeUuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc',
          'Severity' => 'informational',
          'Source' => 'statd' },
        'Metric' => 'EmsEvent',
        'Namespace' => 'Demo' }
    ], (d.filtered.map { |x| x[1] })
    server.close()
  
  end

  def test_metrics_with_azure_inventory
    server = TCPServer.open(8125)
    d = create_driver CONFIG

    d.run(default_tag: 'test') do
      message = @sample_message.clone()
      message['AzureRegion'] = 'azureRegion'
      message['AzureDatacenter'] = 'azureDatacenter'
      message['AzureClusterId'] = 'azureClusterId'
      message['AzureRackId'] = 'azureRackId'
      message['AzureHostId'] = 'azureHostId'
      message['sdeHostName'] = 'sdeHostName'
      d.feed(message)
    end

    assert_equal [
      { 'Dims' =>
        { 
          'Node' => 'node1',
          'NodeUuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc',
          'ClusterUuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
          'AzureClusterId' => 'azureClusterId',
          'AzureDatacenter' => 'azureDatacenter',
          'AzureRackId' => 'azureRackId',
          'AzureRegion' => 'azureRegion',
          'AzureHostId' => 'azureHostId',
          'ontapHostUuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
          'sdeHostName' => 'sdeHostName',
          'DataCenter' => 'TestCenter',
          'MessageName' => 'raid.disk.capacity.info',
          'ontapNodeName' => 'node1',
          'ontapNodeUuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc',
          'Severity' => 'informational',
          'Source' => 'statd' },
        'Metric' => 'EmsEvent',
        'Namespace' => 'Demo' }
    ], (d.filtered.map { |x| x[1] })
    server.close()
  end

  def test_filter_works_when_not_connecting_to_tcp
    server = TCPServer.open(8126)
    d = create_driver CONFIG

    d.run(default_tag: 'test') do
      d.feed(@sample_message)
    end

    assert_equal [
      { 'Dims' =>
        {
          'Node' => 'node1',
          'NodeUuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc',
          'ClusterUuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
          'ontapHostUuid' => 'f81d4fae-7dec-11d0-a765-00a0c91e6bf6',
          'DataCenter' => 'TestCenter',
          'MessageName' => 'raid.disk.capacity.info',
          'ontapNodeName' => 'node1',
          'ontapNodeUuid' => '5f29f789-349e-11e5-b0cf-b58c613d92fc',
          'Severity' => 'informational',
          'Source' => 'statd' },
        'Metric' => 'EmsEvent',
        'Namespace' => 'Demo' }
    ], (d.filtered.map { |x| x[1] })
    server.close()
  end
end
