require 'fluent/plugin/filter'
require 'json'
require 'socket'

module Fluent
  module Plugin
    # Filter plugin that extracts health metrics from OnTap xml
    class PrometheusAlertFilter < Filter
      Fluent::Plugin.register_filter('prometheus_alert', self)

      config_param :host, :string, :default => nil
      config_param :port, :integer, :default => 8125
      config_param :namespace, :string, :default => 'Demo'

      def configure(conf)
        super
        start_tcpsocket
      end

      def shutdown
        if @tcpsocket != nil
          @tcpsocket.close
        end
      end

      def filter(_tag, _time, record)
        records = []

        record['alerts'].each do |alert|
          dims = {}

          # Dimensions for hotpath - metrics and warmpath-logs, These can not be null/empty
          dims['AlertName'] = alert.dig('labels', 'alertname')
          dims['Severity'] = alert.dig('labels', 'severity')
          dims['Status'] = alert.dig('status')
          dims['Pod'] = alert.dig('labels', 'pod') || 'missingPodNameFromAlert'
	
          if record.key?('datacenter')
            dims['DataCenter'] = record['datacenter']
          else
            dims['DataCenter'] = 'attn: Datacenter Variable not set'
            $stdout.write 'Warning: Datacenter environment variable not set'
          end

          statsd_message = {}
          statsd_message['Namespace'] = @namespace
          statsd_message['Metric'] = 'PrometheusAlert'
          statsd_message['Dims'] = dims
          statsd_json = statsd_message.to_json
          statsd_payload = "#{statsd_json}:1|g\n"

          # send to hotpath - metrics
          if @tcpsocket == nil
            start_tcpsocket
          end

          if @tcpsocket != nil
            begin
              @tcpsocket.write statsd_payload
            rescue => e
              $log.error "Error sending PrometheusAlert: #{e.message} (#{e.class})"
              @tcpsocket.close
              @tcpsocket = nil
            end
          end

          # Dimensions for warmpath-logs, These can be empty / null.
          dims['Component'] = alert.dig('labels', 'component')
          dims['Job'] = alert.dig('labels', 'job')
          dims['Pod'] = alert.dig('labels', 'pod')
          dims['Container'] = alert.dig('labels', 'container')
          dims['Description'] = alert.dig('annotations', 'description')

          statsd_message['Dims'] = dims

          # return to warmpath
          records.push(statsd_message)
        end
        records
      end

      def start_tcpsocket
        begin
          $log.info "Opening a TCP connection: #{@host}:#{@port}"
          @tcpsocket = TCPSocket.new(@host,@port)
          @tcpsocket.setsockopt(Socket::SOL_SOCKET, Socket::SO_KEEPALIVE, true)
          @tcpsocket.setsockopt(Socket::SOL_TCP, Socket::TCP_KEEPIDLE, 60)
          @tcpsocket.setsockopt(Socket::SOL_TCP, Socket::TCP_KEEPINTVL, 20)
          @tcpsocket.setsockopt(Socket::SOL_TCP, Socket::TCP_KEEPCNT, 4)
        rescue => e
          $log.error "Can't connect to geneva-metrics service, is the service running and open on #{@host}:#{@port}? Error: #{e.message} (#{e.class})"
          @tcpsocket = nil
        end
      end

    end
  end
end
