/*
 * Decompiled with CFR 0.152.
 */
package ch.vorburger.mariadb4j;

import ch.vorburger.exec.ManagedProcess;
import ch.vorburger.exec.ManagedProcessBuilder;
import ch.vorburger.exec.ManagedProcessException;
import ch.vorburger.exec.OutputStreamLogDispatcher;
import ch.vorburger.mariadb4j.DBConfiguration;
import ch.vorburger.mariadb4j.DBConfigurationBuilder;
import ch.vorburger.mariadb4j.MariaDBOutputStreamLogDispatcher;
import ch.vorburger.mariadb4j.Util;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DB {
    private static final Logger logger = LoggerFactory.getLogger(DB.class);
    protected final DBConfiguration configuration;
    private File baseDir;
    private File libDir;
    private File dataDir;
    private ManagedProcess mysqldProcess;
    protected int dbStartMaxWaitInMS = 30000;

    protected DB(DBConfiguration config) {
        this.configuration = config;
    }

    public DBConfiguration getConfiguration() {
        return this.configuration;
    }

    public static DB newEmbeddedDB(DBConfiguration config) throws ManagedProcessException {
        DB db = new DB(config);
        db.prepareDirectories();
        db.unpackEmbeddedDb();
        db.install();
        return db;
    }

    public static DB newEmbeddedDB(int port) throws ManagedProcessException {
        DBConfigurationBuilder config = new DBConfigurationBuilder();
        config.setPort(port);
        return DB.newEmbeddedDB(config.build());
    }

    ManagedProcess installPreparation() throws ManagedProcessException, IOException {
        logger.info("Installing a new embedded database to: " + this.baseDir);
        File installDbCmdFile = this.newExecutableFile("bin", "mysql_install_db");
        if (!installDbCmdFile.exists()) {
            installDbCmdFile = this.newExecutableFile("scripts", "mysql_install_db");
        }
        if (!installDbCmdFile.exists()) {
            throw new ManagedProcessException("mysql_install_db was not found, neither in bin/ nor in scripts/ under " + this.baseDir.getAbsolutePath());
        }
        ManagedProcessBuilder builder = new ManagedProcessBuilder(installDbCmdFile);
        builder.setOutputStreamLogDispatcher(this.getOutputStreamLogDispatcher("mysql_install_db"));
        builder.getEnvironment().put(this.configuration.getOSLibraryEnvironmentVarName(), this.libDir.getAbsolutePath());
        builder.addFileArgument("--datadir", this.dataDir).setWorkingDirectory(this.baseDir);
        if (!this.configuration.isWindows()) {
            builder.addFileArgument("--basedir", this.baseDir);
            builder.addArgument("--no-defaults");
            builder.addArgument("--force");
            builder.addArgument("--skip-name-resolve");
        }
        ManagedProcess mysqlInstallProcess = builder.build();
        return mysqlInstallProcess;
    }

    protected synchronized void install() throws ManagedProcessException {
        try {
            ManagedProcess mysqlInstallProcess = this.installPreparation();
            mysqlInstallProcess.start();
            mysqlInstallProcess.waitForExit();
        }
        catch (Exception e) {
            throw new ManagedProcessException("An error occurred while installing the database", e);
        }
        logger.info("Installation complete.");
    }

    protected String getWinExeExt() {
        return this.configuration.isWindows() ? ".exe" : "";
    }

    public synchronized void start() throws ManagedProcessException {
        logger.info("Starting up the database...");
        boolean ready = false;
        try {
            this.mysqldProcess = this.startPreparation();
            ready = this.mysqldProcess.startAndWaitForConsoleMessageMaxMs(this.getReadyForConnectionsTag(), this.dbStartMaxWaitInMS);
        }
        catch (Exception e) {
            logger.error("failed to start mysqld", (Throwable)e);
            throw new ManagedProcessException("An error occurred while starting the database", e);
        }
        if (!ready) {
            if (this.mysqldProcess.isAlive()) {
                this.mysqldProcess.destroy();
            }
            throw new ManagedProcessException("Database does not seem to have started up correctly? Magic string not seen in " + this.dbStartMaxWaitInMS + "ms: " + this.getReadyForConnectionsTag() + this.mysqldProcess.getLastConsoleLines());
        }
        logger.info("Database startup complete.");
    }

    protected String getReadyForConnectionsTag() {
        return "mysqld" + this.getWinExeExt() + ": ready for connections.";
    }

    synchronized ManagedProcess startPreparation() throws ManagedProcessException, IOException {
        ManagedProcessBuilder builder = new ManagedProcessBuilder(this.newExecutableFile("bin", "mysqld"));
        builder.setOutputStreamLogDispatcher(this.getOutputStreamLogDispatcher("mysqld"));
        builder.getEnvironment().put(this.configuration.getOSLibraryEnvironmentVarName(), this.libDir.getAbsolutePath());
        builder.addArgument("--no-defaults");
        builder.addArgument("--console");
        builder.addArgument("--skip-grant-tables");
        builder.addArgument("--max_allowed_packet=64M");
        builder.addFileArgument("--basedir", this.baseDir).setWorkingDirectory(this.baseDir);
        builder.addFileArgument("--datadir", this.dataDir);
        this.addPortAndMaybeSocketArguments(builder);
        for (String arg : this.configuration.getArgs()) {
            builder.addArgument(arg);
        }
        this.cleanupOnExit();
        builder.setDestroyOnShutdown(false);
        logger.info("mysqld executable: " + builder.getExecutable());
        return builder.build();
    }

    protected File newExecutableFile(String dir, String exec) {
        return new File(this.baseDir, dir + "/" + exec + this.getWinExeExt());
    }

    protected void addPortAndMaybeSocketArguments(ManagedProcessBuilder builder) throws IOException {
        builder.addArgument("--port=" + this.configuration.getPort());
        if (!this.configuration.isWindows()) {
            builder.addFileArgument("--socket", this.getAbsoluteSocketFile());
        }
    }

    protected void addSocketOrPortArgument(ManagedProcessBuilder builder) throws IOException {
        if (!this.configuration.isWindows()) {
            builder.addFileArgument("--socket", this.getAbsoluteSocketFile());
        } else {
            builder.addArgument("--port=" + this.configuration.getPort());
        }
    }

    protected File getAbsoluteSocketFile() {
        String socket = this.configuration.getSocket();
        File socketFile = new File(socket);
        return socketFile.getAbsoluteFile();
    }

    public void source(String resource) throws ManagedProcessException {
        this.source(resource, null, null, null);
    }

    public void source(String resource, String username, String password, String dbName) throws ManagedProcessException {
        InputStream from = this.getClass().getClassLoader().getResourceAsStream(resource);
        if (from == null) {
            throw new IllegalArgumentException("Could not find script file on the classpath at: " + resource);
        }
        this.run("script file sourced from the classpath at: " + resource, from, username, password, dbName);
    }

    public void run(String command, String username, String password, String dbName) throws ManagedProcessException {
        InputStream from = IOUtils.toInputStream((String)command, (Charset)Charset.defaultCharset());
        this.run("command: " + command, from, username, password, dbName);
    }

    public void run(String command) throws ManagedProcessException {
        this.run(command, null, null, null);
    }

    protected void run(String logInfoText, InputStream fromIS, String username, String password, String dbName) throws ManagedProcessException {
        logger.info("Running a " + logInfoText);
        try {
            ManagedProcessBuilder builder = new ManagedProcessBuilder(this.newExecutableFile("bin", "mysql"));
            builder.setOutputStreamLogDispatcher(this.getOutputStreamLogDispatcher("mysql"));
            builder.setWorkingDirectory(this.baseDir);
            if (username != null) {
                builder.addArgument("-u", username);
            }
            if (password != null) {
                builder.addArgument("-p", password);
            }
            if (dbName != null) {
                builder.addArgument("-D", dbName);
            }
            this.addSocketOrPortArgument(builder);
            if (fromIS != null) {
                builder.setInputStream(fromIS);
            }
            ManagedProcess process = builder.build();
            process.start();
            process.waitForExit();
        }
        catch (Exception e) {
            throw new ManagedProcessException("An error occurred while running a " + logInfoText, e);
        }
        finally {
            IOUtils.closeQuietly((InputStream)fromIS);
        }
        logger.info("Successfully ran the " + logInfoText);
    }

    public void createDB(String dbName) throws ManagedProcessException {
        this.run("create database if not exists `" + dbName + "`;");
    }

    protected OutputStreamLogDispatcher getOutputStreamLogDispatcher(String exec) {
        return new MariaDBOutputStreamLogDispatcher();
    }

    public synchronized void stop() throws ManagedProcessException {
        if (this.mysqldProcess.isAlive()) {
            logger.debug("Stopping the database...");
            this.mysqldProcess.destroy();
            logger.info("Database stopped.");
        } else {
            logger.debug("Database was already stopped.");
        }
    }

    protected void unpackEmbeddedDb() {
        if (this.configuration.getBinariesClassPathLocation() == null) {
            logger.info("Not unpacking any embedded database (as BinariesClassPathLocation configuration is null)");
            return;
        }
        try {
            Util.extractFromClasspathToFile(this.configuration.getBinariesClassPathLocation(), this.baseDir);
            if (!this.configuration.isWindows()) {
                Util.forceExecutable(this.newExecutableFile("bin", "my_print_defaults"));
                Util.forceExecutable(this.newExecutableFile("bin", "mysql_install_db"));
                Util.forceExecutable(this.newExecutableFile("bin", "mysqld"));
                Util.forceExecutable(this.newExecutableFile("bin", "mysql"));
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Error unpacking embedded DB", e);
        }
    }

    protected void prepareDirectories() throws ManagedProcessException {
        this.baseDir = Util.getDirectory(this.configuration.getBaseDir());
        this.libDir = Util.getDirectory(this.configuration.getLibDir());
        try {
            String dataDirPath = this.configuration.getDataDir();
            if (Util.isTemporaryDirectory(dataDirPath)) {
                FileUtils.deleteDirectory((File)new File(dataDirPath));
            }
            this.dataDir = Util.getDirectory(dataDirPath);
        }
        catch (Exception e) {
            throw new ManagedProcessException("An error occurred while preparing the data directory", e);
        }
    }

    protected void cleanupOnExit() {
        String threadName = "Shutdown Hook Deletion Thread for Temporary DB " + this.dataDir.toString();
        final DB db = this;
        Runtime.getRuntime().addShutdownHook(new Thread(threadName){

            @Override
            public void run() {
                try {
                    if (DB.this.mysqldProcess != null && DB.this.mysqldProcess.isAlive()) {
                        logger.info("cleanupOnExit() ShutdownHook now stopping database");
                        db.stop();
                    }
                }
                catch (ManagedProcessException e) {
                    logger.warn("cleanupOnExit() ShutdownHook: An error occurred while stopping the database", (Throwable)e);
                }
                if (DB.this.dataDir.exists() && Util.isTemporaryDirectory(DB.this.dataDir.getAbsolutePath())) {
                    logger.info("cleanupOnExit() ShutdownHook quietly deleting temporary DB data directory: " + DB.this.dataDir);
                    FileUtils.deleteQuietly((File)DB.this.dataDir);
                }
                if (DB.this.baseDir.exists() && Util.isTemporaryDirectory(DB.this.baseDir.getAbsolutePath())) {
                    logger.info("cleanupOnExit() ShutdownHook quietly deleting temporary DB base directory: " + DB.this.baseDir);
                    FileUtils.deleteQuietly((File)DB.this.baseDir);
                }
            }
        });
    }
}

