/*!
 * Based on chai library
 * http://chaijs.com
 * Copyright(c) 2011-2014 Jake Luer <jake@alogicalparadox.com>
 * MIT Licensed
 */

module.exports = function (chai, _) {
  const Assertion = chai.Assertion;
  const flag = _.flag;

  /**
   * ### Language Chains
   *
   * The following are provided as chainable getters to
   * improve the readability of your assertions. They
   * do not provide testing capabilities unless they
   * have been overwritten by a plugin.
   *
   * **Chains**
   *
   * - to
   * - be
   * - been
   * - is
   * - that
   * - which
   * - and
   * - has
   * - have
   * - with
   * - at
   * - of
   * - same
   *
   * @name language chains
   * @api public
   */

  [ 'to', 'be', 'been'
  , 'is', 'and', 'has', 'have'
  , 'with', 'that', 'which', 'at', 'does'
  , 'of', 'same' ].forEach(function (chain) {
    Assertion.addProperty(chain, function () {
      flag(this, chain, true);
      const isComponent = flag(this, 'component') === true;

      return new Proxy(this, {
        get: function(obj, prop) {
          if (typeof prop != 'string') {
            return {};
          }

          if (prop in obj) {
            return obj[prop];
          }

          if (prop === 'deferred' || isComponent) {
            return;
          }

          if (prop === 'then' || prop === 'catch') {
            const promise = flag(this, 'promise');

            if (promise instanceof Promise) {
              return promise[prop];
            }

            return;
          }

          throw new Error(`Unknown property: "${prop}". Please consult docs at: http://nightwatchjs.org/api.`)
        }
      });
    });
  });

  function verifyValueFlag(assertion) {
    setTimeout(function() {
      const obj = flag(assertion, 'attributeFlag') ||
        flag(assertion, 'textFlag') ||
        flag(assertion, '_heightFlag') ||
        flag(assertion, 'cssFlag') ||
        flag(assertion, 'urlValueFlag') ||
        flag(assertion, 'titleValueFlag') ||
        flag(assertion, 'cookieValueFlag') ||
        flag(assertion, 'activeElementFlag') ||
        flag(assertion, 'valueFlag');

      if (!obj) {
        const emitter = flag(assertion, 'emitter');
        emitter.emit('error', new Error('Expect expression error: attribute, value or text is missing.'));
      }
    }, 0);
  }

  /**
   * ### .not
   *
   * Negates any of assertions following in the chain.
   *
   *     expect(foo).to.not.equal('bar');
   *     expect(goodFn).to.not.throw(Error);
   *     expect({ foo: 'baz' }).to.have.property('foo')
   *       .and.not.equal('bar');
   *
   * @name not
   * @api public
   */

  Assertion.addProperty('not', function() {
    flag(this, 'negate', true);
  });

  /**
   * ### .deep
   *
   * Sets the `deep` flag, later used by the `equal` and
   * `property` assertions.
   *
   *     expect(foo).to.deep.equal({ bar: 'baz' });
   *     expect({ foo: { bar: { baz: 'quux' } } })
   *       .to.have.deep.property('foo.bar.baz', 'quux');
   *
   * `.deep.property` special characters can be escaped
   * by adding two slashes before the `.` or `[]`.
   *
   *     var deepCss = { '.link': { '[target]': 42 }};
   *     expect(deepCss).to.have.deep.property('\\.link.\\[target\\]', 42);
   *
   * @name deep
   * @api public
   */

  Assertion.addProperty('deep', function() {
    flag(this, 'deep', true);
  });

  /**
   * ### .nested
   *
   * Enables dot- and bracket-notation in all `.property` and `.include`
   * assertions that follow in the chain.
   *
   *     expect({a: {b: ['x', 'y']}}).to.have.nested.property('a.b[1]');
   *     expect({a: {b: ['x', 'y']}}).to.nested.include({'a.b[1]': 'y'});
   *
   * If `.` or `[]` are part of an actual property name, they can be escaped by
   * adding two backslashes before them.
   *
   *     expect({'.a': {'[b]': 'x'}}).to.have.nested.property('\\.a.\\[b\\]');
   *     expect({'.a': {'[b]': 'x'}}).to.nested.include({'\\.a.\\[b\\]': 'x'});
   *
   * `.nested` cannot be combined with `.own`.
   *
   * @name nested
   * @namespace BDD
   * @api public
   */

  Assertion.addProperty('nested', function () {
    flag(this, 'nested', true);
  });

  /**
   * ### .own
   *
   * Causes all `.property` and `.include` assertions that follow in the chain
   * to ignore inherited properties.
   *
   *     Object.prototype.b = 2;
   *
   *     expect({a: 1}).to.have.own.property('a');
   *     expect({a: 1}).to.have.property('b');
   *     expect({a: 1}).to.not.have.own.property('b');
   *
   *     expect({a: 1}).to.own.include({a: 1});
   *     expect({a: 1}).to.include({b: 2}).but.not.own.include({b: 2});
   *
   * `.own` cannot be combined with `.nested`.
   *
   * @name own
   * @namespace BDD
   * @api public
   */

  Assertion.addProperty('own', function () {
    flag(this, 'own', true);
  });

  /**
   * ### .ordered
   *
   * Causes all `.members` assertions that follow in the chain to require that
   * members be in the same order.
   *
   *     expect([1, 2]).to.have.ordered.members([1, 2])
   *       .but.not.have.ordered.members([2, 1]);
   *
   * When `.include` and `.ordered` are combined, the ordering begins at the
   * start of both arrays.
   *
   *     expect([1, 2, 3]).to.include.ordered.members([1, 2])
   *       .but.not.include.ordered.members([2, 3]);
   *
   * @name ordered
   * @namespace BDD
   * @api public
   */

  Assertion.addProperty('ordered', function () {
    flag(this, 'ordered', true);
  });

  /**
   * ### .any
   *
   * Sets the `any` flag, (opposite of the `all` flag)
   * later used in the `keys` assertion.
   *
   *     expect(foo).to.have.any.keys('bar', 'baz');
   *
   * @name any
   * @api public
   */

  Assertion.addProperty('any', function() {
    flag(this, 'any', true);
    flag(this, 'all', false)
  });

  /**
   * ### .all
   *
   * Sets the `all` flag (opposite of the `any` flag)
   * later used by the `keys` assertion.
   *
   *     expect(foo).to.have.all.keys('bar', 'baz');
   *
   * @name all
   * @api public
   */

  Assertion.addProperty('all', function () {
    flag(this, 'all', true);
    flag(this, 'any', false);
  });

  Assertion.addMethod('before', function(ms) {
    flag(this, 'waitFor', ms);
    flag(this, 'before', true);
  });

  Assertion.addMethod('after', function(ms) {
    flag(this, 'after', true);
    flag(this, 'waitFor', ms);
  });

  /**
   * ### .include(value)
   *
   * The `include` and `contain` assertions can be used as either property
   * based language chains or as methods to assert the inclusion of an object
   * in an array or a substring in a string. When used as language chains,
   * they toggle the `contains` flag for the `keys` assertion.
   *
   *     expect([1,2,3]).to.include(2);
   *     expect('foobar').to.contain('foo');
   *     expect({ foo: 'bar', hello: 'universe' }).to.include.keys('foo');
   *
   * @name include
   * @alias contain
   * @alias includes
   * @alias contains
   * @param {Object|String|Number} obj
   * @param {String} message _optional_
   * @api public
   */
  function includeChainingBehavior() {
    flag(this, 'contains', true);
  }

  function include(val, msg) {
    if (msg) {
      flag(this, 'message', msg);
    }
    flag(this, 'contains', val);

    verifyValueFlag(this);
  }

  Assertion.addChainableMethod('include', include, includeChainingBehavior);
  Assertion.addChainableMethod('contain', include, includeChainingBehavior);
  Assertion.addChainableMethod('contains', include, includeChainingBehavior);
  Assertion.addChainableMethod('includes', include, includeChainingBehavior);

  function startWith(val, msg) {
    if (msg) {
      flag(this, 'message', msg);
    }

    flag(this, 'startsWith', val);

    verifyValueFlag(this);
  }

  Assertion.addMethod('startWith', startWith);
  Assertion.addMethod('startsWith', startWith);

  function endWith(val, msg) {
    if (msg) {
      flag(this, 'message', msg);
    }
    flag(this, 'endsWith', val);

    verifyValueFlag(this);
  }

  Assertion.addMethod('endWith', endWith);
  Assertion.addMethod('endsWith', endWith);

  /**
   * ### .match(regexp)
   *
   * Asserts that the target matches a regular expression.
   *
   *     expect('foobar').to.match(/^foo/);
   *
   * @name match
   * @param {RegExp} re
   * @param {String} [msg]
   * @api public
   */
  function matches(re, msg) {
    if (!(re instanceof RegExp)) {
      throw new Error('matches requires first parameter to be a RegExp. "' + (typeof re) + '" given.');
    }

    if (msg) {
      flag(this, 'message', msg);
    }
    flag(this, 'matches', re);

    verifyValueFlag(this);
  }

  Assertion.addMethod('match', matches);
  Assertion.addMethod('matches', matches);

  /**
   * ### .equal(value)
   *
   * Asserts that the target is strictly equal (`===`) to `value`.
   * Alternately, if the `deep` flag is set, asserts that
   * the target is deeply equal to `value`.
   *
   *     expect('hello').to.equal('hello');
   *     expect(42).to.equal(42);
   *     expect(1).to.not.equal(true);
   *     expect({ foo: 'bar' }).to.not.equal({ foo: 'bar' });
   *     expect({ foo: 'bar' }).to.deep.equal({ foo: 'bar' });
   *
   * @name equal
   * @alias equals
   * @alias eq
   * @alias deep.equal
   * @param {*} val
   * @param {String} [msg]
   * @api public
   */

  function assertEqual(val, msg) {
    if (msg) {
      flag(this, 'message', msg);
    }
    flag(this, 'equal', val);
  }

  Assertion.addMethod('equal', assertEqual);
  Assertion.addMethod('equals', assertEqual);
  Assertion.addMethod('eq', assertEqual);

  const chainableMethods = {
    toContain: include,
    toEndWith: endWith,
    toEqual: assertEqual,
    toBe: assertEqual,
    toMatch: matches
  };

  Object.keys(chainableMethods).forEach(function(key) {
    Assertion.addChainableMethod(key, function(val, msg) {
      chainableMethods[key].call(this, val, msg);

      return flag(this, 'api');
    });
  });
};
