import * as tslib_1 from "tslib";
import { Inject, Injectable, Optional } from '@angular/core';
import { compose } from '@ngrx/store';
import { ENTITY_COLLECTION_META_REDUCERS } from './constants';
import { EntityCollectionReducerFactory, } from './entity-collection-reducer';
/**
 * Registry of entity types and their previously-constructed reducers.
 * Can create a new CollectionReducer, which it registers for subsequent use.
 */
var EntityCollectionReducerRegistry = /** @class */ (function () {
    function EntityCollectionReducerRegistry(entityCollectionReducerFactory, entityCollectionMetaReducers) {
        this.entityCollectionReducerFactory = entityCollectionReducerFactory;
        this.entityCollectionReducers = {};
        this.entityCollectionMetaReducer = compose.apply(null, entityCollectionMetaReducers || []);
    }
    /**
     * Get the registered EntityCollectionReducer<T> for this entity type or create one and register it.
     * @param entityName Name of the entity type for this reducer
     */
    EntityCollectionReducerRegistry.prototype.getOrCreateReducer = function (entityName) {
        var reducer = this.entityCollectionReducers[entityName];
        if (!reducer) {
            reducer = this.entityCollectionReducerFactory.create(entityName);
            reducer = this.registerReducer(entityName, reducer);
            this.entityCollectionReducers[entityName] = reducer;
        }
        return reducer;
    };
    /**
     * Register an EntityCollectionReducer for an entity type
     * @param entityName - the name of the entity type
     * @param reducer - reducer for that entity type
     *
     * Examples:
     *   registerReducer('Hero', myHeroReducer);
     *   registerReducer('Villain', myVillainReducer);
     */
    EntityCollectionReducerRegistry.prototype.registerReducer = function (entityName, reducer) {
        reducer = this.entityCollectionMetaReducer(reducer);
        return (this.entityCollectionReducers[entityName.trim()] = reducer);
    };
    /**
     * Register a batch of EntityCollectionReducers.
     * @param reducers - reducers to merge into existing reducers
     *
     * Examples:
     *   registerReducers({
     *     Hero: myHeroReducer,
     *     Villain: myVillainReducer
     *   });
     */
    EntityCollectionReducerRegistry.prototype.registerReducers = function (reducers) {
        var _this = this;
        var keys = reducers ? Object.keys(reducers) : [];
        keys.forEach(function (key) { return _this.registerReducer(key, reducers[key]); });
    };
    EntityCollectionReducerRegistry = tslib_1.__decorate([
        Injectable(),
        tslib_1.__param(1, Optional()),
        tslib_1.__param(1, Inject(ENTITY_COLLECTION_META_REDUCERS)),
        tslib_1.__metadata("design:paramtypes", [EntityCollectionReducerFactory, Array])
    ], EntityCollectionReducerRegistry);
    return EntityCollectionReducerRegistry;
}());
export { EntityCollectionReducerRegistry };
//# sourceMappingURL=data:application/json;base64,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