import * as tslib_1 from "tslib";
import { Inject, Injectable, InjectionToken } from '@angular/core';
import { UPDATE } from '@ngrx/store';
import { empty, Observable, of } from 'rxjs';
import { catchError, concatMap, debounceTime, filter, map, share, switchMap, take, takeUntil, timeout, } from 'rxjs/operators';
import { IMPORT_STATE, PERFORM_ACTION } from './actions';
import { STORE_DEVTOOLS_CONFIG, StoreDevtoolsConfig, } from './config';
import { DevtoolsDispatcher } from './devtools-dispatcher';
import { isActionFiltered, sanitizeAction, sanitizeActions, sanitizeState, sanitizeStates, shouldFilterActions, unliftState, } from './utils';
export var ExtensionActionTypes = {
    START: 'START',
    DISPATCH: 'DISPATCH',
    STOP: 'STOP',
    ACTION: 'ACTION',
};
export var REDUX_DEVTOOLS_EXTENSION = new InjectionToken('Redux Devtools Extension');
var DevtoolsExtension = /** @class */ (function () {
    function DevtoolsExtension(devtoolsExtension, config, dispatcher) {
        this.config = config;
        this.dispatcher = dispatcher;
        this.devtoolsExtension = devtoolsExtension;
        this.createActionStreams();
    }
    DevtoolsExtension.prototype.notify = function (action, state) {
        var _this = this;
        if (!this.devtoolsExtension) {
            return;
        }
        // Check to see if the action requires a full update of the liftedState.
        // If it is a simple action generated by the user's app and the recording
        // is not locked/paused, only send the action and the current state (fast).
        //
        // A full liftedState update (slow: serializes the entire liftedState) is
        // only required when:
        //   a) redux-devtools-extension fires the @@Init action (ignored by
        //      @ngrx/store-devtools)
        //   b) an action is generated by an @ngrx module (e.g. @ngrx/effects/init
        //      or @ngrx/store/update-reducers)
        //   c) the state has been recomputed due to time-traveling
        //   d) any action that is not a PerformAction to err on the side of
        //      caution.
        if (action.type === PERFORM_ACTION) {
            if (state.isLocked || state.isPaused) {
                return;
            }
            var currentState = unliftState(state);
            if (shouldFilterActions(this.config) &&
                isActionFiltered(currentState, action, this.config.predicate, this.config.actionsSafelist, this.config.actionsBlocklist)) {
                return;
            }
            var sanitizedState_1 = this.config.stateSanitizer
                ? sanitizeState(this.config.stateSanitizer, currentState, state.currentStateIndex)
                : currentState;
            var sanitizedAction_1 = this.config.actionSanitizer
                ? sanitizeAction(this.config.actionSanitizer, action, state.nextActionId)
                : action;
            this.sendToReduxDevtools(function () {
                return _this.extensionConnection.send(sanitizedAction_1, sanitizedState_1);
            });
        }
        else {
            // Requires full state update
            var sanitizedLiftedState_1 = tslib_1.__assign({}, state, { stagedActionIds: state.stagedActionIds, actionsById: this.config.actionSanitizer
                    ? sanitizeActions(this.config.actionSanitizer, state.actionsById)
                    : state.actionsById, computedStates: this.config.stateSanitizer
                    ? sanitizeStates(this.config.stateSanitizer, state.computedStates)
                    : state.computedStates });
            this.sendToReduxDevtools(function () {
                return _this.devtoolsExtension.send(null, sanitizedLiftedState_1, _this.getExtensionConfig(_this.config));
            });
        }
    };
    DevtoolsExtension.prototype.createChangesObservable = function () {
        var _this = this;
        if (!this.devtoolsExtension) {
            return empty();
        }
        return new Observable(function (subscriber) {
            var connection = _this.devtoolsExtension.connect(_this.getExtensionConfig(_this.config));
            _this.extensionConnection = connection;
            connection.init();
            connection.subscribe(function (change) { return subscriber.next(change); });
            return connection.unsubscribe;
        });
    };
    DevtoolsExtension.prototype.createActionStreams = function () {
        var _this = this;
        // Listens to all changes
        var changes$ = this.createChangesObservable().pipe(share());
        // Listen for the start action
        var start$ = changes$.pipe(filter(function (change) { return change.type === ExtensionActionTypes.START; }));
        // Listen for the stop action
        var stop$ = changes$.pipe(filter(function (change) { return change.type === ExtensionActionTypes.STOP; }));
        // Listen for lifted actions
        var liftedActions$ = changes$.pipe(filter(function (change) { return change.type === ExtensionActionTypes.DISPATCH; }), map(function (change) { return _this.unwrapAction(change.payload); }), concatMap(function (action) {
            if (action.type === IMPORT_STATE) {
                // State imports may happen in two situations:
                // 1. Explicitly by user
                // 2. User activated the "persist state accross reloads" option
                //    and now the state is imported during reload.
                // Because of option 2, we need to give possible
                // lazy loaded reducers time to instantiate.
                // As soon as there is no UPDATE action within 1 second,
                // it is assumed that all reducers are loaded.
                return _this.dispatcher.pipe(filter(function (action) { return action.type === UPDATE; }), timeout(1000), debounceTime(1000), map(function () { return action; }), catchError(function () { return of(action); }), take(1));
            }
            else {
                return of(action);
            }
        }));
        // Listen for unlifted actions
        var actions$ = changes$.pipe(filter(function (change) { return change.type === ExtensionActionTypes.ACTION; }), map(function (change) { return _this.unwrapAction(change.payload); }));
        var actionsUntilStop$ = actions$.pipe(takeUntil(stop$));
        var liftedUntilStop$ = liftedActions$.pipe(takeUntil(stop$));
        this.start$ = start$.pipe(takeUntil(stop$));
        // Only take the action sources between the start/stop events
        this.actions$ = this.start$.pipe(switchMap(function () { return actionsUntilStop$; }));
        this.liftedActions$ = this.start$.pipe(switchMap(function () { return liftedUntilStop$; }));
    };
    DevtoolsExtension.prototype.unwrapAction = function (action) {
        return typeof action === 'string' ? eval("(" + action + ")") : action;
    };
    DevtoolsExtension.prototype.getExtensionConfig = function (config) {
        var extensionOptions = {
            name: config.name,
            features: config.features,
            serialize: config.serialize,
        };
        if (config.maxAge !== false /* support === 0 */) {
            extensionOptions.maxAge = config.maxAge;
        }
        return extensionOptions;
    };
    DevtoolsExtension.prototype.sendToReduxDevtools = function (send) {
        try {
            send();
        }
        catch (err) {
            console.warn('@ngrx/store-devtools: something went wrong inside the redux devtools', err);
        }
    };
    DevtoolsExtension = tslib_1.__decorate([
        Injectable(),
        tslib_1.__param(0, Inject(REDUX_DEVTOOLS_EXTENSION)),
        tslib_1.__param(1, Inject(STORE_DEVTOOLS_CONFIG)),
        tslib_1.__metadata("design:paramtypes", [Object, StoreDevtoolsConfig,
            DevtoolsDispatcher])
    ], DevtoolsExtension);
    return DevtoolsExtension;
}());
export { DevtoolsExtension };
//# sourceMappingURL=data:application/json;base64,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