import * as tslib_1 from "tslib";
/** @ng2api @module directives */
/** */
import { Directive, Output, EventEmitter, ContentChildren, QueryList, Host, Self, Optional } from '@angular/core';
import { UISref } from './uiSref';
import { anyTrueR, tail, unnestR, UIRouterGlobals, Param, PathUtils, identity, uniqR, } from '@uirouter/core';
import { BehaviorSubject, of, from, combineLatest, concat } from 'rxjs';
import { switchMap, map } from 'rxjs/operators';
/** @internalapi */
const inactiveStatus = {
    active: false,
    exact: false,
    entering: false,
    exiting: false,
    targetStates: [],
};
/**
 * Returns a Predicate<PathNode[]>
 *
 * The predicate returns true when the target state (and param values)
 * match the (tail of) the path, and the path's param values
 *
 * @internalapi
 */
const pathMatches = (target) => {
    if (!target.exists())
        return () => false;
    const state = target.$state();
    const targetParamVals = target.params();
    const targetPath = PathUtils.buildPath(target);
    const paramSchema = targetPath
        .map(node => node.paramSchema)
        .reduce(unnestR, [])
        .filter((param) => targetParamVals.hasOwnProperty(param.id));
    return (path) => {
        const tailNode = tail(path);
        if (!tailNode || tailNode.state !== state)
            return false;
        const paramValues = PathUtils.paramValues(path);
        return Param.equals(paramSchema, paramValues, targetParamVals);
    };
};
const ɵ0 = pathMatches;
/**
 * Given basePath: [a, b], appendPath: [c, d]),
 * Expands the path to [c], [c, d]
 * Then appends each to [a,b,] and returns: [a, b, c], [a, b, c, d]
 *
 * @internalapi
 */
function spreadToSubPaths(basePath, appendPath) {
    return appendPath.map(node => basePath.concat(PathUtils.subPath(appendPath, n => n.state === node.state)));
}
/**
 * Given a TransEvt (Transition event: started, success, error)
 * and a UISref Target State, return a SrefStatus object
 * which represents the current status of that Sref:
 * active, activeEq (exact match), entering, exiting
 *
 * @internalapi
 */
function getSrefStatus(event, srefTarget) {
    const pathMatchesTarget = pathMatches(srefTarget);
    const tc = event.trans.treeChanges();
    const isStartEvent = event.evt === 'start';
    const isSuccessEvent = event.evt === 'success';
    const activePath = isSuccessEvent ? tc.to : tc.from;
    const isActive = () => spreadToSubPaths([], activePath)
        .map(pathMatchesTarget)
        .reduce(anyTrueR, false);
    const isExact = () => pathMatchesTarget(activePath);
    const isEntering = () => spreadToSubPaths(tc.retained, tc.entering)
        .map(pathMatchesTarget)
        .reduce(anyTrueR, false);
    const isExiting = () => spreadToSubPaths(tc.retained, tc.exiting)
        .map(pathMatchesTarget)
        .reduce(anyTrueR, false);
    return {
        active: isActive(),
        exact: isExact(),
        entering: isStartEvent ? isEntering() : false,
        exiting: isStartEvent ? isExiting() : false,
        targetStates: [srefTarget],
    };
}
/** @internalapi */
function mergeSrefStatus(left, right) {
    return {
        active: left.active || right.active,
        exact: left.exact || right.exact,
        entering: left.entering || right.entering,
        exiting: left.exiting || right.exiting,
        targetStates: left.targetStates.concat(right.targetStates),
    };
}
/**
 * A directive which emits events when a paired [[UISref]] status changes.
 *
 * This directive is primarily used by the [[UISrefActive]] directives to monitor `UISref`(s).
 *
 * This directive shares two attribute selectors with `UISrefActive`:
 *
 * - `[uiSrefActive]`
 * - `[uiSrefActiveEq]`.
 *
 * Thus, whenever a `UISrefActive` directive is created, a `UISrefStatus` directive is also created.
 *
 * Most apps should simply use `UISrefActive`, but some advanced components may want to process the
 * [[SrefStatus]] events directly.
 *
 * ```js
 * <li (uiSrefStatus)="onSrefStatusChanged($event)">
 *   <a uiSref="book" [uiParams]="{ bookId: book.id }">Book {{ book.name }}</a>
 * </li>
 * ```
 *
 * The `uiSrefStatus` event is emitted whenever an enclosed `uiSref`'s status changes.
 * The event emitted is of type [[SrefStatus]], and has boolean values for `active`, `exact`, `entering`, and `exiting`; also has a [[StateOrName]] `identifier`value.
 *
 * The values from this event can be captured and stored on a component (then applied, e.g., using ngClass).
 *
 * ---
 *
 * A single `uiSrefStatus` can enclose multiple `uiSref`.
 * Each status boolean (`active`, `exact`, `entering`, `exiting`) will be true if *any of the enclosed `uiSref` status is true*.
 * In other words, all enclosed `uiSref` statuses  are merged to a single status using `||` (logical or).
 *
 * ```js
 * <li (uiSrefStatus)="onSrefStatus($event)" uiSref="admin">
 *   Home
 *   <ul>
 *     <li> <a uiSref="admin.users">Users</a> </li>
 *     <li> <a uiSref="admin.groups">Groups</a> </li>
 *   </ul>
 * </li>
 * ```
 *
 * In the above example, `$event.active === true` when either `admin.users` or `admin.groups` is active.
 *
 * ---
 *
 * This API is subject to change.
 */
let UISrefStatus = class UISrefStatus {
    constructor(_hostUiSref, _globals) {
        /** current statuses of the state/params the uiSref directive is linking to */
        this.uiSrefStatus = new EventEmitter(false);
        this._globals = _globals;
        this._hostUiSref = _hostUiSref;
        this.status = Object.assign({}, inactiveStatus);
    }
    ngAfterContentInit() {
        // Map each transition start event to a stream of:
        // start -> (success|error)
        const transEvents$ = this._globals.start$.pipe(switchMap((trans) => {
            const event = (evt) => ({ evt, trans });
            const transStart$ = of(event('start'));
            const transResult = trans.promise.then(() => event('success'), () => event('error'));
            const transFinish$ = from(transResult);
            return concat(transStart$, transFinish$);
        }));
        const withHostSref = (childrenSrefs) => childrenSrefs
            .concat(this._hostUiSref)
            .filter(identity)
            .reduce(uniqR, []);
        // Watch the @ContentChildren UISref[] components and get their target states
        this._srefs$ = new BehaviorSubject(withHostSref(this._srefs.toArray()));
        this._srefChangesSub = this._srefs.changes.subscribe(srefs => this._srefs$.next(withHostSref(srefs)));
        const targetStates$ = this._srefs$.pipe(switchMap((srefs) => combineLatest(srefs.map(sref => sref.targetState$))));
        // Calculate the status of each UISref based on the transition event.
        // Reduce the statuses (if multiple) by or-ing each flag.
        this._subscription = transEvents$
            .pipe(switchMap((evt) => {
            return targetStates$.pipe(map((targets) => {
                const statuses = targets.map(target => getSrefStatus(evt, target));
                return statuses.reduce(mergeSrefStatus);
            }));
        }))
            .subscribe(this._setStatus.bind(this));
    }
    ngOnDestroy() {
        if (this._subscription)
            this._subscription.unsubscribe();
        if (this._srefChangesSub)
            this._srefChangesSub.unsubscribe();
        if (this._srefs$)
            this._srefs$.unsubscribe();
        this._subscription = this._srefChangesSub = this._srefs$ = undefined;
    }
    _setStatus(status) {
        this.status = status;
        this.uiSrefStatus.emit(status);
    }
};
tslib_1.__decorate([
    Output('uiSrefStatus'),
    tslib_1.__metadata("design:type", Object)
], UISrefStatus.prototype, "uiSrefStatus", void 0);
tslib_1.__decorate([
    ContentChildren(UISref, { descendants: true }),
    tslib_1.__metadata("design:type", QueryList)
], UISrefStatus.prototype, "_srefs", void 0);
UISrefStatus = tslib_1.__decorate([
    Directive({
        selector: '[uiSrefStatus],[uiSrefActive],[uiSrefActiveEq]',
        exportAs: 'uiSrefStatus',
    }),
    tslib_1.__param(0, Host()), tslib_1.__param(0, Self()), tslib_1.__param(0, Optional()),
    tslib_1.__metadata("design:paramtypes", [UISref, UIRouterGlobals])
], UISrefStatus);
export { UISrefStatus };
export { ɵ0 };
//# sourceMappingURL=data:application/json;base64,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