import * as tslib_1 from "tslib";
/** @ng2api @module directives */
/** */
import { Directive, Output, EventEmitter, ContentChildren, QueryList, Host, Self, Optional } from '@angular/core';
import { UISref } from './uiSref';
import { anyTrueR, tail, unnestR, UIRouterGlobals, Param, PathUtils, identity, uniqR, } from '@uirouter/core';
import { BehaviorSubject, of, from, combineLatest, concat } from 'rxjs';
import { switchMap, map } from 'rxjs/operators';
/** @internalapi */
var inactiveStatus = {
    active: false,
    exact: false,
    entering: false,
    exiting: false,
    targetStates: [],
};
/**
 * Returns a Predicate<PathNode[]>
 *
 * The predicate returns true when the target state (and param values)
 * match the (tail of) the path, and the path's param values
 *
 * @internalapi
 */
var pathMatches = function (target) {
    if (!target.exists())
        return function () { return false; };
    var state = target.$state();
    var targetParamVals = target.params();
    var targetPath = PathUtils.buildPath(target);
    var paramSchema = targetPath
        .map(function (node) { return node.paramSchema; })
        .reduce(unnestR, [])
        .filter(function (param) { return targetParamVals.hasOwnProperty(param.id); });
    return function (path) {
        var tailNode = tail(path);
        if (!tailNode || tailNode.state !== state)
            return false;
        var paramValues = PathUtils.paramValues(path);
        return Param.equals(paramSchema, paramValues, targetParamVals);
    };
};
var ɵ0 = pathMatches;
/**
 * Given basePath: [a, b], appendPath: [c, d]),
 * Expands the path to [c], [c, d]
 * Then appends each to [a,b,] and returns: [a, b, c], [a, b, c, d]
 *
 * @internalapi
 */
function spreadToSubPaths(basePath, appendPath) {
    return appendPath.map(function (node) { return basePath.concat(PathUtils.subPath(appendPath, function (n) { return n.state === node.state; })); });
}
/**
 * Given a TransEvt (Transition event: started, success, error)
 * and a UISref Target State, return a SrefStatus object
 * which represents the current status of that Sref:
 * active, activeEq (exact match), entering, exiting
 *
 * @internalapi
 */
function getSrefStatus(event, srefTarget) {
    var pathMatchesTarget = pathMatches(srefTarget);
    var tc = event.trans.treeChanges();
    var isStartEvent = event.evt === 'start';
    var isSuccessEvent = event.evt === 'success';
    var activePath = isSuccessEvent ? tc.to : tc.from;
    var isActive = function () {
        return spreadToSubPaths([], activePath)
            .map(pathMatchesTarget)
            .reduce(anyTrueR, false);
    };
    var isExact = function () { return pathMatchesTarget(activePath); };
    var isEntering = function () {
        return spreadToSubPaths(tc.retained, tc.entering)
            .map(pathMatchesTarget)
            .reduce(anyTrueR, false);
    };
    var isExiting = function () {
        return spreadToSubPaths(tc.retained, tc.exiting)
            .map(pathMatchesTarget)
            .reduce(anyTrueR, false);
    };
    return {
        active: isActive(),
        exact: isExact(),
        entering: isStartEvent ? isEntering() : false,
        exiting: isStartEvent ? isExiting() : false,
        targetStates: [srefTarget],
    };
}
/** @internalapi */
function mergeSrefStatus(left, right) {
    return {
        active: left.active || right.active,
        exact: left.exact || right.exact,
        entering: left.entering || right.entering,
        exiting: left.exiting || right.exiting,
        targetStates: left.targetStates.concat(right.targetStates),
    };
}
/**
 * A directive which emits events when a paired [[UISref]] status changes.
 *
 * This directive is primarily used by the [[UISrefActive]] directives to monitor `UISref`(s).
 *
 * This directive shares two attribute selectors with `UISrefActive`:
 *
 * - `[uiSrefActive]`
 * - `[uiSrefActiveEq]`.
 *
 * Thus, whenever a `UISrefActive` directive is created, a `UISrefStatus` directive is also created.
 *
 * Most apps should simply use `UISrefActive`, but some advanced components may want to process the
 * [[SrefStatus]] events directly.
 *
 * ```js
 * <li (uiSrefStatus)="onSrefStatusChanged($event)">
 *   <a uiSref="book" [uiParams]="{ bookId: book.id }">Book {{ book.name }}</a>
 * </li>
 * ```
 *
 * The `uiSrefStatus` event is emitted whenever an enclosed `uiSref`'s status changes.
 * The event emitted is of type [[SrefStatus]], and has boolean values for `active`, `exact`, `entering`, and `exiting`; also has a [[StateOrName]] `identifier`value.
 *
 * The values from this event can be captured and stored on a component (then applied, e.g., using ngClass).
 *
 * ---
 *
 * A single `uiSrefStatus` can enclose multiple `uiSref`.
 * Each status boolean (`active`, `exact`, `entering`, `exiting`) will be true if *any of the enclosed `uiSref` status is true*.
 * In other words, all enclosed `uiSref` statuses  are merged to a single status using `||` (logical or).
 *
 * ```js
 * <li (uiSrefStatus)="onSrefStatus($event)" uiSref="admin">
 *   Home
 *   <ul>
 *     <li> <a uiSref="admin.users">Users</a> </li>
 *     <li> <a uiSref="admin.groups">Groups</a> </li>
 *   </ul>
 * </li>
 * ```
 *
 * In the above example, `$event.active === true` when either `admin.users` or `admin.groups` is active.
 *
 * ---
 *
 * This API is subject to change.
 */
var UISrefStatus = /** @class */ (function () {
    function UISrefStatus(_hostUiSref, _globals) {
        /** current statuses of the state/params the uiSref directive is linking to */
        this.uiSrefStatus = new EventEmitter(false);
        this._globals = _globals;
        this._hostUiSref = _hostUiSref;
        this.status = Object.assign({}, inactiveStatus);
    }
    UISrefStatus.prototype.ngAfterContentInit = function () {
        var _this = this;
        // Map each transition start event to a stream of:
        // start -> (success|error)
        var transEvents$ = this._globals.start$.pipe(switchMap(function (trans) {
            var event = function (evt) { return ({ evt: evt, trans: trans }); };
            var transStart$ = of(event('start'));
            var transResult = trans.promise.then(function () { return event('success'); }, function () { return event('error'); });
            var transFinish$ = from(transResult);
            return concat(transStart$, transFinish$);
        }));
        var withHostSref = function (childrenSrefs) {
            return childrenSrefs
                .concat(_this._hostUiSref)
                .filter(identity)
                .reduce(uniqR, []);
        };
        // Watch the @ContentChildren UISref[] components and get their target states
        this._srefs$ = new BehaviorSubject(withHostSref(this._srefs.toArray()));
        this._srefChangesSub = this._srefs.changes.subscribe(function (srefs) { return _this._srefs$.next(withHostSref(srefs)); });
        var targetStates$ = this._srefs$.pipe(switchMap(function (srefs) { return combineLatest(srefs.map(function (sref) { return sref.targetState$; })); }));
        // Calculate the status of each UISref based on the transition event.
        // Reduce the statuses (if multiple) by or-ing each flag.
        this._subscription = transEvents$
            .pipe(switchMap(function (evt) {
            return targetStates$.pipe(map(function (targets) {
                var statuses = targets.map(function (target) { return getSrefStatus(evt, target); });
                return statuses.reduce(mergeSrefStatus);
            }));
        }))
            .subscribe(this._setStatus.bind(this));
    };
    UISrefStatus.prototype.ngOnDestroy = function () {
        if (this._subscription)
            this._subscription.unsubscribe();
        if (this._srefChangesSub)
            this._srefChangesSub.unsubscribe();
        if (this._srefs$)
            this._srefs$.unsubscribe();
        this._subscription = this._srefChangesSub = this._srefs$ = undefined;
    };
    UISrefStatus.prototype._setStatus = function (status) {
        this.status = status;
        this.uiSrefStatus.emit(status);
    };
    tslib_1.__decorate([
        Output('uiSrefStatus'),
        tslib_1.__metadata("design:type", Object)
    ], UISrefStatus.prototype, "uiSrefStatus", void 0);
    tslib_1.__decorate([
        ContentChildren(UISref, { descendants: true }),
        tslib_1.__metadata("design:type", QueryList)
    ], UISrefStatus.prototype, "_srefs", void 0);
    UISrefStatus = tslib_1.__decorate([
        Directive({
            selector: '[uiSrefStatus],[uiSrefActive],[uiSrefActiveEq]',
            exportAs: 'uiSrefStatus',
        }),
        tslib_1.__param(0, Host()), tslib_1.__param(0, Self()), tslib_1.__param(0, Optional()),
        tslib_1.__metadata("design:paramtypes", [UISref, UIRouterGlobals])
    ], UISrefStatus);
    return UISrefStatus;
}());
export { UISrefStatus };
export { ɵ0 };
//# sourceMappingURL=data:application/json;base64,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