/** @ng2api @module core */
/** */
import { NgModuleFactory, Compiler, NgModuleFactoryLoader } from '@angular/core';
import { UIRouter, Resolvable, NATIVE_INJECTOR_TOKEN, isString, unnestR, inArray, uniqR, } from '@uirouter/core';
import { UIROUTER_ROOT_MODULE, UIROUTER_MODULE_TOKEN } from '../uiRouterNgModule';
import { applyModuleConfig } from '../uiRouterConfig';
/**
 * Returns a function which lazy loads a nested module
 *
 * This is primarily used by the [[ng2LazyLoadBuilder]] when processing [[Ng2StateDeclaration.loadChildren]].
 *
 * It could also be used manually as a [[StateDeclaration.lazyLoad]] property to lazy load an `NgModule` and its state(s).
 *
 * #### Example:
 * Using `import()` and named export of `HomeModule`
 * ```js
 * declare var System;
 * var futureState = {
 *   name: 'home.**',
 *   url: '/home',
 *   lazyLoad: loadNgModule(() => import('./home/home.module').then(result => result.HomeModule))
 * }
 * ```
 *
 * #### Example:
 * Using a path (string) to the module
 * ```js
 * var futureState = {
 *   name: 'home.**',
 *   url: '/home',
 *   lazyLoad: loadNgModule('./home/home.module#HomeModule')
 * }
 * ```
 *
 *
 * @param moduleToLoad a path (string) to the NgModule to load.
 *    Or a function which loads the NgModule code which should
 *    return a reference to  the `NgModule` class being loaded (or a `Promise` for it).
 *
 * @returns A function which takes a transition, which:
 * - Gets the Injector (scoped properly for the destination state)
 * - Loads and creates the NgModule
 * - Finds the "replacement state" for the target state, and adds the new NgModule Injector to it (as a resolve)
 * - Returns the new states array
 */
export function loadNgModule(moduleToLoad) {
    return function (transition, stateObject) {
        var ng2Injector = transition.injector().get(NATIVE_INJECTOR_TOKEN);
        var createModule = function (factory) { return factory.create(ng2Injector); };
        var applyModule = function (moduleRef) { return applyNgModule(transition, moduleRef, ng2Injector, stateObject); };
        return loadModuleFactory(moduleToLoad, ng2Injector)
            .then(createModule)
            .then(applyModule);
    };
}
/**
 * Returns the module factory that can be used to instantiate a module
 *
 * For strings this:
 * - Finds the correct NgModuleFactoryLoader
 * - Loads the new NgModuleFactory from the path string (async)
 *
 * For a Type<any> or Promise<Type<any>> this:
 * - Compiles the component type (if not running with AOT)
 * - Returns the NgModuleFactory resulting from compilation (or direct loading if using AOT) as a Promise
 *
 * @internalapi
 */
export function loadModuleFactory(moduleToLoad, ng2Injector) {
    if (isString(moduleToLoad)) {
        return ng2Injector.get(NgModuleFactoryLoader).load(moduleToLoad);
    }
    var compiler = ng2Injector.get(Compiler);
    var unwrapEsModuleDefault = function (x) { return (x && x.__esModule && x['default'] ? x['default'] : x); };
    return Promise.resolve(moduleToLoad())
        .then(unwrapEsModuleDefault)
        .then(function (t) {
        if (t instanceof NgModuleFactory) {
            return t;
        }
        return compiler.compileModuleAsync(t);
    });
}
/**
 * Apply the UI-Router Modules found in the lazy loaded module.
 *
 * Apply the Lazy Loaded NgModule's newly created Injector to the right state in the state tree.
 *
 * Lazy loading uses a placeholder state which is removed (and replaced) after the module is loaded.
 * The NgModule should include a state with the same name as the placeholder.
 *
 * Find the *newly loaded state* with the same name as the *placeholder state*.
 * The NgModule's Injector (and ComponentFactoryResolver) will be added to that state.
 * The Injector/Factory are used when creating Components for the `replacement` state and all its children.
 *
 * @internalapi
 */
export function applyNgModule(transition, ng2Module, parentInjector, lazyLoadState) {
    var injector = ng2Module.injector;
    var uiRouter = injector.get(UIRouter);
    var registry = uiRouter.stateRegistry;
    var originalName = lazyLoadState.name;
    var originalState = registry.get(originalName);
    // Check if it's a future state (ends with .**)
    var isFuture = /^(.*)\.\*\*$/.exec(originalName);
    // Final name (without the .**)
    var replacementName = isFuture && isFuture[1];
    var newRootModules = multiProviderParentChildDelta(parentInjector, injector, UIROUTER_ROOT_MODULE).reduce(uniqR, []);
    var newChildModules = multiProviderParentChildDelta(parentInjector, injector, UIROUTER_MODULE_TOKEN).reduce(uniqR, []);
    if (newRootModules.length) {
        console.log(newRootModules); // tslint:disable-line:no-console
        throw new Error('Lazy loaded modules should not contain a UIRouterModule.forRoot() module');
    }
    var newStateObjects = newChildModules
        .map(function (module) { return applyModuleConfig(uiRouter, injector, module); })
        .reduce(unnestR, [])
        .reduce(uniqR, []);
    if (isFuture) {
        var replacementState = registry.get(replacementName);
        if (!replacementState || replacementState === originalState) {
            throw new Error("The Future State named '" + originalName + "' lazy loaded an NgModule. " +
                ("The lazy loaded NgModule must have a state named '" + replacementName + "' ") +
                ("which replaces the (placeholder) '" + originalName + "' Future State. ") +
                ("Add a '" + replacementName + "' state to the lazy loaded NgModule ") +
                "using UIRouterModule.forChild({ states: CHILD_STATES }).");
        }
    }
    // Supply the newly loaded states with the Injector from the lazy loaded NgModule.
    // If a tree of states is lazy loaded, only add the injector to the root of the lazy loaded tree.
    // The children will get the injector by resolve inheritance.
    var newParentStates = newStateObjects.filter(function (state) { return !inArray(newStateObjects, state.parent); });
    // Add the Injector to the top of the lazy loaded state tree as a resolve
    newParentStates.forEach(function (state) { return state.resolvables.push(Resolvable.fromData(NATIVE_INJECTOR_TOKEN, injector)); });
    return {};
}
/**
 * Returns the new dependency injection values from the Child Injector
 *
 * When a DI token is defined as multi: true, the child injector
 * can add new values for the token.
 *
 * This function returns the values added by the child injector,  and excludes all values from the parent injector.
 *
 * @internalapi
 */
export function multiProviderParentChildDelta(parent, child, token) {
    var childVals = child.get(token, []);
    var parentVals = parent.get(token, []);
    return childVals.filter(function (val) { return parentVals.indexOf(val) === -1; });
}
//# sourceMappingURL=data:application/json;base64,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